<?php

declare(strict_types=1);

namespace Drupal\flowdrop_runtime\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Extension\ModuleExtensionList;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Controller for the execution overview page.
 *
 * Provides a landing page for all execution-related actions in FlowDrop.
 */
class ExecutionController extends ControllerBase {

  /**
   * The module extension list.
   *
   * @var \Drupal\Core\Extension\ModuleExtensionList
   */
  protected ModuleExtensionList $moduleExtensionList;

  /**
   * Constructs an ExecutionController.
   *
   * @param \Drupal\Core\Extension\ModuleExtensionList $moduleExtensionList
   *   The module extension list service.
   */
  public function __construct(ModuleExtensionList $moduleExtensionList) {
    $this->moduleExtensionList = $moduleExtensionList;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get("extension.list.module"),
    );
  }

  /**
   * Renders the execution overview page.
   *
   * @return array<string, mixed>
   *   A render array for the execution overview.
   */
  public function overview(): array {
    // Get module path for assets.
    $modulePath = "/" . $this->moduleExtensionList->getPath("flowdrop_ui_components");

    $build = [
      "#type" => "container",
      "#attributes" => [
        "class" => ["fd-fullscreen-layout"],
      ],
      "#attached" => [
        "library" => [
          "flowdrop_ui_components/fullscreen-layout",
          "flowdrop_ui_components/base",
        ],
      ],
    ];

    // Main content.
    $build["main"] = [
      "#type" => "container",
      "#attributes" => [
        "class" => ["fd-fullscreen-main", "fd-fullscreen-main--constrained"],
      ],
    ];

    $build["main"]["description"] = [
      "#type" => "html_tag",
      "#tag" => "p",
      "#value" => $this->t("Execute workflows and pipelines, or interact with running executions."),
      "#attributes" => [
        "class" => ["fd-text-secondary", "fd-mb-6"],
      ],
    ];

    $links = [
      "#type" => "container",
      "#attributes" => [
        "class" => ["fd-actions-grid", "fd-stagger-in"],
      ],
    ];

    // Execute Workflow.
    $links["execute_workflow"] = [
      "#type" => "component",
      "#component" => "flowdrop_ui_components:action-link",
      "#props" => [
        "title" => $this->t("Execute Workflow"),
        "description" => $this->t("Start a workflow execution with pipeline options"),
        "url" => Url::fromRoute("flowdrop.execute.workflow")->toString(),
        "icon" => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor"><path fill-rule="evenodd" d="M4.5 5.653c0-1.426 1.529-2.33 2.779-1.643l11.54 6.348c1.295.712 1.295 2.573 0 3.285L7.28 19.991c-1.25.687-2.779-.217-2.779-1.643V5.653z" clip-rule="evenodd"/></svg>',
      ],
    ];

    // Execute Pipeline.
    $links["execute_pipeline"] = [
      "#type" => "component",
      "#component" => "flowdrop_ui_components:action-link",
      "#props" => [
        "title" => $this->t("Execute Pipeline"),
        "description" => $this->t("Execute an existing pipeline directly"),
        "url" => Url::fromRoute("flowdrop.execute.pipeline")->toString(),
        "icon" => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor"><path fill-rule="evenodd" d="M3 6a3 3 0 013-3h12a3 3 0 013 3v12a3 3 0 01-3 3H6a3 3 0 01-3-3V6zm4.5 7.5a.75.75 0 01.75.75v2.25a.75.75 0 01-1.5 0v-2.25a.75.75 0 01.75-.75zm3.75-1.5a.75.75 0 00-1.5 0v4.5a.75.75 0 001.5 0V12zm2.25-3a.75.75 0 01.75.75v6.75a.75.75 0 01-1.5 0V9.75A.75.75 0 0113.5 9zm3.75-1.5a.75.75 0 00-1.5 0v9a.75.75 0 001.5 0v-9z" clip-rule="evenodd"/></svg>',
      ],
    ];

    // Pipeline Interaction.
    $links["interact"] = [
      "#type" => "component",
      "#component" => "flowdrop_ui_components:action-link",
      "#props" => [
        "title" => $this->t("Pipeline Interaction"),
        "description" => $this->t("Manage and control running pipelines"),
        "url" => Url::fromRoute("flowdrop.execute.interact")->toString(),
        "icon" => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor"><path fill-rule="evenodd" d="M2.25 12c0-5.385 4.365-9.75 9.75-9.75s9.75 4.365 9.75 9.75-4.365 9.75-9.75 9.75S2.25 17.385 2.25 12zm11.378-3.917c-.89-.777-2.366-.777-3.255 0a.75.75 0 01-.988-1.129c1.454-1.272 3.776-1.272 5.23 0 1.513 1.324 1.513 3.518 0 4.842a3.75 3.75 0 01-.837.552c-.676.328-1.028.774-1.028 1.152v.75a.75.75 0 01-1.5 0v-.75c0-1.279 1.06-2.107 1.875-2.502.182-.088.351-.199.503-.331.83-.727.83-1.857 0-2.584zM12 18a.75.75 0 100-1.5.75.75 0 000 1.5z" clip-rule="evenodd"/></svg>',
      ],
    ];

    $build["main"]["links"] = $links;

    return $build;
  }

}
