<?php

declare(strict_types=1);

namespace Drupal\flowdrop_runtime\Service;

use Drupal\flowdrop\DTO\ParameterBagInterface;

/**
 * Interface for parameter resolution services.
 *
 * The ParameterResolver is responsible for resolving parameter values
 * from multiple sources (runtime inputs, workflow values, defaults)
 * into a single ParameterBag for node execution.
 *
 * Resolution Priority (highest to lowest):
 * 1. Runtime input (from upstream node connection) - if connectable=TRUE
 * 2. Workflow value (set by user in workflow editor) - if configurable=TRUE
 * 3. Entity default (from config entity YAML)
 * 4. Schema default (from plugin's getParameterSchema())
 *
 * Flag Resolution (entity config → system defaults):
 * - connectable: entity config, else FALSE
 * - configurable: entity config, else FALSE
 * - required: entity config, else FALSE
 *
 * @see \Drupal\flowdrop\DTO\ParameterBagInterface
 * @see docs/development/flowdrop-node-processor.md
 */
interface ParameterResolverInterface {

  /**
   * Resolve parameters for node execution.
   *
   * Takes parameter definitions from multiple sources and resolves
   * them into a single ParameterBag with validated values.
   *
   * @param array<string, mixed> $parameterSchema
   *   The plugin's parameter schema from getParameterSchema().
   *   Contains standard JSON Schema properties (type, default, enum, etc.).
   *   Expected structure:
   *   [
   *     "type" => "object",
   *     "properties" => [
   *       "paramName" => [
   *         "type" => "string",
   *         "description" => "Parameter description",
   *         "default" => "value",
   *       ],
   *     ],
   *   ].
   * @param array<string, mixed> $entityConfig
   *   The config entity's parameter configuration.
   *   Controls connectable/configurable/required flags.
   *   Expected structure:
   *   [
   *     "paramName" => [
   *       "configurable" => TRUE,
   *       "connectable" => TRUE,
   *       "required" => FALSE,
   *       "default" => "override_value",
   *     ],
   *   ].
   * @param array<string, mixed> $workflowValues
   *   Values from the workflow node instance (user-configured values).
   * @param array<string, mixed> $runtimeInputs
   *   Values from upstream node connections (runtime values).
   *
   * @return \Drupal\flowdrop\DTO\ParameterBagInterface
   *   The resolved parameters ready for plugin execution.
   *
   * @throws \Drupal\flowdrop_runtime\Exception\MissingParameterException
   *   When a required parameter is missing from all sources.
   * @throws \Drupal\flowdrop_runtime\Exception\ParameterValidationException
   *   When a parameter value fails schema validation.
   */
  public function resolve(
    array $parameterSchema,
    array $entityConfig,
    array $workflowValues,
    array $runtimeInputs,
  ): ParameterBagInterface;

}
