# FlowDrop Trigger Module

Event-driven workflow triggering for FlowDrop. Automatically execute workflows when Drupal events occur (entity CRUD, user login/logout, etc.).

## Overview

The `flowdrop_trigger` module enables event-driven workflow automation in FlowDrop. Trigger configurations are stored as dedicated config entities (`FlowDropTriggerConfig`) and are created when a trigger node is dropped in the workflow editor.

## Architecture

### Key Concepts

1. **FlowDropTriggerConfig Entity**: Stores trigger configuration (event type, conditions, orchestrator settings)
2. **Entity ID Format**: `{workflow_id}__{node_id}` - Unique per workflow node
3. **Immediate Creation**: Entity is created when node is dropped in editor (via API)
4. **Single Source of Truth**: All config lives in the entity, node only stores reference

### Flow

```
┌─────────────────────────────────────────────────────────────────────────┐
│  WORKFLOW EDITOR                                                        │
│                                                                         │
│  1. User drops "Trigger" node                                           │
│     │                                                                   │
│     ▼                                                                   │
│  2. POST /api/flowdrop/trigger-config                                   │
│     { workflow_id, node_id, event_type }                                │
│     │                                                                   │
│     ▼                                                                   │
│  3. FlowDropTriggerConfig entity created                                │
│     ID: my_workflow__node_abc123                                        │
│     │                                                                   │
│     ▼                                                                   │
│  4. Node stores: { trigger_config_id: "my_workflow__node_abc123" }      │
└─────────────────────────────────────────────────────────────────────────┘

┌─────────────────────────────────────────────────────────────────────────┐
│  EVENT PROCESSING                                                       │
│                                                                         │
│  1. Drupal event occurs (entity.insert, user.login, etc.)               │
│     │                                                                   │
│     ▼                                                                   │
│  2. TriggerManager queries FlowDropTriggerConfig by event_type          │
│     │                                                                   │
│     ▼                                                                   │
│  3. Filter by conditions (entity_type, bundle, roles)                   │
│     │                                                                   │
│     ▼                                                                   │
│  4. Execute via orchestrator (sync/async)                               │
└─────────────────────────────────────────────────────────────────────────┘
```

## Features

- **Entity Events**: Trigger on entity insert, update, delete, presave
- **User Events**: Trigger on user login and logout
- **Flexible Conditions**: Filter by entity type, bundle, user role
- **Orchestrator Settings**: Configure sync/async, pipeline mode, retry, priority
- **Dynamic Schema**: Event-type specific configuration forms
- **Admin UI**: List and edit trigger configurations

## Requirements

- Drupal 10.3+ or 11.x
- FlowDrop core modules:
  - `flowdrop`
  - `flowdrop_node_type`
  - `flowdrop_workflow`
  - `flowdrop_runtime`
  - `flowdrop_pipeline`

## Installation

```bash
drush en flowdrop_trigger
drush cr
```

## API Endpoints

| Method | Endpoint | Description |
|--------|----------|-------------|
| POST | `/api/flowdrop/triggers` | Create trigger config |
| GET | `/api/flowdrop/triggers/{id}` | Get config with schema |
| PATCH | `/api/flowdrop/triggers/{id}` | Update config |
| DELETE | `/api/flowdrop/triggers/{id}` | Delete config |
| GET | `/api/flowdrop/triggers/schema` | Get schema for event type |
| GET | `/api/flowdrop/triggers/event-types` | List available event types |

### Create Trigger Config

```json
POST /api/flowdrop/triggers
{
  "workflow_id": "my_workflow",
  "node_id": "node_abc123",
  "event_type": "entity.insert"
}

Response:
{
  "id": "my_workflow__node_abc123",
  "schema": { ... },
  "uiSchema": { ... },
  "values": { ... },
  "created": true
}
```

### Update Trigger Config

```json
PATCH /api/flowdrop/triggers/my_workflow__node_abc123
{
  "conditions": {
    "entity_types": ["node"],
    "bundles": ["article"]
  },
  "orchestrator_settings": {
    "type": "asynchronous",
    "priority": "high"
  }
}
```

## Event Types

### Entity Events

| Event | Description | Default Orchestrator |
|-------|-------------|---------------------|
| `entity.insert` | Entity created | Asynchronous |
| `entity.update` | Entity updated | Asynchronous |
| `entity.delete` | Entity deleted | Asynchronous |
| `entity.presave` | Before entity save | Synchronous |

### User Events

| Event | Description | Default Orchestrator |
|-------|-------------|---------------------|
| `user.login` | User logged in | Asynchronous |
| `user.logout` | User logged out | Asynchronous |

## Trigger Configuration Schema

```yaml
id: my_workflow__node_abc123
label: 'Article Creation Trigger'
description: 'Triggers when new articles are created'
workflow_id: my_workflow
node_id: node_abc123
event_type: entity.insert
conditions:
  entity_types:
    - node
  bundles:
    - article
orchestrator_settings:
  type: asynchronous
  pipeline_mode: new
  priority: normal
  timeout: 300
  retry:
    enabled: true
    max_attempts: 3
    delay: 60
    backoff: exponential
initial_data_mapping: {}
status: true
weight: 0
```

## Permissions

| Permission | Description |
|------------|-------------|
| `administer flowdrop triggers` | Create, edit, delete trigger configurations |
| `view flowdrop triggers` | View trigger configuration list and details |

## Admin UI

Access trigger configuration list at:
`/admin/structure/flowdrop-trigger`

The admin interface provides:
- List of all trigger configurations
- Add new trigger configuration (manual creation)
- Edit existing trigger configurations
- Delete trigger configurations

Note: Triggers are typically created automatically when adding a Trigger node
in the workflow editor. The admin UI is provided for management and debugging.

## Extending

### Adding Custom Event Types

Create a new event type plugin:

```php
<?php

namespace Drupal\my_module\Plugin\FlowDropEventType;

use Drupal\flowdrop_trigger\Attribute\FlowDropEventType;
use Drupal\flowdrop_trigger\Plugin\FlowDropEventTypeBase;

#[FlowDropEventType(
  id: "my_custom.event",
  label: new TranslatableMarkup("My Custom Event"),
  description: new TranslatableMarkup("Triggered on custom event"),
  category: "custom",
  defaultOrchestrator: "asynchronous",
  defaultExecutionMode: "async",
)]
class MyCustomEvent extends FlowDropEventTypeBase {

  public function extractTriggerData(
    EntityInterface $entity,
    ?EntityInterface $originalEntity = NULL,
    array $context = [],
  ): array {
    return $this->getBaseTriggerData($entity);
  }

}
```

## Troubleshooting

### Triggers Not Firing

1. Verify the workflow is enabled (status = true)
2. Check the trigger config is enabled
3. Verify entity type and bundle conditions match
4. Check Drupal logs: `drush watchdog:show --type=flowdrop_trigger`

### Orphan Trigger Configs

Trigger configs that reference non-existent workflows are logged as warnings but not automatically deleted. Check logs for:
```
Trigger @trigger references non-existent workflow @workflow
```

## License

GPL-2.0-or-later
