<?php

declare(strict_types=1);

namespace Drupal\flowdrop_trigger\Plugin\FlowDropEventType;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\flowdrop_trigger\Attribute\FlowDropEventType;
use Drupal\flowdrop_trigger\FlowDropEventTypeInterface;
use Drupal\flowdrop_trigger\Plugin\FlowDropEventTypeBase;

/**
 * Event type for entity delete events.
 *
 * Triggers when an entity is deleted from the database.
 * This event fires after the entity has been successfully deleted.
 * The entity data is captured before deletion for use in workflows.
 */
#[FlowDropEventType(
  id: "entity.delete",
  label: new TranslatableMarkup("Entity Delete"),
  description: new TranslatableMarkup("Triggered when an entity is deleted"),
  category: "entity",
  defaultOrchestrator: "asynchronous",
  defaultExecutionMode: "async",
  supportedEntityTypes: [],
)]
class EntityDelete extends FlowDropEventTypeBase implements FlowDropEventTypeInterface {

  /**
   * {@inheritdoc}
   */
  public function extractTriggerData(
    EntityInterface $entity,
    ?EntityInterface $originalEntity = NULL,
    array $context = [],
  ): array {
    $data = $this->getBaseTriggerData($entity);

    // Add delete-specific data.
    $data["is_new"] = FALSE;
    $data["operation"] = "delete";
    $data["deleted"] = TRUE;

    return $data;
  }

  /**
   * {@inheritdoc}
   */
  public function getParameterSchema(): array {
    $baseSchema = parent::getParameterSchema();

    // Entity delete doesn't need additional configuration beyond base.
    return $baseSchema;
  }

}
