<?php

declare(strict_types=1);

namespace Drupal\flowdrop\DTO;

/**
 * Represents a single validation warning.
 *
 * Used within ValidationResult to provide information about
 * non-critical validation issues that don't block execution.
 */
final class ValidationWarning {

  /**
   * Constructs a ValidationWarning.
   *
   * @param string $parameter
   *   The parameter name associated with the warning.
   * @param string $message
   *   Human-readable warning message.
   */
  public function __construct(
    private readonly string $parameter,
    private readonly string $message,
  ) {}

  /**
   * Get the parameter name.
   *
   * @return string
   *   The parameter associated with this warning.
   */
  public function getParameter(): string {
    return $this->parameter;
  }

  /**
   * Get the warning message.
   *
   * @return string
   *   Human-readable warning message.
   */
  public function getMessage(): string {
    return $this->message;
  }

  /**
   * Convert to array representation.
   *
   * @return array{parameter: string, message: string}
   *   Array representation of the warning.
   */
  public function toArray(): array {
    return [
      "parameter" => $this->parameter,
      "message" => $this->message,
    ];
  }

  /**
   * Create from array representation.
   *
   * @param array{parameter: string, message: string} $data
   *   The array data.
   *
   * @return self
   *   A new ValidationWarning instance.
   */
  public static function fromArray(array $data): self {
    return new self(
      parameter: $data["parameter"],
      message: $data["message"],
    );
  }

}
