<?php

declare(strict_types=1);

namespace Drupal\flowdrop\Hook;

use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;

/**
 * FlowDrop hooks implementation.
 */
class FlowDropHooks {

  use StringTranslationTrait;

  /**
   * Implements hook_flowdrop_node_processor_info_alter().
   *
   * @param array<string, mixed> $data
   *   The processor info data.
   */
  #[Hook('flowdrop_node_processor_info_alter')]
  public function flowdropNodeProcessorInfoAlter(array &$data): void {}

  /**
   * Implements hook_toolbar().
   *
   * Adds a FlowDrop item to the admin toolbar for quick access.
   *
   * @return array<string, mixed>
   *   A render array for the toolbar item.
   */
  #[Hook('toolbar')]
  public function toolbar(): array {
    $items = [];

    // Check if the dashboard route exists before adding toolbar.
    if (!$this->routeExists("flowdrop.dashboard")) {
      return $items;
    }

    $items["flowdrop"] = [
      "#type" => "toolbar_item",
      "#weight" => 50,
      "tab" => [
        "#type" => "link",
        "#title" => $this->t("FlowDrop"),
        "#url" => Url::fromRoute("flowdrop.dashboard"),
        "#attributes" => [
          "title" => $this->t("FlowDrop workflow automation"),
          "class" => ["toolbar-icon", "toolbar-icon-flowdrop"],
        ],
      ],
      "#attached" => [
        "library" => [
          "flowdrop/toolbar",
        ],
      ],
    ];

    return $items;
  }

  /**
   * Builds the toolbar tray links.
   *
   * @return array<string, array<string, mixed>>
   *   Array of link definitions.
   */
  protected function buildToolbarLinks(): array {
    $links = [];

    // Dashboard link.
    if ($this->routeExists("flowdrop.dashboard")) {
      $links["dashboard"] = [
        "title" => $this->t("Dashboard"),
        "url" => Url::fromRoute("flowdrop.dashboard"),
        "attributes" => [
          "title" => $this->t("FlowDrop dashboard"),
        ],
      ];
    }

    // Check if runtime module is enabled and route exists.
    if (\Drupal::moduleHandler()->moduleExists("flowdrop_runtime")) {
      if ($this->routeExists("flowdrop_runtime.workflow_execution")) {
        $links["execute_workflow"] = [
          "title" => $this->t("Execute Workflow"),
          "url" => Url::fromRoute("flowdrop_runtime.workflow_execution"),
          "attributes" => [
            "title" => $this->t("Execute a workflow"),
          ],
        ];
      }
    }

    // Check if workflow module is enabled and route exists.
    if (\Drupal::moduleHandler()->moduleExists("flowdrop_workflow")) {
      if ($this->routeExists("entity.flowdrop_workflow.collection")) {
        $links["workflows"] = [
          "title" => $this->t("Workflows"),
          "url" => Url::fromRoute("entity.flowdrop_workflow.collection"),
          "attributes" => [
            "title" => $this->t("Manage workflows"),
          ],
        ];
      }
    }

    // Check if pipeline module is enabled and route exists.
    if (\Drupal::moduleHandler()->moduleExists("flowdrop_pipeline")) {
      if ($this->routeExists("entity.flowdrop_pipeline.collection")) {
        $links["pipelines"] = [
          "title" => $this->t("Pipelines"),
          "url" => Url::fromRoute("entity.flowdrop_pipeline.collection"),
          "attributes" => [
            "title" => $this->t("View pipelines"),
          ],
        ];
      }
    }

    return $links;
  }

  /**
   * Checks if a route exists.
   *
   * @param string $routeName
   *   The route name.
   *
   * @return bool
   *   TRUE if the route exists, FALSE otherwise.
   */
  protected function routeExists(string $routeName): bool {
    try {
      /** @var \Drupal\Core\Routing\RouteProviderInterface $routeProvider */
      $routeProvider = \Drupal::service("router.route_provider");
      $routeProvider->getRouteByName($routeName);
      return TRUE;
    }
    catch (\Exception $e) {
      return FALSE;
    }
  }

}
