<?php

declare(strict_types=1);

namespace Drupal\flowdrop\Plugin\FlowDropNodeProcessor;

use Drupal\flowdrop\DTO\ValidationResult;

/**
 * Interface for FlowDropNodeProcessor plugins.
 *
 * This interface defines the contract for node processor plugins in FlowDrop.
 * Plugins must implement the Unified Parameter System via getParameterSchema().
 *
 * Note: Visual type is managed by the config entity (FlowDropNodeType),
 * not by the plugin. This allows site builders to choose the visual format.
 *
 * @see \Drupal\flowdrop\Plugin\FlowDropNodeProcessor\AbstractFlowDropNodeProcessor
 * @see docs/development/unified-parameter-system-spec.md
 */
interface FlowDropNodeProcessorInterface extends NodeExecutorInterface {

  /**
   * Get the component ID.
   *
   * @return string
   *   The unique component identifier.
   */
  public function getId(): string;

  /**
   * Get the component name.
   *
   * @return string
   *   The human-readable component name.
   */
  public function getName(): string;

  /**
   * Get the component description.
   *
   * @return string
   *   The component description.
   */
  public function getDescription(): string;

  /**
   * Get the component version.
   *
   * @return string
   *   The component version.
   */
  public function getVersion(): string;

  /**
   * Validate parameters for the node.
   *
   * Returns a structured ValidationResult instead of a boolean to enable:
   * - Detailed error messages for each validation failure
   * - Warnings for non-critical issues
   * - Centralized logging by the orchestration layer.
   *
   * Note: Processors should NOT log validation failures - the orchestration
   * layer handles this via ExecutionLogger.
   *
   * @param array<string, mixed> $params
   *   The parameters to validate.
   *
   * @return \Drupal\flowdrop\DTO\ValidationResult
   *   The validation result with errors and warnings.
   */
  public function validateParams(array $params): ValidationResult;

  /**
   * Get the output schema for this node.
   *
   * @return array<string, mixed>
   *   The output schema in JSON Schema-like format.
   */
  public function getOutputSchema(): array;

  /**
   * Get the parameter schema for this node.
   *
   * Returns a standard JSON Schema structure defining all parameters this
   * node accepts. The schema describes the data contract (types, defaults,
   * constraints) but NOT the UI/workflow behavior (connectable, configurable).
   *
   * UI/workflow behavior (connectable, configurable, required) is controlled
   * entirely by the config entity, not by the plugin schema.
   *
   * Example:
   * @code
   * [
   *   "type" => "object",
   *   "properties" => [
   *     "operation" => [
   *       "type" => "string",
   *       "description" => "The operation to perform",
   *       "default" => "add",
   *       "enum" => ["add", "subtract"],
   *     ],
   *     "input_value" => [
   *       "type" => "number",
   *       "description" => "The input value",
   *       "minimum" => 0,
   *     ],
   *     "temperature" => [
   *       "type" => "number",
   *       "description" => "Sampling temperature",
   *       "default" => 0.7,
   *       "minimum" => 0,
   *       "maximum" => 2,
   *     ],
   *   ],
   * ]
   * @endcode
   *
   * Supported JSON Schema properties:
   * - type: string, number, integer, boolean, array, object, mixed
   * - description: Human-readable description
   * - default: Default value
   * - enum: Array of allowed values
   * - minimum, maximum: Numeric constraints
   * - minLength, maxLength: String length constraints
   * - pattern: Regex pattern for strings
   * - format: email, uri, date, date-time, etc.
   *
   * @return array<string, mixed>
   *   The parameter schema in JSON Schema format.
   *
   * @see docs/development/flowdrop-node-processor.md
   */
  public function getParameterSchema(): array;

}
