<?php

declare(strict_types=1);

namespace Drupal\flowdrop_node_processor\Plugin\FlowDropNodeProcessor;

use Drupal\flowdrop\Attribute\FlowDropNodeProcessor;
use Drupal\flowdrop\DTO\ParameterBagInterface;
use Drupal\flowdrop\DTO\ValidationResult;
use Drupal\flowdrop\Plugin\FlowDropNodeProcessor\AbstractFlowDropNodeProcessor;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Executor for A/B Logic nodes.
 *
 * A/B nodes provide simple conditional logic with random behavior.
 */
#[FlowDropNodeProcessor(
  id: "a_or_b",
  label: new \Drupal\Core\StringTranslation\TranslatableMarkup("A/B"),
  description: "Simple boolean gateway for A/B use case",
  version: "1.0.0"
)]
class AorB extends AbstractFlowDropNodeProcessor {

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ): static {
    return new static($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public function validateParams(array $params): ValidationResult {
    // A/B nodes can accept any inputs or none.
    return ValidationResult::success();
  }

  /**
   * {@inheritdoc}
   */
  protected function process(ParameterBagInterface $params): array {
    $conditionResult = (bool) random_int(0, 1);
    $activeBranch = $conditionResult ? "a" : "b";

    return [
      "active_branches" => $activeBranch,
      "input_value" => NULL,
      "execution_metadata" => [
        "timestamp" => time(),
        "gateway_type" => "branch",
        "flow_control" => TRUE,
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getParameterSchema(): array {
    return [
      "type" => "object",
      "properties" => [
        "weight_a" => [
          "type" => "number",
          "title" => "Weight A",
          "description" => "Weight for variation \"A\"",
          "required" => TRUE,
          "default" => 50,
        ],
        "weight_b" => [
          "type" => "number",
          "title" => "Weight B",
          "description" => "Weight for variation \"B\"",
          "required" => TRUE,
          "default" => 50,
        ],
        "branches" => [
          "type" => "array",
          "description" => "The available branches",
          "default" => [
            [
              "name" => "A",
              "value" => "a",
            ],
            [
              "name" => "B",
              "value" => "b",
            ],
          ],
          "items" => [
            "type" => "object",
            "properties" => [
              "name" => [
                "type" => "string",
                "description" => "The name of the branch",
              ],
              "value" => [
                "type" => "string",
                "description" => "The value of the branch",
              ],
            ],
          ],
        ],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getOutputSchema(): array {
    return [];
  }

}
