<?php

declare(strict_types=1);

namespace Drupal\flowdrop_node_processor\Plugin\FlowDropNodeProcessor;

use Drupal\flowdrop\Attribute\FlowDropNodeProcessor;
use Drupal\flowdrop\DTO\ParameterBagInterface;
use Drupal\flowdrop\DTO\ValidationResult;
use Drupal\flowdrop\Plugin\FlowDropNodeProcessor\AbstractFlowDropNodeProcessor;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Executor for Text Input nodes.
 *
 * Provides a simple text input field for user data entry.
 */
#[FlowDropNodeProcessor(
  id: "text_input",
  label: new \Drupal\Core\StringTranslation\TranslatableMarkup("Text Input"),
  description: "User text input field",
  version: "1.0.0"
)]
class TextInput extends AbstractFlowDropNodeProcessor {

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ): static {
    return new static($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public function validateParams(array $params): ValidationResult {
    // Text input can accept any inputs or none.
    return ValidationResult::success();
  }

  /**
   * {@inheritdoc}
   */
  protected function process(ParameterBagInterface $params): array {
    // Return the configured default value or input data if provided.
    $text = $params->get("defaultValue", "");

    $inputData = $params->get("data");
    if ($inputData !== NULL) {
      $text = $inputData;
    }

    return [
      "text" => $text,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getParameterSchema(): array {
    return [
      "type" => "object",
      "properties" => [
        "data" => [
          "type" => "mixed",
          "title" => "Input Data",
          "description" => "Input data to override default value",
        ],
        "defaultValue" => [
          "type" => "string",
          "title" => "Default Value",
          "description" => "Default text value",
          "default" => "",
        ],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getOutputSchema(): array {
    return [
      "type" => "object",
      "properties" => [
        "text" => [
          "type" => "string",
          "description" => "The input text value",
        ],
      ],
    ];
  }

}
