<?php

declare(strict_types=1);

namespace Drupal\flowdrop_runtime\Exception;

/**
 * Exception thrown when a required parameter is missing.
 *
 * This exception is thrown by the ParameterResolver when a parameter
 * marked as required is not provided via any source (runtime input,
 * workflow value, or default).
 *
 * @see \Drupal\flowdrop_runtime\Service\ParameterResolver
 */
class MissingParameterException extends RuntimeException {

  /**
   * The name of the missing parameter.
   *
   * @var string
   */
  protected string $parameterName;

  /**
   * Constructs a MissingParameterException.
   *
   * @param string $parameterName
   *   The name of the missing parameter.
   * @param string $message
   *   The exception message.
   * @param int $code
   *   The exception code.
   * @param \Throwable|null $previous
   *   The previous exception.
   */
  public function __construct(
    string $parameterName,
    string $message = "",
    int $code = 0,
    ?\Throwable $previous = NULL,
  ) {
    $this->parameterName = $parameterName;

    if (empty($message)) {
      $message = "Required parameter '{$parameterName}' is missing.";
    }

    parent::__construct($message, $code, $previous);
  }

  /**
   * Get the name of the missing parameter.
   *
   * @return string
   *   The parameter name.
   */
  public function getParameterName(): string {
    return $this->parameterName;
  }

}
