<?php

declare(strict_types=1);

namespace Drupal\flowdrop_trigger\Drush\Commands;

use Drush\Attributes as CLI;
use Drush\Commands\AutowireTrait;
use Drush\Commands\DrushCommands;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\flowdrop_trigger\Service\TriggerManager;

/**
 * Drush commands for testing FlowDrop triggers.
 */
final class TriggerTestCommands extends DrushCommands {

  use AutowireTrait;

  /**
   * Constructs a TriggerTestCommands object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\flowdrop_trigger\Service\TriggerManager $triggerManager
   *   The trigger manager.
   */
  public function __construct(
    private readonly EntityTypeManagerInterface $entityTypeManager,
    private readonly TriggerManager $triggerManager,
  ) {
    parent::__construct();
  }

  /**
   * Manually fire a trigger event for testing.
   *
   * @param string $eventType
   *   The event type (e.g., entity.insert, entity.update, entity.delete).
   * @param string $entityType
   *   The entity type (e.g., node, user).
   * @param string|int $entityId
   *   The entity ID.
   * @param array<string, mixed> $options
   *   Command options including "original" flag for update events.
   */
  #[CLI\Command(name: "flowdrop:trigger:test", aliases: ["fd:trigger:test"])]
  #[CLI\Argument(name: "eventType", description: "The event type (e.g., entity.insert, entity.update, entity.delete)")]
  #[CLI\Argument(name: "entityType", description: "The entity type (e.g., node, user)")]
  #[CLI\Argument(name: "entityId", description: "The entity ID")]
  #[CLI\Option(name: "original", description: "Include original entity for update events")]
  #[CLI\Usage(name: "flowdrop:trigger:test entity.insert node 1", description: "Fire an entity.insert event for node 1")]
  #[CLI\Usage(name: "flowdrop:trigger:test entity.update node 1", description: "Fire an entity.update event for node 1 (includes original entity)")]
  #[CLI\Usage(name: "flowdrop:trigger:test entity.delete node 1", description: "Fire an entity.delete event for node 1")]
  public function testTrigger(
    string $eventType,
    string $entityType,
    string|int $entityId,
    array $options = ["original" => TRUE],
  ): void {
    try {
      $storage = $this->entityTypeManager->getStorage($entityType);
      $entity = $storage->load($entityId);

      if ($entity === NULL) {
        $this->logger()->error("Entity not found: @type @id", [
          "@type" => $entityType,
          "@id" => $entityId,
        ]);
        return;
      }

      $original = NULL;
      if ($eventType === "entity.update" && ($options["original"] ?? TRUE)) {
        $original = $storage->loadUnchanged($entityId);
      }

      $this->logger()->notice("Firing trigger event @event for @type @id", [
        "@event" => $eventType,
        "@type" => $entityType,
        "@id" => $entityId,
      ]);

      $this->triggerManager->processEvent($eventType, $entity, $original);

      $this->logger()->success("Trigger event fired successfully. Check logs for execution details.");
    }
    catch (\Exception $e) {
      $this->logger()->error("Failed to fire trigger: @message", [
        "@message" => $e->getMessage(),
      ]);
    }
  }

  /**
   * List all active triggers grouped by event type.
   */
  #[CLI\Command(name: "flowdrop:trigger:list", aliases: ["fd:trigger:list"])]
  #[CLI\Usage(name: "flowdrop:trigger:list", description: "List all active trigger configurations")]
  public function listTriggers(): void {
    $triggers = $this->triggerManager->getAllActiveTriggers();

    if (empty($triggers)) {
      $this->logger()->warning("No active triggers found.");
      return;
    }

    $this->output()->writeln("\n<info>Active Triggers:</info>\n");

    foreach ($triggers as $eventType => $configs) {
      $this->output()->writeln("<comment>Event Type: {$eventType}</comment>");
      foreach ($configs as $config) {
        $this->output()->writeln(sprintf(
          "  - ID: %s | Workflow: %s | Node: %s | Weight: %d",
          $config["id"],
          $config["workflow_id"],
          $config["node_id"],
          $config["weight"]
        ));
      }
      $this->output()->writeln("");
    }
  }

  /**
   * Test a specific trigger configuration by ID.
   *
   * @param string $triggerId
   *   The trigger configuration ID (format: workflow_id__node_id).
   * @param string $eventType
   *   The event type to simulate.
   * @param string $entityType
   *   The entity type.
   * @param string|int $entityId
   *   The entity ID.
   */
  #[CLI\Command(name: "flowdrop:trigger:test-config", aliases: ["fd:trigger:test-config"])]
  #[CLI\Argument(name: "triggerId", description: "The trigger configuration ID (format: workflow_id__node_id)")]
  #[CLI\Argument(name: "eventType", description: "The event type to simulate")]
  #[CLI\Argument(name: "entityType", description: "The entity type")]
  #[CLI\Argument(name: "entityId", description: "The entity ID")]
  #[CLI\Usage(name: "flowdrop:trigger:test-config my_workflow__node_123 entity.insert node 1", description: "Test a specific trigger configuration")]
  public function testTriggerConfig(
    string $triggerId,
    string $eventType,
    string $entityType,
    string|int $entityId,
  ): void {
    try {
      $storage = $this->entityTypeManager->getStorage("flowdrop_trigger_config");
      $triggerConfig = $storage->load($triggerId);

      if ($triggerConfig === NULL) {
        $this->logger()->error("Trigger config not found: @id", [
          "@id" => $triggerId,
        ]);
        return;
      }

      $entityStorage = $this->entityTypeManager->getStorage($entityType);
      $entity = $entityStorage->load($entityId);

      if ($entity === NULL) {
        $this->logger()->error("Entity not found: @type @id", [
          "@type" => $entityType,
          "@id" => $entityId,
        ]);
        return;
      }

      $this->logger()->notice("Testing trigger config @trigger with event @event", [
        "@trigger" => $triggerId,
        "@event" => $eventType,
      ]);

      // Check if trigger matches conditions.
      $context = [
        "entity_type" => $entity->getEntityTypeId(),
        "bundle" => $entity->bundle(),
        "entity" => $entity,
      ];

      if (!$triggerConfig->matches($eventType, $context)) {
        $this->logger()->warning("Trigger config conditions do not match. Event will not fire.");
        return;
      }

      $original = NULL;
      if ($eventType === "entity.update") {
        $original = $entityStorage->loadUnchanged($entityId);
      }

      $this->triggerManager->processEvent($eventType, $entity, $original);

      $this->logger()->success("Trigger config tested successfully.");
    }
    catch (\Exception $e) {
      $this->logger()->error("Failed to test trigger config: @message", [
        "@message" => $e->getMessage(),
      ]);
    }
  }

}
