<?php

declare(strict_types=1);

namespace Drupal\flowdrop_trigger;

use Drupal\Core\Config\Entity\ConfigEntityInterface;

/**
 * Interface for FlowDrop Trigger Configuration entities.
 *
 * Trigger configurations define when and how a workflow should be executed
 * in response to system events (entity CRUD, user login/logout, cron, etc.).
 *
 * Each trigger config is created when a trigger node is dropped in the
 * workflow editor. The entity ID follows the format: {workflow_id}__{node_id}
 *
 * @see \Drupal\flowdrop_trigger\Entity\FlowDropTriggerConfig
 */
interface FlowDropTriggerConfigInterface extends ConfigEntityInterface {

  /**
   * Get the target workflow ID.
   *
   * @return string
   *   The workflow entity ID this trigger activates.
   */
  public function getWorkflowId(): string;

  /**
   * Set the target workflow ID.
   *
   * @param string $workflowId
   *   The workflow entity ID.
   *
   * @return static
   *   The entity instance for chaining.
   */
  public function setWorkflowId(string $workflowId): static;

  /**
   * Get the node ID within the workflow.
   *
   * @return string
   *   The node ID from the workflow canvas.
   */
  public function getNodeId(): string;

  /**
   * Set the node ID within the workflow.
   *
   * @param string $nodeId
   *   The node ID from the workflow canvas.
   *
   * @return static
   *   The entity instance for chaining.
   */
  public function setNodeId(string $nodeId): static;

  /**
   * Get the event type plugin ID.
   *
   * @return string
   *   The event type (e.g., "entity.insert", "user.login", "cron").
   */
  public function getEventType(): string;

  /**
   * Set the event type plugin ID.
   *
   * @param string $eventType
   *   The event type plugin ID.
   *
   * @return static
   *   The entity instance for chaining.
   */
  public function setEventType(string $eventType): static;

  /**
   * Get the trigger conditions.
   *
   * @return array<string, mixed>
   *   Conditions for matching. Structure varies by event type:
   *   - Entity events: entity_types, bundles, field_conditions
   *   - User events: roles, user_conditions
   *   - Cron events: schedule expression
   */
  public function getConditions(): array;

  /**
   * Set the trigger conditions.
   *
   * @param array<string, mixed> $conditions
   *   The conditions array.
   *
   * @return static
   *   The entity instance for chaining.
   */
  public function setConditions(array $conditions): static;

  /**
   * Get orchestrator settings.
   *
   * @return array<string, mixed>
   *   Settings for workflow execution:
   *   - type: Orchestrator plugin ID (synchronous, asynchronous)
   *   - pipeline_mode: "new" | "reuse" | "singleton"
   *   - pipeline_id: Pipeline ID for reuse mode
   *   - retry: { enabled, max_attempts, delay, backoff }
   *   - priority: "low" | "normal" | "high" | "critical"
   *   - timeout: Execution timeout in seconds
   */
  public function getOrchestratorSettings(): array;

  /**
   * Set orchestrator settings.
   *
   * @param array<string, mixed> $settings
   *   The orchestrator settings.
   *
   * @return static
   *   The entity instance for chaining.
   */
  public function setOrchestratorSettings(array $settings): static;

  /**
   * Get initial data mapping configuration.
   *
   * @return array<string, mixed>
   *   Mapping of event data to workflow input parameters.
   *   Keys are workflow input names, values are JSONPath expressions
   *   or literal values.
   */
  public function getInitialDataMapping(): array;

  /**
   * Set initial data mapping configuration.
   *
   * @param array<string, mixed> $mapping
   *   The data mapping configuration.
   *
   * @return static
   *   The entity instance for chaining.
   */
  public function setInitialDataMapping(array $mapping): static;

  /**
   * Get the weight for ordering multiple triggers.
   *
   * @return int
   *   The weight value. Lower values execute first.
   */
  public function getWeight(): int;

  /**
   * Set the weight for ordering.
   *
   * @param int $weight
   *   The weight value.
   *
   * @return static
   *   The entity instance for chaining.
   */
  public function setWeight(int $weight): static;

  /**
   * Check if this trigger matches the given event context.
   *
   * @param string $eventType
   *   The event type that occurred.
   * @param array<string, mixed> $context
   *   Event context data:
   *   - entity_type: The entity type ID
   *   - bundle: The entity bundle
   *   - entity: The entity object or serialized data
   *   - Additional context based on event type.
   *
   * @return bool
   *   TRUE if this trigger should fire for the given context.
   */
  public function matches(string $eventType, array $context): bool;

  /**
   * Get the FlowDropEventType plugin for this trigger.
   *
   * @return \Drupal\flowdrop_trigger\FlowDropEventTypeInterface|null
   *   The event type plugin instance, or NULL if not found.
   */
  public function getEventTypePlugin(): ?FlowDropEventTypeInterface;

  /**
   * Generate the entity ID from workflow and node IDs.
   *
   * @param string $workflowId
   *   The workflow ID.
   * @param string $nodeId
   *   The node ID.
   *
   * @return string
   *   The generated entity ID in format: {workflow_id}__{node_id}
   */
  public static function generateId(string $workflowId, string $nodeId): string;

  /**
   * Parse an entity ID to extract workflow and node IDs.
   *
   * @param string $entityId
   *   The entity ID.
   *
   * @return array{workflow_id: string, node_id: string}|null
   *   Array with workflow_id and node_id, or NULL if invalid format.
   */
  public static function parseId(string $entityId): ?array;

}
