<?php

declare(strict_types=1);

namespace Drupal\flowdrop\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleExtensionList;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Routing\RouteProviderInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Controller for the FlowDrop dashboard.
 *
 * Provides a modern, immersive dashboard experience using FlowDrop UI
 * Components (SDC) for a consistent visual language with the workflow editor.
 */
class FlowDropDashboardController extends ControllerBase {

  /**
   * The module extension list.
   *
   * @var \Drupal\Core\Extension\ModuleExtensionList
   */
  protected ModuleExtensionList $moduleExtensionList;

  /**
   * The route provider.
   *
   * @var \Drupal\Core\Routing\RouteProviderInterface
   */
  protected RouteProviderInterface $routeProvider;

  /**
   * Constructs a FlowDropDashboardController.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager service.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler service.
   * @param \Drupal\Core\Extension\ModuleExtensionList $moduleExtensionList
   *   The module extension list service.
   * @param \Drupal\Core\Routing\RouteProviderInterface $routeProvider
   *   The route provider service.
   */
  public function __construct(
    EntityTypeManagerInterface $entityTypeManager,
    ModuleHandlerInterface $moduleHandler,
    ModuleExtensionList $moduleExtensionList,
    RouteProviderInterface $routeProvider,
  ) {
    $this->entityTypeManager = $entityTypeManager;
    $this->moduleHandler = $moduleHandler;
    $this->moduleExtensionList = $moduleExtensionList;
    $this->routeProvider = $routeProvider;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get("entity_type.manager"),
      $container->get("module_handler"),
      $container->get("extension.list.module"),
      $container->get("router.route_provider"),
    );
  }

  /**
   * Renders the main FlowDrop dashboard.
   *
   * @return array<string, mixed>
   *   A render array for the dashboard page.
   */
  public function dashboard(): array {
    $build = [
      "#type" => "container",
      "#attributes" => [
        "class" => ["fd-fullscreen-layout"],
      ],
      "#attached" => [
        "library" => [
          "flowdrop_ui_components/fullscreen-layout",
          "flowdrop_ui_components/base",
        ],
      ],
    ];

    // Main content area.
    $build["main"] = [
      "#type" => "container",
      "#attributes" => [
        "class" => ["fd-fullscreen-main", "fd-fullscreen-main--constrained"],
      ],
    ];

    // Stats row.
    $build["main"]["stats"] = $this->buildStatsSection();

    // Quick actions.
    $build["main"]["quick_actions"] = $this->buildQuickActionsSection();

    // Navigation sections.
    $build["main"]["navigation"] = $this->buildNavigationSection();

    // Configuration sections.
    $build["main"]["config"] = $this->buildConfigSection();

    // Structure sections.
    $build["main"]["structure"] = $this->buildStructureSection();

    return $build;
  }

  /**
   * Builds the page header.
   *
   * @param string $pageTitle
   *   The page title to display.
   * @param bool $showCreateWorkflow
   *   Whether to show the "Create Workflow" action.
   *
   * @return array<string, mixed>
   *   A render array for the header.
   */
  protected function buildHeader(string $pageTitle, bool $showCreateWorkflow = FALSE): array {
    $primaryAction = NULL;
    if ($showCreateWorkflow && $this->moduleHandler()->moduleExists("flowdrop_workflow")) {
      $url = $this->getRouteUrl("entity.flowdrop_workflow.add_form");
      if ($url !== NULL) {
        $primaryAction = [
          "label" => (string) $this->t("Create Workflow"),
          "url" => $url,
        ];
      }
    }

    // Get module path for assets.
    $modulePath = "/" . $this->moduleExtensionList->getPath("flowdrop_ui_components");

    // Build props array, only including primary_action if it's not NULL.
    $props = [
      "page_title" => $pageTitle,
      "module_path" => $modulePath,
    ];
    if ($primaryAction !== NULL) {
      $props["primary_action"] = $primaryAction;
    }

    return [
      "#type" => "component",
      "#component" => "flowdrop_ui_components:page-header",
      "#props" => $props,
    ];
  }

  /**
   * Builds the statistics section.
   *
   * @return array<string, mixed>
   *   A render array for the stats section.
   */
  protected function buildStatsSection(): array {
    $section = [
      "#type" => "container",
      "#attributes" => [
        "class" => ["fd-section", "fd-animate-in"],
      ],
    ];

    // Collect stat card items for the grid.
    $statItems = [];

    // Operational entities - ordered by workflow lifecycle.
    // 1. Workflows (definitions - the blueprint).
    if ($this->moduleHandler()->moduleExists("flowdrop_workflow")) {
      $count = $this->getEntityCount("flowdrop_workflow");
      $url = $this->getRouteUrl("entity.flowdrop_workflow.collection");
      $statItems["workflows"] = [
        "#type" => "component",
        "#component" => "flowdrop_ui_components:stat-card",
        "#props" => [
          "value" => (string) $count,
          "label" => (string) $this->t("Workflows"),
          "variant" => "default",
          "url" => $url,
          "icon" => $this->getWorkflowIcon(),
        ],
      ];
    }

    // 2. Pipelines (instances created from workflows).
    if ($this->moduleHandler()->moduleExists("flowdrop_pipeline")) {
      $count = $this->getEntityCount("flowdrop_pipeline");
      $url = $this->getRouteUrl("entity.flowdrop_pipeline.collection");
      $statItems["pipelines"] = [
        "#type" => "component",
        "#component" => "flowdrop_ui_components:stat-card",
        "#props" => [
          "value" => (string) $count,
          "label" => (string) $this->t("Pipelines"),
          "variant" => "default",
          "url" => $url,
          "icon" => $this->getPipelineIcon(),
        ],
      ];
    }

    // 3. Jobs (executions of pipelines).
    if ($this->moduleHandler()->moduleExists("flowdrop_job")) {
      $count = $this->getEntityCount("flowdrop_job");
      $url = $this->getRouteUrl("entity.flowdrop_job.collection");
      $statItems["jobs"] = [
        "#type" => "component",
        "#component" => "flowdrop_ui_components:stat-card",
        "#props" => [
          "value" => (string) $count,
          "label" => (string) $this->t("Jobs"),
          "variant" => "default",
          "url" => $url,
          "icon" => $this->getJobIcon(),
        ],
      ];
    }

    // 4. Triggers (automation that starts workflows).
    if ($this->moduleHandler()->moduleExists("flowdrop_trigger")) {
      $count = $this->getEntityCount("flowdrop_trigger_config");
      $url = $this->getRouteUrl("entity.flowdrop_trigger_config.collection");
      $statItems["triggers"] = [
        "#type" => "component",
        "#component" => "flowdrop_ui_components:stat-card",
        "#props" => [
          "value" => (string) $count,
          "label" => (string) $this->t("Triggers"),
          "variant" => "default",
          "url" => $url,
          "icon" => $this->getTriggerIcon(),
        ],
      ];
    }

    // Configuration entities - building blocks for workflows.
    // 5. Node Types (building blocks used in workflows).
    if ($this->moduleHandler()->moduleExists("flowdrop_node_type")) {
      $count = $this->getEntityCount("flowdrop_node_type");
      $url = $this->getRouteUrl("flowdrop.config.node_types");
      $statItems["node_types"] = [
        "#type" => "component",
        "#component" => "flowdrop_ui_components:stat-card",
        "#props" => [
          "value" => (string) $count,
          "label" => (string) $this->t("Node Types"),
          "variant" => "default",
          "url" => $url,
          "icon" => $this->getNodeTypeIcon(),
        ],
      ];
    }

    // 6. Categories (organization of node types).
    if ($this->moduleHandler()->moduleExists("flowdrop_node_category")) {
      $count = $this->getEntityCount("flowdrop_node_category");
      $url = $this->getRouteUrl("entity.flowdrop_node_category.collection");
      $statItems["categories"] = [
        "#type" => "component",
        "#component" => "flowdrop_ui_components:stat-card",
        "#props" => [
          "value" => (string) $count,
          "label" => (string) $this->t("Categories"),
          "variant" => "default",
          "url" => $url,
          "icon" => $this->getCategoryIcon(),
        ],
      ];
    }

    // Use the grid component for consistent responsive behavior.
    $section["stats"] = [
      "#type" => "component",
      "#component" => "flowdrop_ui_components:grid",
      "#props" => [
        "variant" => "stats",
        "stagger" => TRUE,
      ],
      "#slots" => [
        "default" => $statItems,
      ],
    ];

    return $section;
  }

  /**
   * Builds the quick actions section.
   *
   * @return array<string, mixed>
   *   A render array for quick actions.
   */
  protected function buildQuickActionsSection(): array {
    $section = [
      "#type" => "container",
      "#attributes" => [
        "class" => ["fd-section"],
      ],
    ];

    $section["header"] = [
      "#type" => "container",
      "#attributes" => [
        "class" => ["fd-section__header"],
      ],
      "title" => [
        "#type" => "html_tag",
        "#tag" => "h2",
        "#value" => $this->t("Quick Actions"),
        "#attributes" => [
          "class" => ["fd-section__title"],
        ],
      ],
    ];

    // Collect action link items for the grid.
    $actionItems = [];

    // Execute workflow.
    if ($this->moduleHandler()->moduleExists("flowdrop_runtime")) {
      $url = $this->getRouteUrl("flowdrop.execute");
      if ($url !== NULL) {
        $actionItems["execute_workflow"] = [
          "#type" => "component",
          "#component" => "flowdrop_ui_components:action-link",
          "#props" => [
            "title" => (string) $this->t("Run Workflows"),
            "description" => (string) $this->t("Start a workflow execution with various options"),
            "url" => $url,
            "icon" => $this->getExecuteIcon(),
          ],
        ];
      }

      $url = $this->getRouteUrl("flowdrop_runtime.pipeline_execution");
      if ($url !== NULL) {
        $actionItems["execute_pipeline"] = [
          "#type" => "component",
          "#component" => "flowdrop_ui_components:action-link",
          "#props" => [
            "title" => (string) $this->t("Execute Pipeline"),
            "description" => (string) $this->t("Execute an existing pipeline directly"),
            "url" => $url,
            "icon" => $this->getPipelineIcon(),
          ],
        ];
      }
    }

    // Add Workflow.
    $url = $this->getRouteUrl("flowdrop.workflow.add");
    if ($url !== NULL) {
      $actionItems["add_workflow"] = [
        "#type" => "component",
        "#component" => "flowdrop_ui_components:action-link",
        "#props" => [
          "title" => (string) $this->t("Add Workflow"),
          "description" => (string) $this->t("Create a new workflow"),
          "url" => $url,
          "icon" => $this->getCreateIcon(),
        ],
      ];
    }

    // Use the grid component for consistent responsive behavior.
    $section["actions"] = [
      "#type" => "component",
      "#component" => "flowdrop_ui_components:grid",
      "#props" => [
        "variant" => "actions",
        "stagger" => TRUE,
      ],
      "#slots" => [
        "default" => $actionItems,
      ],
    ];

    return $section;
  }

  /**
   * Builds the navigation section with links to all areas.
   *
   * @return array<string, mixed>
   *   A render array for navigation.
   */
  protected function buildNavigationSection(): array {
    $section = [
      "#type" => "container",
      "#attributes" => [
        "class" => ["fd-section"],
      ],
    ];

    $section["header"] = [
      "#type" => "container",
      "#attributes" => [
        "class" => ["fd-section__header"],
      ],
      "title" => [
        "#type" => "html_tag",
        "#tag" => "h2",
        "#value" => $this->t("Manage"),
        "#attributes" => [
          "class" => ["fd-section__title"],
        ],
      ],
    ];

    // Collect link items for the grid.
    $linkItems = [];

    // Operational entities - ordered by workflow lifecycle.
    // 1. Workflows (definitions - the blueprint).
    if ($this->moduleHandler()->moduleExists("flowdrop_workflow")) {
      $count = $this->getEntityCount("flowdrop_workflow");
      $url = $this->getRouteUrl("entity.flowdrop_workflow.collection");
      if ($url !== NULL) {
        $linkItems["workflows"] = [
          "#type" => "component",
          "#component" => "flowdrop_ui_components:action-link",
          "#props" => [
            "title" => (string) $this->t("Workflows"),
            "description" => (string) $this->t("Create and manage workflow definitions"),
            "url" => $url,
            "icon" => $this->getWorkflowIcon(),
            "badge" => $count,
          ],
        ];
      }
    }

    // 2. Pipelines (instances created from workflows).
    if ($this->moduleHandler()->moduleExists("flowdrop_pipeline")) {
      $count = $this->getEntityCount("flowdrop_pipeline");
      $url = $this->getRouteUrl("entity.flowdrop_pipeline.collection");
      if ($url !== NULL) {
        $linkItems["pipelines"] = [
          "#type" => "component",
          "#component" => "flowdrop_ui_components:action-link",
          "#props" => [
            "title" => (string) $this->t("Pipelines"),
            "description" => (string) $this->t("View and manage pipeline instances"),
            "url" => $url,
            "icon" => $this->getPipelineIcon(),
            "badge" => $count,
          ],
        ];
      }
    }

    // 3. Jobs (executions of pipelines).
    if ($this->moduleHandler()->moduleExists("flowdrop_job")) {
      $count = $this->getEntityCount("flowdrop_job");
      $url = $this->getRouteUrl("entity.flowdrop_job.collection");
      if ($url !== NULL) {
        $linkItems["jobs"] = [
          "#type" => "component",
          "#component" => "flowdrop_ui_components:action-link",
          "#props" => [
            "title" => (string) $this->t("Jobs"),
            "description" => (string) $this->t("View and manage job executions"),
            "url" => $url,
            "icon" => $this->getJobIcon(),
            "badge" => $count,
          ],
        ];
      }
    }

    // 4. Triggers (automation that starts workflows).
    if ($this->moduleHandler()->moduleExists("flowdrop_trigger")) {
      $count = $this->getEntityCount("flowdrop_trigger_config");
      $url = $this->getRouteUrl("entity.flowdrop_trigger_config.collection");
      if ($url !== NULL) {
        $linkItems["triggers"] = [
          "#type" => "component",
          "#component" => "flowdrop_ui_components:action-link",
          "#props" => [
            "title" => (string) $this->t("Triggers"),
            "description" => (string) $this->t("Configure event-based workflow triggers"),
            "url" => $url,
            "icon" => $this->getTriggerIcon(),
            "badge" => $count,
          ],
        ];
      }
    }

    // Use the grid component for consistent responsive behavior.
    $section["links"] = [
      "#type" => "component",
      "#component" => "flowdrop_ui_components:grid",
      "#props" => [
        "variant" => "actions",
        "stagger" => TRUE,
      ],
      "#slots" => [
        "default" => $linkItems,
      ],
    ];

    return $section;
  }

  /**
   * Builds the config section with links to all areas.
   *
   * @return array<string, mixed>
   *   A render array for navigation.
   */
  protected function buildConfigSection(): array {
    $section = [
      "#type" => "container",
      "#attributes" => [
        "class" => ["fd-section"],
      ],
    ];

    $section["header"] = [
      "#type" => "container",
      "#attributes" => [
        "class" => ["fd-section__header"],
      ],
      "title" => [
        "#type" => "html_tag",
        "#tag" => "h2",
        "#value" => $this->t("Configuration"),
        "#attributes" => [
          "class" => ["fd-section__title"],
        ],
      ],
    ];

    // Collect link items for the grid.
    $linkItems = [];

    // Configuration entities - building blocks for workflows.
    // Settings.
    $url = $this->getRouteUrl("flowdrop.settings");
    if ($url !== NULL) {
      $linkItems["settings"] = [
        "#type" => "component",
        "#component" => "flowdrop_ui_components:action-link",
        "#props" => [
          "title" => (string) $this->t("Core Settings"),
          "description" => (string) $this->t("Configure FlowDrop logging and execution options"),
          "url" => $url,
          "icon" => $this->getSettingsIcon(),
        ],
      ];
    }

    // Trigger Settings.
    if ($this->moduleHandler()->moduleExists("flowdrop_trigger")) {
      $url = $this->getRouteUrl("flowdrop.config.triggers");
      if ($url !== NULL) {
        $linkItems["trigger_settings"] = [
          "#type" => "component",
          "#component" => "flowdrop_ui_components:action-link",
          "#props" => [
            "title" => (string) $this->t("Trigger Settings"),
            "description" => (string) $this->t("Configure which events trigger workflows"),
            "url" => $url,
            "icon" => $this->getTriggerIcon(),
          ],
        ];
      }
    }

    // 5. Node Types (building blocks used in workflows).
    if ($this->moduleHandler()->moduleExists("flowdrop_node_type")) {
      $count = $this->getEntityCount("flowdrop_node_type");
      $url = $this->getRouteUrl("flowdrop.config.node_types");
      if ($url !== NULL) {
        $linkItems["node_types"] = [
          "#type" => "component",
          "#component" => "flowdrop_ui_components:action-link",
          "#props" => [
            "title" => (string) $this->t("Node Types"),
            "description" => (string) $this->t("Configure available node types for workflows"),
            "url" => $url,
            "icon" => $this->getNodeTypeIcon(),
            "badge" => $count,
          ],
        ];
      }
    }

    // 6. Categories (organization of node types).
    if ($this->moduleHandler()->moduleExists("flowdrop_node_category")) {
      $count = $this->getEntityCount("flowdrop_node_category");
      $url = $this->getRouteUrl("entity.flowdrop_node_category.collection");
      if ($url !== NULL) {
        $linkItems["categories"] = [
          "#type" => "component",
          "#component" => "flowdrop_ui_components:action-link",
          "#props" => [
            "title" => (string) $this->t("Categories"),
            "description" => (string) $this->t("Organize node types into categories"),
            "url" => $url,
            "icon" => $this->getCategoryIcon(),
            "badge" => $count,
          ],
        ];
      }
    }

    // Use the grid component for consistent responsive behavior.
    $section["links"] = [
      "#type" => "component",
      "#component" => "flowdrop_ui_components:grid",
      "#props" => [
        "variant" => "actions",
        "stagger" => TRUE,
      ],
      "#slots" => [
        "default" => $linkItems,
      ],
    ];

    return $section;
  }

  /**
   * Builds the structure section with links to entity bundles.
   *
   * @return array<string, mixed>
   *   A render array for navigation.
   */
  protected function buildStructureSection(): array {
    $section = [
      "#type" => "container",
      "#attributes" => [
        "class" => ["fd-section"],
      ],
    ];

    $section["header"] = [
      "#type" => "container",
      "#attributes" => [
        "class" => ["fd-section__header"],
      ],
      "title" => [
        "#type" => "html_tag",
        "#tag" => "h2",
        "#value" => $this->t("Structure"),
        "#attributes" => [
          "class" => ["fd-section__title"],
        ],
      ],
    ];

    // Collect link items for the grid.
    $linkItems = [];

    // Structure entities - entity bundles.
    // 7. Pipeline Types (bundles for pipeline entities).
    if ($this->moduleHandler()->moduleExists("flowdrop_pipeline")) {
      $count = $this->getEntityCount("flowdrop_pipeline_type");
      $url = $this->getRouteUrl("flowdrop.structure.pipeline_types");
      if ($url !== NULL) {
        $linkItems["pipeline_types"] = [
          "#type" => "component",
          "#component" => "flowdrop_ui_components:action-link",
          "#props" => [
            "title" => (string) $this->t("Pipeline Types"),
            "description" => (string) $this->t("Manage pipeline type bundles"),
            "url" => $url,
            "icon" => $this->getPipelineIcon(),
            "badge" => $count,
          ],
        ];
      }
    }

    // 8. Job Types (bundles for job entities).
    if ($this->moduleHandler()->moduleExists("flowdrop_job")) {
      $count = $this->getEntityCount("flowdrop_job_type");
      $url = $this->getRouteUrl("flowdrop.structure.job_types");
      if ($url !== NULL) {
        $linkItems["job_types"] = [
          "#type" => "component",
          "#component" => "flowdrop_ui_components:action-link",
          "#props" => [
            "title" => (string) $this->t("Job Types"),
            "description" => (string) $this->t("Manage job type bundles"),
            "url" => $url,
            "icon" => $this->getJobIcon(),
            "badge" => $count,
          ],
        ];
      }
    }

    // Use the grid component for consistent responsive behavior.
    $section["links"] = [
      "#type" => "component",
      "#component" => "flowdrop_ui_components:grid",
      "#props" => [
        "variant" => "actions",
        "stagger" => TRUE,
      ],
      "#slots" => [
        "default" => $linkItems,
      ],
    ];

    return $section;
  }

  /**
   * Renders the configuration overview page.
   *
   * @return array<string, mixed>
   *   A render array for the configuration page.
   */
  public function configOverview(): array {
    $build = [
      "#type" => "container",
      "#attributes" => [
        "class" => ["fd-fullscreen-layout"],
      ],
      "#attached" => [
        "library" => [
          "flowdrop_ui_components/fullscreen-layout",
          "flowdrop_ui_components/base",
        ],
      ],
    ];

    // Main content.
    $build["main"] = [
      "#type" => "container",
      "#attributes" => [
        "class" => ["fd-fullscreen-main", "fd-fullscreen-main--constrained"],
      ],
    ];

    $build["main"]["description"] = [
      "#type" => "html_tag",
      "#tag" => "p",
      "#value" => $this->t("Configure node types, categories, and other FlowDrop settings."),
      "#attributes" => [
        "class" => ["fd-text-secondary", "fd-mb-6"],
      ],
    ];

    // Collect link items for the grid.
    $linkItems = [];

    if ($this->moduleHandler()->moduleExists("flowdrop_node_type")) {
      $url = $this->getRouteUrl("flowdrop.config.node_types");
      if ($url !== NULL) {
        $linkItems["node_types"] = [
          "#type" => "component",
          "#component" => "flowdrop_ui_components:action-link",
          "#props" => [
            "title" => (string) $this->t("Node Types"),
            "description" => (string) $this->t("Configure available node types for workflows"),
            "url" => $url,
            "icon" => $this->getNodeTypeIcon(),
          ],
        ];
      }
    }

    if ($this->moduleHandler()->moduleExists("flowdrop_node_category")) {
      $url = $this->getRouteUrl("entity.flowdrop_node_category.collection");
      if ($url !== NULL) {
        $linkItems["categories"] = [
          "#type" => "component",
          "#component" => "flowdrop_ui_components:action-link",
          "#props" => [
            "title" => (string) $this->t("Categories"),
            "description" => (string) $this->t("Organize node types into categories"),
            "url" => $url,
            "icon" => $this->getCategoryIcon(),
          ],
        ];
      }
    }

    // Trigger settings.
    if ($this->moduleHandler()->moduleExists("flowdrop_trigger")) {
      $url = $this->getRouteUrl("flowdrop.config.triggers");
      if ($url !== NULL) {
        $linkItems["triggers"] = [
          "#type" => "component",
          "#component" => "flowdrop_ui_components:action-link",
          "#props" => [
            "title" => (string) $this->t("Trigger Settings"),
            "description" => (string) $this->t("Configure entity, form, and user event triggers"),
            "url" => $url,
            "icon" => $this->getTriggerIcon(),
          ],
        ];
      }
    }

    // General FlowDrop settings.
    $url = $this->getRouteUrl("flowdrop.settings");
    if ($url !== NULL) {
      $linkItems["settings"] = [
        "#type" => "component",
        "#component" => "flowdrop_ui_components:action-link",
        "#props" => [
          "title" => (string) $this->t("Settings"),
          "description" => (string) $this->t("Configure logging, execution, and real-time settings"),
          "url" => $url,
          "icon" => $this->getSettingsIcon(),
        ],
      ];
    }

    // Use the grid component for consistent responsive behavior.
    $build["main"]["links"] = [
      "#type" => "component",
      "#component" => "flowdrop_ui_components:grid",
      "#props" => [
        "variant" => "actions",
        "stagger" => TRUE,
      ],
      "#slots" => [
        "default" => $linkItems,
      ],
    ];

    return $build;
  }

  /**
   * Renders the structure overview page.
   *
   * @return array<string, mixed>
   *   A render array for the structure page.
   */
  public function structureOverview(): array {
    $build = [
      "#type" => "container",
      "#attributes" => [
        "class" => ["fd-fullscreen-layout"],
      ],
      "#attached" => [
        "library" => [
          "flowdrop_ui_components/fullscreen-layout",
          "flowdrop_ui_components/base",
        ],
      ],
    ];

    // Main content.
    $build["main"] = [
      "#type" => "container",
      "#attributes" => [
        "class" => ["fd-fullscreen-main", "fd-fullscreen-main--constrained"],
      ],
    ];

    $build["main"]["description"] = [
      "#type" => "html_tag",
      "#tag" => "p",
      "#value" => $this->t("Manage pipeline types, job types, and other entity bundles."),
      "#attributes" => [
        "class" => ["fd-text-secondary", "fd-mb-6"],
      ],
    ];

    // Collect link items for the grid.
    $linkItems = [];

    if ($this->moduleHandler()->moduleExists("flowdrop_pipeline")) {
      $url = $this->getRouteUrl("flowdrop.structure.pipeline_types");
      if ($url !== NULL) {
        $linkItems["pipeline_types"] = [
          "#type" => "component",
          "#component" => "flowdrop_ui_components:action-link",
          "#props" => [
            "title" => (string) $this->t("Pipeline Types"),
            "description" => (string) $this->t("Manage pipeline type bundles"),
            "url" => $url,
            "icon" => $this->getPipelineIcon(),
          ],
        ];
      }
    }

    if ($this->moduleHandler()->moduleExists("flowdrop_job")) {
      $url = $this->getRouteUrl("flowdrop.structure.job_types");
      if ($url !== NULL) {
        $linkItems["job_types"] = [
          "#type" => "component",
          "#component" => "flowdrop_ui_components:action-link",
          "#props" => [
            "title" => (string) $this->t("Job Types"),
            "description" => (string) $this->t("Manage job type bundles"),
            "url" => $url,
            "icon" => $this->getJobIcon(),
          ],
        ];
      }
    }

    // Use the grid component for consistent responsive behavior.
    $build["main"]["links"] = [
      "#type" => "component",
      "#component" => "flowdrop_ui_components:grid",
      "#props" => [
        "variant" => "actions",
        "stagger" => TRUE,
      ],
      "#slots" => [
        "default" => $linkItems,
      ],
    ];

    return $build;
  }

  /**
   * Safely gets a URL for a route.
   *
   * @param string $routeName
   *   The route name.
   * @param array<string, mixed> $parameters
   *   Optional route parameters.
   *
   * @return string|null
   *   The URL string, or NULL if the route doesn't exist.
   */
  protected function getRouteUrl(string $routeName, array $parameters = []): ?string {
    try {
      // Check if route exists.
      $this->routeProvider->getRouteByName($routeName);
      return Url::fromRoute($routeName, $parameters)->toString();
    }
    catch (\Exception $e) {
      // Route doesn't exist.
      return NULL;
    }
  }

  /**
   * Gets the count of entities of a given type.
   *
   * @param string $entityTypeId
   *   The entity type ID.
   *
   * @return int
   *   The entity count.
   */
  protected function getEntityCount(string $entityTypeId): int {
    try {
      return (int) $this->entityTypeManager()
        ->getStorage($entityTypeId)
        ->getQuery()
        ->accessCheck(TRUE)
        ->count()
        ->execute();
    }
    catch (\Exception $e) {
      return 0;
    }
  }

  /**
   * Returns workflow icon SVG.
   *
   * @return string
   *   The SVG markup.
   */
  protected function getWorkflowIcon(): string {

    return '<svg xmlns="http://www.w3.org/2000/svg" style="transform: rotate(90deg);" fill="none" viewBox="0 0 24 24" stroke-width="1.5" stroke="currentColor" class="size-6">
  <path stroke-linecap="round" stroke-linejoin="round" d="M7.217 10.907a2.25 2.25 0 1 0 0 2.186m0-2.186c.18.324.283.696.283 1.093s-.103.77-.283 1.093m0-2.186 9.566-5.314m-9.566 7.5 9.566 5.314m0 0a2.25 2.25 0 1 0 3.935 2.186 2.25 2.25 0 0 0-3.935-2.186Zm0-12.814a2.25 2.25 0 1 0 3.933-2.185 2.25 2.25 0 0 0-3.933 2.185Z" />
</svg>
';
  }

  /**
   * Returns pipeline icon SVG.
   *
   * @return string
   *   The SVG markup.
   */
  protected function getPipelineIcon(): string {
    return '<svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="1.5" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" d="M2.25 18L9 11.25l4.306 4.307a11.95 11.95 0 015.814-5.519l2.74-1.22m0 0l-5.94-2.28m5.94 2.28l-2.28 5.94"/></svg>';
  }

  /**
   * Returns job icon SVG.
   *
   * @return string
   *   The SVG markup.
   */
  protected function getJobIcon(): string {
    return '<svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="1.5" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" d="M20.25 14.15v4.25c0 .414-.336.75-.75.75h-4.5a.75.75 0 01-.75-.75v-4.25m0 0l4.5-4.5m-4.5 4.5l-4.5 4.5M3.75 9.75h4.5a.75.75 0 01.75.75v4.5a.75.75 0 01-.75.75h-4.5a.75.75 0 01-.75-.75v-4.5a.75.75 0 01.75-.75zm9-3h4.5a.75.75 0 01.75.75v4.5a.75.75 0 01-.75.75h-4.5a.75.75 0 01-.75-.75v-4.5a.75.75 0 01.75-.75z"/></svg>';
  }

  /**
   * Returns node type icon SVG.
   *
   * @return string
   *   The SVG markup.
   */
  protected function getNodeTypeIcon(): string {
    return '<svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="1.5" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" d="M3.75 6A2.25 2.25 0 016 3.75h2.25A2.25 2.25 0 0110.5 6v2.25a2.25 2.25 0 01-2.25 2.25H6a2.25 2.25 0 01-2.25-2.25V6zM3.75 15.75A2.25 2.25 0 016 13.5h2.25a2.25 2.25 0 012.25 2.25V18a2.25 2.25 0 01-2.25 2.25H6A2.25 2.25 0 013.75 18v-2.25zM13.5 6a2.25 2.25 0 012.25-2.25H18A2.25 2.25 0 0120.25 6v2.25A2.25 2.25 0 0118 10.5h-2.25a2.25 2.25 0 01-2.25-2.25V6zM13.5 15.75a2.25 2.25 0 012.25-2.25H18a2.25 2.25 0 012.25 2.25V18A2.25 2.25 0 0118 20.25h-2.25A2.25 2.25 0 0113.5 18v-2.25z"/></svg>';
  }

  /**
   * Returns execute icon SVG.
   *
   * @return string
   *   The SVG markup.
   */
  protected function getExecuteIcon(): string {
    return '<svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="1.5" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" d="M5.25 5.653c0-1.426 1.526-2.33 2.777-1.643l11.54 6.348c1.554.855 1.554 3.031 0 3.886l-11.54 6.347c-1.251.688-2.777-.217-2.777-1.643V5.653z"/></svg>';
  }

  /**
   * Returns settings icon SVG.
   *
   * @return string
   *   The SVG markup.
   */
  protected function getCreateIcon(): string {
    return '<svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="1.5" stroke="currentColor" class="size-6">
  <path stroke-linecap="round" stroke-linejoin="round" d="M12 9v6m3-3H9m12 0a9 9 0 1 1-18 0 9 9 0 0 1 18 0Z" />
</svg>
';
  }

  /**
   * Returns settings icon SVG.
   *
   * @return string
   *   The SVG markup.
   */
  protected function getSettingsIcon(): string {
    return '<svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="1.5" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" d="M9.594 3.94c.09-.542.56-.94 1.11-.94h2.593c.55 0 1.02.398 1.11.94l.213 1.281c.063.374.313.686.645.87.074.04.147.083.22.127.324.196.72.257 1.075.124l1.217-.456a1.125 1.125 0 011.37.49l1.296 2.247a1.125 1.125 0 01-.26 1.431l-1.003.827c-.293.24-.438.613-.431.992a6.759 6.759 0 010 .255c-.007.378.138.75.43.99l1.005.828c.424.35.534.954.26 1.43l-1.298 2.247a1.125 1.125 0 01-1.369.491l-1.217-.456c-.355-.133-.75-.072-1.076.124a6.57 6.57 0 01-.22.128c-.331.183-.581.495-.644.869l-.213 1.28c-.09.543-.56.941-1.11.941h-2.594c-.55 0-1.02-.398-1.11-.94l-.213-1.281c-.062-.374-.312-.686-.644-.87a6.52 6.52 0 01-.22-.127c-.325-.196-.72-.257-1.076-.124l-1.217.456a1.125 1.125 0 01-1.369-.49l-1.297-2.247a1.125 1.125 0 01.26-1.431l1.004-.827c.292-.24.437-.613.43-.992a6.932 6.932 0 010-.255c.007-.378-.138-.75-.43-.99l-1.004-.828a1.125 1.125 0 01-.26-1.43l1.297-2.247a1.125 1.125 0 011.37-.491l1.216.456c.356.133.751.072 1.076-.124.072-.044.146-.087.22-.128.332-.183.582-.495.644-.869l.214-1.281z"/><path stroke-linecap="round" stroke-linejoin="round" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z"/></svg>';
  }

  /**
   * Returns trigger icon SVG.
   *
   * @return string
   *   The SVG markup.
   */
  protected function getTriggerIcon(): string {
    return '<svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="1.5" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" d="M3.75 13.5l10.5-11.25L12 10.5h8.25l-10.5 11.25L12 13.5H3.75z"/></svg>';
  }

  /**
   * Returns category icon SVG.
   *
   * @return string
   *   The SVG markup.
   */
  protected function getCategoryIcon(): string {
    return '<svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="1.5" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" d="M9.568 3H5.25A2.25 2.25 0 003 5.25v4.318a2.25 2.25 0 00.662 1.582l10.5 10.5a2.25 2.25 0 003.182 0l4.318-4.318a2.25 2.25 0 000-3.182l-10.5-10.5A2.25 2.25 0 009.568 3z"/><path stroke-linecap="round" stroke-linejoin="round" d="M6 6h.008v.008H6V6z"/></svg>';
  }

}
