<?php

declare(strict_types=1);

namespace Drupal\flowdrop\Plugin\FlowDropNodeProcessor;

use Drupal\flowdrop\DTO\ConfigEdit\ConfigEdit;

/**
 * Interface for node processors that provide dynamic configuration editing.
 *
 * Implement this interface on your FlowDropNodeProcessor plugin to indicate
 * that the node supports dynamic configuration editing. This allows:
 *
 * - Dynamic schema fetching from REST endpoints at runtime
 * - External edit links to admin forms or third-party configuration UIs
 * - API endpoints for CRUD operations on configuration entities
 *
 * When a plugin implements this interface, the FlowDrop frontend will
 * use the provided configuration to render appropriate UI controls
 * for editing the node's configuration.
 *
 * Example implementation:
 * @code
 * class MyNodeProcessor extends AbstractFlowDropNodeProcessor
 *   implements ConfigEditProviderInterface {
 *
 *   public function getConfigEdit(): ConfigEdit {
 *     return new ConfigEdit(
 *       dynamicSchema: new DynamicSchemaEndpoint(
 *         url: "/api/my-node/schema",
 *         method: "GET",
 *       ),
 *       externalEditLink: new ExternalEditLink(
 *         url: "/admin/my-node/{id}/edit",
 *         label: "Configure Node",
 *       ),
 *     );
 *   }
 * }
 * @endcode
 *
 * @see \Drupal\flowdrop\DTO\ConfigEdit\ConfigEdit
 * @see config-edit.md for full specification
 */
interface ConfigEditProviderInterface {

  /**
   * Get the configuration edit options for this node processor.
   *
   * Returns a value object containing all options for how the node's
   * configuration can be edited dynamically. This may include:
   * - Dynamic schema endpoint for fetching config schema at runtime
   * - External edit link for opening admin/third-party forms
   * - API endpoints for CRUD operations
   * - UI behavior options (loading messages, caching, etc.)
   *
   * @return \Drupal\flowdrop\DTO\ConfigEdit\ConfigEdit
   *   The configuration edit options.
   */
  public function getConfigEdit(): ConfigEdit;

}
