<?php

declare(strict_types=1);

namespace Drupal\flowdrop\Plugin\FlowDropNodeProcessor;

use Drupal\flowdrop\DTO\ExecutionContextDTO;

/**
 * Interface for processors that need access to execution context.
 *
 * Implement this interface when your processor needs access to:
 * - Initial workflow data (trigger payload, input data)
 * - Workflow, pipeline, or execution identifiers
 * - Execution metadata.
 *
 * The runtime will automatically inject the ExecutionContextDTO before
 * calling execute() on processors that implement this interface.
 *
 * Example usage:
 * @code
 * class MyProcessor extends AbstractFlowDropNodeProcessor
 *     implements ExecutionContextAwareInterface {
 *
 *   protected ?ExecutionContextDTO $executionContext = null;
 *
 *   public function setExecutionContext(ExecutionContextDTO $context): void {
 *     $this->executionContext = $context;
 *   }
 *
 *   protected function process(ParameterBagInterface $params): array {
 *     // Type-safe access to workflow data
 *     $initialData = $this->executionContext->getInitialData();
 *     $workflowId = $this->executionContext->getWorkflowId();
 *     $userId = $this->executionContext->getInitialDataValue('user_id');
 *
 *     return ['data' => $initialData];
 *   }
 * }
 * @endcode
 *
 * @see \Drupal\flowdrop\DTO\ExecutionContextDTO
 * @see \Drupal\flowdrop\Plugin\FlowDropNodeProcessor\AbstractFlowDropNodeProcessor
 */
interface ExecutionContextAwareInterface {

  /**
   * Sets the execution context for this processor.
   *
   * This method is called by the runtime before execute() is invoked.
   * Store the context for use in your process() method.
   *
   * @param \Drupal\flowdrop\DTO\ExecutionContextDTO $context
   *   The execution context containing workflow data and identifiers.
   */
  public function setExecutionContext(ExecutionContextDTO $context): void;

}
