<?php

declare(strict_types=1);

namespace Drupal\flowdrop_job;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityListBuilder;

/**
 * Provides a list controller for the flowdrop job entity type.
 */
final class FlowDropJobListBuilder extends EntityListBuilder {

  /**
   * {@inheritdoc}
   */
  public function buildHeader(): array {
    $header['id'] = $this->t('ID');
    $header['label'] = $this->t('Label');
    $header['status'] = $this->t('Status');
    $header['uid'] = $this->t('Author');
    $header['created'] = $this->t('Created');
    $header['changed'] = $this->t('Updated');
    return $header + parent::buildHeader();
  }

  /**
   * {@inheritdoc}
   *
   * @return array<string, mixed>
   *   An array of row data for the entity.
   */
  public function buildRow(EntityInterface $entity): array {
    /** @var \Drupal\flowdrop_job\FlowDropJobInterface $entity */
    $row["id"] = $entity->id();
    $row["label"] = $entity->toLink();
    $row["status"] = $this->buildStatusPill((bool) $entity->get("status")->value);
    $username_options = [
      "label" => "hidden",
    ];
    $row["uid"]["data"] = $entity->get("uid")->view($username_options);
    $row["created"]["data"] = $entity->get("created")->view(["label" => "hidden"]);
    $row["changed"]["data"] = $entity->get("changed")->view(["label" => "hidden"]);
    return $row + parent::buildRow($entity);
  }

  /**
   * Builds a status pill render array for table cell display.
   *
   * @param bool $isEnabled
   *   Whether the entity is enabled.
   *
   * @return array<string, mixed>
   *   A table cell render array containing the status pill component.
   */
  private function buildStatusPill(bool $isEnabled): array {
    return [
      "data" => [
        "#type" => "component",
        "#component" => "flowdrop_ui_components:pill",
        "#props" => [
          "label" => $isEnabled ? (string) $this->t("Enabled") : (string) $this->t("Disabled"),
          "variant" => $isEnabled ? "success" : "neutral",
          "style" => "soft",
          "size" => "sm",
        ],
      ],
    ];
  }

}
