<?php

declare(strict_types=1);

namespace Drupal\flowdrop_node_processor\Plugin\FlowDropNodeProcessor;

use Drupal\flowdrop\Attribute\FlowDropNodeProcessor;
use Drupal\flowdrop\DTO\ParameterBagInterface;
use Drupal\flowdrop\DTO\ValidationResult;
use Drupal\flowdrop\Plugin\FlowDropNodeProcessor\AbstractFlowDropNodeProcessor;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Executor for Chat Model nodes.
 */
#[FlowDropNodeProcessor(
  id: "chat_model",
  label: new \Drupal\Core\StringTranslation\TranslatableMarkup("Chat Model"),
  description: "AI chat model integration",
  version: "1.0.0"
)]
class ChatModel extends AbstractFlowDropNodeProcessor {

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ): static {
    return new static($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public function validateParams(array $params): ValidationResult {
    // Chat model nodes can accept any inputs or none.
    return ValidationResult::success();
  }

  /**
   * {@inheritdoc}
   */
  protected function process(ParameterBagInterface $params): array {
    $model = $params->get("model", "gpt-3.5-turbo");
    $temperature = $params->get("temperature", 0.7);
    $message = $params->get("message", "");

    // Simulate chat model response.
    $response = "This is a simulated response from the {$model} model. In a real implementation, this would call the actual AI model API.";

    return [
      "response" => $response,
      "model" => $model,
      "temperature" => $temperature,
      // Rough estimation.
      "tokens_used" => strlen($response) / 4,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getParameterSchema(): array {
    return [
      "type" => "object",
      "properties" => [
        "message" => [
          "type" => "string",
          "title" => "Message",
          "description" => "The message to send to the model",
          "required" => FALSE,
        ],
        "model" => [
          "type" => "string",
          "title" => "Model",
          "description" => "The chat model to use",
          "default" => "gpt-3.5-turbo",
        ],
        "temperature" => [
          "type" => "number",
          "title" => "Temperature",
          "description" => "Model temperature (0.0 to 2.0)",
          "default" => 0.7,
          "minimum" => 0,
          "maximum" => 2,
          "step" => "0.01",
          "format" => "range",
        ],
        "maxTokens" => [
          "type" => "integer",
          "title" => "Max Tokens",
          "description" => "Maximum tokens in response",
          "default" => 1000,
        ],
        "systemPrompt" => [
          "type" => "string",
          "title" => "System Prompt",
          "description" => "System prompt for the model",
          "format" => "multiline",
          "default" => "",
        ],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getOutputSchema(): array {
    return [
      "type" => "object",
      "properties" => [
        "response" => [
          "type" => "string",
          "description" => "The model response",
        ],
        "model" => [
          "type" => "string",
          "description" => "The model used",
        ],
        "temperature" => [
          "type" => "number",
          "description" => "The temperature setting",
        ],
        "tokens_used" => [
          "type" => "integer",
          "description" => "Number of tokens used",
        ],
      ],
    ];
  }

}
