<?php

declare(strict_types=1);

namespace Drupal\flowdrop_node_processor\Plugin\FlowDropNodeProcessor;

use Drupal\flowdrop\Attribute\FlowDropNodeProcessor;
use Drupal\flowdrop\DTO\ParameterBagInterface;
use Drupal\flowdrop\DTO\ValidationResult;
use Drupal\flowdrop\Plugin\FlowDropNodeProcessor\AbstractFlowDropNodeProcessor;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Executor for Chat Output nodes.
 */
#[FlowDropNodeProcessor(
  id: "chat_output",
  label: new \Drupal\Core\StringTranslation\TranslatableMarkup("Chat Output"),
  description: "Output chat messages in various formats",
  version: "1.0.0"
)]
class ChatOutput extends AbstractFlowDropNodeProcessor {

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ): static {
    return new static($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public function validateParams(array $params): ValidationResult {
    // Chat output can accept any inputs or none.
    return ValidationResult::success();
  }

  /**
   * {@inheritdoc}
   */
  protected function process(ParameterBagInterface $params): array {
    $message = $params->get("message", "");
    $format = $params->get("format", "text");
    $timestamp = $params->get("showTimestamp", FALSE) ? time() : NULL;

    // If inputs are provided, use the first input as the message.
    $inputMessage = $params->get("message");
    if ($inputMessage !== NULL) {
      $message = $inputMessage;
    }

    return [
      "message" => $message,
      "format" => $format,
      "timestamp" => $timestamp,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getParameterSchema(): array {
    return [
      "type" => "object",
      "properties" => [
        "message" => [
          "type" => "string",
          "title" => "Message",
          "description" => "Default chat message",
          "default" => "",
        ],
        "format" => [
          "type" => "string",
          "title" => "Format",
          "description" => "Message format (text, markdown, html)",
          "default" => "text",
        ],
        "showTimestamp" => [
          "type" => "boolean",
          "title" => "Show Timestamp",
          "description" => "Whether to include timestamp",
          "default" => FALSE,
        ],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getOutputSchema(): array {
    return [
      "type" => "object",
      "properties" => [
        "message" => [
          "type" => "string",
          "description" => "The chat message",
        ],
        "format" => [
          "type" => "string",
          "description" => "The message format",
        ],
        "timestamp" => [
          "type" => "integer",
          "description" => "The message timestamp",
        ],
      ],
    ];
  }

}
