<?php

declare(strict_types=1);

namespace Drupal\flowdrop_node_processor\Plugin\FlowDropNodeProcessor;

use Drupal\flowdrop\Attribute\FlowDropNodeProcessor;
use Drupal\flowdrop\DTO\ParameterBagInterface;
use Drupal\flowdrop\DTO\ValidationResult;
use Drupal\flowdrop\Plugin\FlowDropNodeProcessor\AbstractFlowDropNodeProcessor;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Executor for DateTime nodes.
 */
#[FlowDropNodeProcessor(
  id: "date_time",
  label: new \Drupal\Core\StringTranslation\TranslatableMarkup("Date Time"),
  description: "Date and time operations",
  version: "1.0.0"
)]
class DateTime extends AbstractFlowDropNodeProcessor {

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ): static {
    return new static($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public function validateParams(array $params): ValidationResult {
    // DateTime nodes can accept any inputs or none.
    return ValidationResult::success();
  }

  /**
   * {@inheritdoc}
   */
  protected function process(ParameterBagInterface $params): array {
    $format = $params->get("format", "Y-m-d H:i:s");
    $timezone = $params->get("timezone", "UTC");
    $operation = $params->get("operation", "current");

    $now = new \DateTime("now", new \DateTimeZone($timezone));

    $result = "";
    switch ($operation) {
      case "current":
        $result = $now->format($format);
        break;

      case "timestamp":
        $result = $now->getTimestamp();
        break;

      case "iso":
        $result = $now->format("c");
        break;

      case "unix":
        $result = time();
        break;
    }

    return [
      "datetime" => $result,
      "timestamp" => $now->getTimestamp(),
      "format" => $format,
      "timezone" => $timezone,
      "operation" => $operation,
      "iso" => $now->format("c"),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getParameterSchema(): array {
    return [
      "type" => "object",
      "properties" => [
        "datetime" => [
          "type" => "string",
          "title" => "DateTime",
          "description" => "Optional datetime input to process",
          "required" => FALSE,
        ],
        "operation" => [
          "type" => "string",
          "title" => "Operation",
          "description" => "DateTime operation",
          "default" => "current",
          "enum" => [
            "current",
            "timestamp",
            "iso",
            "unix",
          ],
        ],
        "format" => [
          "type" => "string",
          "title" => "Format",
          "description" => "PHP date format string",
          "default" => "Y-m-d H:i:s",
        ],
        "timezone" => [
          "type" => "string",
          "title" => "Timezone",
          "description" => "Timezone to use",
          "default" => "UTC",
        ],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getOutputSchema(): array {
    return [
      "type" => "object",
      "properties" => [
        "datetime" => [
          "type" => "string",
          "description" => "The formatted datetime",
        ],
        "timestamp" => [
          "type" => "integer",
          "description" => "The Unix timestamp",
        ],
        "format" => [
          "type" => "string",
          "description" => "The format used",
        ],
        "timezone" => [
          "type" => "string",
          "description" => "The timezone used",
        ],
        "operation" => [
          "type" => "string",
          "description" => "The operation performed",
        ],
        "iso" => [
          "type" => "string",
          "description" => "ISO 8601 formatted datetime",
        ],
      ],
    ];
  }

}
