<?php

declare(strict_types=1);

namespace Drupal\flowdrop_node_processor\Plugin\FlowDropNodeProcessor;

use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\flowdrop\Attribute\FlowDropNodeProcessor;
use Drupal\flowdrop\DTO\ParameterBagInterface;
use Drupal\flowdrop\DTO\ValidationResult;
use Drupal\flowdrop\Plugin\FlowDropNodeProcessor\AbstractFlowDropNodeProcessor;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Notes node processor for adding documentation and comments to workflows.
 */
#[FlowDropNodeProcessor(
  id: "note",
  label: new TranslatableMarkup("Note"),
  description: "Add notes and comments",
  version: "1.0.0"
)]
final class Note extends AbstractFlowDropNodeProcessor {

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ): static {
    return new static($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public function validateParams(array $params): ValidationResult {
    // Notes nodes don't require any inputs.
    return ValidationResult::success();
  }

  /**
   * {@inheritdoc}
   */
  protected function process(ParameterBagInterface $params): array {
    // Notes nodes don't process data, they just store documentation.
    return [
      "content" => $params->get("content", ""),
      "noteType" => $params->get("noteType", "info"),
      "message" => "Note content available for reference",
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getParameterSchema(): array {
    return [
      "type" => "object",
      "properties" => [
        "content" => [
          "type" => "string",
          "title" => "Note Content",
          "description" => "Documentation or comment text (supports Markdown)",
          "format" => "markdown",
          "default" => "# Workflow Notes\n\nAdd your documentation here using **Markdown** formatting.\n\n## Features\n- Supports **bold** and *italic* text\n- Create lists and code blocks\n- Add links and more!",
        ],
        "noteType" => [
          "type" => "string",
          "title" => "Note Type",
          "description" => "Visual style and color of the note",
          "default" => "info",
          "enum" => ["info", "warning", "success", "error", "note"],
        ],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getOutputSchema(): array {
    return [
      "type" => "object",
      "properties" => [
        "content" => [
          "type" => "string",
          "title" => "Note Content",
          "description" => "The markdown content of the note",
          "format" => "markdown",
        ],
        "noteType" => [
          "type" => "string",
          "title" => "Note Type",
          "description" => "The visual type of the note (info, warning, success, error, note)",
        ],
        "message" => [
          "type" => "string",
          "title" => "Message",
          "description" => "Status message about the note",
        ],
      ],
    ];
  }

}
