<?php

declare(strict_types=1);

namespace Drupal\flowdrop_runtime\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Extension\ModuleExtensionList;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Controller for the execution overview page.
 *
 * Provides a landing page for all execution-related actions in FlowDrop.
 */
class ExecutionController extends ControllerBase {

  /**
   * The module extension list.
   *
   * @var \Drupal\Core\Extension\ModuleExtensionList
   */
  protected ModuleExtensionList $moduleExtensionList;

  /**
   * Constructs an ExecutionController.
   *
   * @param \Drupal\Core\Extension\ModuleExtensionList $moduleExtensionList
   *   The module extension list service.
   */
  public function __construct(ModuleExtensionList $moduleExtensionList) {
    $this->moduleExtensionList = $moduleExtensionList;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get("extension.list.module"),
    );
  }

  /**
   * Renders the execution overview page.
   *
   * @return array<string, mixed>
   *   A render array for the execution overview.
   */
  public function overview(): array {
    // Get module path for assets.
    $modulePath = "/" . $this->moduleExtensionList->getPath("flowdrop_ui_components");

    $build = [
      "#type" => "container",
      "#attributes" => [
        "class" => ["fd-fullscreen-layout"],
      ],
      "#attached" => [
        "library" => [
          "flowdrop_ui_components/fullscreen-layout",
          "flowdrop_ui_components/base",
        ],
      ],
    ];

    // Main content.
    $build["main"] = [
      "#type" => "container",
      "#attributes" => [
        "class" => ["fd-fullscreen-main", "fd-fullscreen-main--constrained"],
      ],
    ];

    $build["main"]["description"] = [
      "#type" => "html_tag",
      "#tag" => "p",
      "#value" => $this->t("Execute workflows and pipelines, or interact with running executions."),
      "#attributes" => [
        "class" => ["fd-text-secondary", "fd-mb-6"],
      ],
    ];

    $links = [
      "#type" => "container",
      "#attributes" => [
        "class" => ["fd-actions-grid", "fd-stagger-in"],
      ],
    ];

    // Execute Workflow.
    $links["execute_workflow"] = [
      "#type" => "component",
      "#component" => "flowdrop_ui_components:action-link",
      "#props" => [
        "title" => $this->t("Run Workflow"),
        "description" => $this->t("Start a workflow execution with pipeline options"),
        "url" => Url::fromRoute("flowdrop.execute.workflow")->toString(),
        "icon" => '<svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="1.5" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" d="M5.25 5.653c0-1.426 1.526-2.33 2.777-1.643l11.54 6.348c1.554.855 1.554 3.031 0 3.886l-11.54 6.347c-1.251.688-2.777-.217-2.777-1.643V5.653z"/></svg>',
      ],
    ];

    // Execute Pipeline.
    $links["execute_pipeline"] = [
      "#type" => "component",
      "#component" => "flowdrop_ui_components:action-link",
      "#props" => [
        "title" => $this->t("Run Pipeline"),
        "description" => $this->t("Run an existing pipeline directly"),
        "url" => Url::fromRoute("flowdrop.execute.pipeline")->toString(),
        "icon" => '<svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="1.5" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" d="M5.25 5.653c0-1.426 1.526-2.33 2.777-1.643l11.54 6.348c1.554.855 1.554 3.031 0 3.886l-11.54 6.347c-1.251.688-2.777-.217-2.777-1.643V5.653z"/></svg>',
      ],
    ];

    // Pipeline Interaction.
    $links["interact"] = [
      "#type" => "component",
      "#component" => "flowdrop_ui_components:action-link",
      "#props" => [
        "title" => $this->t("Pipeline Interaction"),
        "description" => $this->t("Manage and control running pipelines"),
        "url" => Url::fromRoute("flowdrop.execute.interact")->toString(),
        "icon" => '<svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="1.5" stroke="currentColor" class="size-6">
  <path stroke-linecap="round" stroke-linejoin="round" d="M9.879 7.519c1.171-1.025 3.071-1.025 4.242 0 1.172 1.025 1.172 2.687 0 3.712-.203.179-.43.326-.67.442-.745.361-1.45.999-1.45 1.827v.75M21 12a9 9 0 1 1-18 0 9 9 0 0 1 18 0Zm-9 5.25h.008v.008H12v-.008Z" />
</svg>
',
      ],
    ];

    $build["main"]["links"] = $links;

    return $build;
  }

}
