<?php

declare(strict_types=1);

namespace Drupal\flowdrop_runtime\Service\Logging;

use Drupal\flowdrop\DTO\ParameterBagInterface;
use Drupal\flowdrop\DTO\ValidationResult;
use Drupal\flowdrop_runtime\DTO\LogVerbosity;
use Drupal\flowdrop_runtime\DTO\Runtime\NodeExecutionResult;

/**
 * Centralized execution logging service interface.
 *
 * Handles all routine logging for node execution, controlled by verbosity
 * settings. This removes the need for individual processors to implement
 * their own logging.
 *
 * Processors should NOT log:
 * - "Node executed successfully" (handled here)
 * - "Input received: @data" (handled here based on verbosity)
 * - "Validation failed for parameter @param" (handled here)
 * - "Processing complete" (handled here)
 *
 * Processors SHOULD still log:
 * - External service retries
 * - Fallback usage
 * - Rate limit hits
 * - Unexpected data shapes worth flagging
 *
 * @see \Drupal\flowdrop_runtime\Service\Logging\ExecutionLogger
 */
interface ExecutionLoggerInterface {

  /**
   * Set the logging verbosity level.
   *
   * @param \Drupal\flowdrop_runtime\DTO\LogVerbosity $verbosity
   *   The verbosity level.
   *
   * @return static
   *   The logger for chaining.
   */
  public function setVerbosity(LogVerbosity $verbosity): static;

  /**
   * Get the current logging verbosity level.
   *
   * @return \Drupal\flowdrop_runtime\DTO\LogVerbosity
   *   The current verbosity level.
   */
  public function getVerbosity(): LogVerbosity;

  /**
   * Log node execution start.
   *
   * Logged at STANDARD verbosity and above.
   *
   * @param string $executionId
   *   The execution ID.
   * @param string $nodeId
   *   The node instance ID.
   * @param string $nodeType
   *   The node type ID.
   */
  public function logNodeStart(
    string $executionId,
    string $nodeId,
    string $nodeType,
  ): void;

  /**
   * Log resolved parameters.
   *
   * Logged at VERBOSE verbosity (summary) or DEBUG (full values).
   *
   * @param string $executionId
   *   The execution ID.
   * @param string $nodeId
   *   The node instance ID.
   * @param \Drupal\flowdrop\DTO\ParameterBagInterface $params
   *   The resolved parameters.
   */
  public function logParameters(
    string $executionId,
    string $nodeId,
    ParameterBagInterface $params,
  ): void;

  /**
   * Log node execution completion.
   *
   * Logged at STANDARD verbosity and above.
   *
   * @param string $executionId
   *   The execution ID.
   * @param \Drupal\flowdrop_runtime\DTO\Runtime\NodeExecutionResult $result
   *   The execution result.
   */
  public function logNodeComplete(
    string $executionId,
    NodeExecutionResult $result,
  ): void;

  /**
   * Log validation result (pre-execution).
   *
   * Failures logged at STANDARD, warnings at VERBOSE.
   *
   * @param string $executionId
   *   The execution ID.
   * @param string $nodeId
   *   The node instance ID.
   * @param \Drupal\flowdrop\DTO\ValidationResult $result
   *   The validation result.
   */
  public function logValidation(
    string $executionId,
    string $nodeId,
    ValidationResult $result,
  ): void;

  /**
   * Log node execution failure.
   *
   * Always logged regardless of verbosity level.
   *
   * @param string $executionId
   *   The execution ID.
   * @param string $nodeId
   *   The node instance ID.
   * @param \Throwable $exception
   *   The exception that caused the failure.
   * @param float $executionTime
   *   The execution time in seconds.
   */
  public function logNodeFailure(
    string $executionId,
    string $nodeId,
    \Throwable $exception,
    float $executionTime,
  ): void;

  /**
   * Log node skipped (branch not active).
   *
   * Logged at STANDARD verbosity and above.
   *
   * @param string $executionId
   *   The execution ID.
   * @param string $nodeId
   *   The node instance ID.
   * @param string $nodeType
   *   The node type ID.
   * @param string $reason
   *   The reason for skipping.
   */
  public function logNodeSkipped(
    string $executionId,
    string $nodeId,
    string $nodeType,
    string $reason,
  ): void;

  /**
   * Log workflow start.
   *
   * Logged at STANDARD verbosity and above.
   *
   * @param string $executionId
   *   The execution ID.
   * @param string $workflowId
   *   The workflow ID.
   * @param int $nodeCount
   *   Number of nodes to execute.
   */
  public function logWorkflowStart(
    string $executionId,
    string $workflowId,
    int $nodeCount,
  ): void;

  /**
   * Log workflow completion.
   *
   * Logged at STANDARD verbosity and above.
   *
   * @param string $executionId
   *   The execution ID.
   * @param string $workflowId
   *   The workflow ID.
   * @param int $executedNodes
   *   Number of nodes executed.
   * @param int $skippedNodes
   *   Number of nodes skipped.
   * @param float $totalTime
   *   Total execution time in seconds.
   */
  public function logWorkflowComplete(
    string $executionId,
    string $workflowId,
    int $executedNodes,
    int $skippedNodes,
    float $totalTime,
  ): void;

  /**
   * Log workflow failure.
   *
   * Always logged regardless of verbosity level.
   *
   * @param string $executionId
   *   The execution ID.
   * @param string $workflowId
   *   The workflow ID.
   * @param \Throwable $exception
   *   The exception that caused the failure.
   */
  public function logWorkflowFailure(
    string $executionId,
    string $workflowId,
    \Throwable $exception,
  ): void;

}
