# Manual Trigger Testing Guide

This guide explains how to manually test FlowDrop triggers for development and debugging.

## Quick Start

After clearing cache, use the new Drush commands:

```bash
drush cr
drush flowdrop:trigger:list
drush flowdrop:trigger:test entity.insert node 1
```

## Method 1: Drush Commands (Recommended)

### List Active Triggers

```bash
drush flowdrop:trigger:list
# or
drush fd:trigger:list
```

This shows all active trigger configurations grouped by event type.

### Test a Trigger Event

```bash
# Test entity.insert event
drush flowdrop:trigger:test entity.insert node 1

# Test entity.update event (includes original entity)
drush flowdrop:trigger:test entity.update node 1

# Test entity.delete event
drush flowdrop:trigger:test entity.delete node 1

# Test with different entity types
drush flowdrop:trigger:test entity.insert user 1
drush flowdrop:trigger:test entity.insert taxonomy_term 1
```

### Test a Specific Trigger Configuration

```bash
# Test a specific trigger config by ID
drush flowdrop:trigger:test-config my_workflow__node_123 entity.insert node 1
```

This tests whether a specific trigger configuration matches and fires.

## Method 2: Drush PHP Eval

For more control, use Drush's PHP evaluator:

```bash
# Test entity.insert
drush php:eval "\$entity = \Drupal::entityTypeManager()->getStorage('node')->load(1); \$manager = \Drupal::service('flowdrop_trigger.trigger_manager'); \$manager->processEvent('entity.insert', \$entity);"

# Test entity.update (with original)
drush php:eval "\$storage = \Drupal::entityTypeManager()->getStorage('node'); \$entity = \$storage->load(1); \$original = \$storage->loadUnchanged(1); \$manager = \Drupal::service('flowdrop_trigger.trigger_manager'); \$manager->processEvent('entity.update', \$entity, \$original);"

# Test entity.delete
drush php:eval "\$entity = \Drupal::entityTypeManager()->getStorage('node')->load(1); \$manager = \Drupal::service('flowdrop_trigger.trigger_manager'); \$manager->processEvent('entity.delete', \$entity);"
```

## Method 3: Simulate Real Entity Events

Create, update, or delete entities to trigger hooks naturally:

```bash
# Create a new node (triggers entity.insert)
drush content:create node article --title="Test Article" --body="Test body"

# Update an existing node (triggers entity.update)
drush php:eval "\$node = \Drupal::entityTypeManager()->getStorage('node')->load(1); \$node->setTitle('Updated Title'); \$node->save();"

# Delete a node (triggers entity.delete)
drush entity:delete node 1
```

## Method 4: Custom PHP Script

Create a custom PHP script for repeated testing:

```php
<?php
// test-trigger.php

use Drupal\Core\DrupalKernel;
use Symfony\Component\HttpFoundation\Request;

$autoloader = require_once 'web/autoload.php';
$kernel = DrupalKernel::createFromRequest(Request::createFromGlobals(), $autoloader, 'prod');
$kernel->boot();
$container = $kernel->getContainer();

// Get services
$entityTypeManager = $container->get('entity_type.manager');
$triggerManager = $container->get('flowdrop_trigger.trigger_manager');

// Load entity
$storage = $entityTypeManager->getStorage('node');
$entity = $storage->load(1);

if ($entity) {
  // Fire trigger
  $triggerManager->processEvent('entity.insert', $entity);
  echo "Trigger fired successfully!\n";
} else {
  echo "Entity not found!\n";
}
```

Run with:
```bash
php test-trigger.php
```

## Available Event Types

### Entity Events

| Event Type | Description | When It Fires |
|------------|-------------|---------------|
| `entity.insert` | Entity created | After entity is saved for the first time |
| `entity.update` | Entity updated | After entity is updated |
| `entity.delete` | Entity deleted | After entity is deleted |
| `entity.presave` | Before entity save | Before entity is saved (sync only) |
| `entity.load` | Entity loaded | When entity is loaded from database |
| `entity.create` | Entity created (before save) | When entity object is created |

### User Events

| Event Type | Description |
|------------|-------------|
| `user.login` | User logged in |
| `user.logout` | User logged out |

## Checking Logs

Monitor trigger execution in logs:

```bash
# Watch logs in real-time
drush watchdog:tail

# Filter by channel
drush watchdog:tail --filter=flowdrop_trigger

# View recent logs
drush watchdog:show --count=50 --filter=flowdrop_trigger
```

## Debugging Tips

### 1. Check Trigger Configuration

```bash
# List all active triggers
drush flowdrop:trigger:list

# Check specific trigger config via API
curl -H "X-CSRF-Token: $(drush eval 'echo \Drupal::csrfToken()->get("rest")')" \
  http://localhost/api/flowdrop/triggers/my_workflow__node_123
```

### 2. Verify Workflow Status

```bash
# Check if workflow is enabled
drush php:eval "\$workflow = \Drupal::entityTypeManager()->getStorage('flowdrop_workflow')->load('my_workflow'); echo \$workflow ? (\$workflow->status() ? 'Enabled' : 'Disabled') : 'Not found';"
```

### 3. Check Trigger Conditions

Trigger conditions are checked in `FlowDropTriggerConfig::matches()`. Common conditions:
- `entity_types`: Filter by entity type (e.g., `["node", "user"]`)
- `bundles`: Filter by bundle (e.g., `["article", "page"]`)
- `user_roles`: Filter by user role (for user events)

### 4. Enable Debug Logging

```bash
# Set log level to debug
drush config:set system.logging error_level debug
drush cr
```

## Common Issues

### Trigger Not Firing

1. **Check trigger status**: Ensure trigger config has `status: true`
2. **Check workflow status**: Ensure workflow is enabled
3. **Check conditions**: Verify entity type/bundle matches trigger conditions
4. **Check event type**: Ensure you're testing the correct event type
5. **Check logs**: Look for error messages in watchdog

### Trigger Fires But Workflow Doesn't Execute

1. **Check orchestrator**: Verify orchestrator settings in trigger config
2. **Check pipeline**: Look for pipeline execution errors
3. **Check workflow definition**: Ensure workflow JSON is valid
4. **Check permissions**: Verify user has permission to execute workflows

## Example: Complete Test Workflow

```bash
# 1. List all triggers
drush flowdrop:trigger:list

# 2. Create a test node
drush content:create node article --title="Trigger Test" --body="Testing triggers"

# 3. Get the node ID (from output above, e.g., 42)

# 4. Test entity.insert trigger
drush flowdrop:trigger:test entity.insert node 42

# 5. Update the node
drush php:eval "\$node = \Drupal::entityTypeManager()->getStorage('node')->load(42); \$node->setTitle('Updated'); \$node->save();"

# 6. Test entity.update trigger
drush flowdrop:trigger:test entity.update node 42

# 7. Check logs
drush watchdog:show --count=20 --filter=flowdrop_trigger

# 8. Clean up
drush entity:delete node 42
```

## Advanced: Testing with Custom Context

For more complex testing scenarios, you can create a custom entity or modify the context:

```php
<?php
// Custom test script
$container = \Drupal::getContainer();
$entityTypeManager = $container->get('entity_type.manager');
$triggerManager = $container->get('flowdrop_trigger.trigger_manager');

// Create a test entity programmatically
$node = $entityTypeManager->getStorage('node')->create([
  'type' => 'article',
  'title' => 'Test Article',
  'body' => ['value' => 'Test body'],
]);
$node->save();

// Fire trigger
$triggerManager->processEvent('entity.insert', $node);

// Clean up
$node->delete();
```

## See Also

- [README.md](README.md) - Module overview and API documentation
- [config-edit.md](config-edit.md) - Trigger configuration guide


