<?php

declare(strict_types=1);

namespace Drupal\flowdrop_trigger;

use Drupal\Component\Plugin\PluginInspectionInterface;
use Drupal\Core\Entity\EntityInterface;

/**
 * Interface for FlowDrop Event Type plugins.
 *
 * Event type plugins define different types of events that can trigger
 * workflow execution. Each plugin specifies:
 * - The event identifier (e.g., "entity.insert")
 * - Default orchestration settings
 * - How to extract data from the event context.
 */
interface FlowDropEventTypeInterface extends PluginInspectionInterface {

  /**
   * Get the event type ID.
   *
   * @return string
   *   The event type ID (e.g., "entity.insert", "user.login").
   */
  public function getEventId(): string;

  /**
   * Get the human-readable label.
   *
   * @return string
   *   The label for this event type.
   */
  public function getLabel(): string;

  /**
   * Get the event description.
   *
   * @return string
   *   A description of when this event fires.
   */
  public function getDescription(): string;

  /**
   * Get the event category.
   *
   * @return string
   *   The category (entity, user, system).
   */
  public function getCategory(): string;

  /**
   * Get the default orchestrator for this event type.
   *
   * @return string
   *   The orchestrator type (synchronous, asynchronous).
   */
  public function getDefaultOrchestrator(): string;

  /**
   * Get the default execution mode.
   *
   * @return string
   *   The execution mode (sync, async).
   */
  public function getDefaultExecutionMode(): string;

  /**
   * Get supported entity types for this event.
   *
   * @return array<string>
   *   Array of supported entity type IDs, or empty for all.
   */
  public function getSupportedEntityTypes(): array;

  /**
   * Check if this event type supports the given entity type.
   *
   * @param string $entityType
   *   The entity type ID.
   *
   * @return bool
   *   TRUE if supported, FALSE otherwise.
   */
  public function supportsEntityType(string $entityType): bool;

  /**
   * Extract trigger data from the event context.
   *
   * This method extracts relevant data from the entity and event
   * context to be passed to the workflow execution.
   *
   * @param \Drupal\Core\Entity\EntityInterface|null $entity
   *   The entity involved in the event, or NULL for non-entity events.
   * @param \Drupal\Core\Entity\EntityInterface|null $originalEntity
   *   The original entity state (for updates), or NULL.
   * @param array<string, mixed> $context
   *   Additional context data.
   *
   * @return array<string, mixed>
   *   The extracted trigger data.
   */
  public function extractTriggerData(
    ?EntityInterface $entity,
    ?EntityInterface $originalEntity = NULL,
    array $context = [],
  ): array;

  /**
   * Get the unified parameter schema for this event type.
   *
   * Returns a JSON Schema describing the parameters
   * available for triggers using this event type.
   *
   * @return array<string, mixed>
   *   The JSON schema for parameters.
   */
  public function getParameterSchema(): array;

}
