<?php

declare(strict_types=1);

namespace Drupal\flowdrop_trigger\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * General settings form for FlowDrop Trigger module.
 *
 * Provides global configuration options that apply across all trigger
 * categories, such as default orchestrator and logging settings.
 *
 * Category-specific settings are managed in dedicated forms:
 * - TriggerSettingsEntityForm: Entity event settings
 * - TriggerSettingsFormEventsForm: Form event settings
 * - TriggerSettingsUserForm: User event settings
 *
 * @see \Drupal\flowdrop_trigger\Form\TriggerSettingsEntityForm
 * @see \Drupal\flowdrop_trigger\Form\TriggerSettingsFormEventsForm
 * @see \Drupal\flowdrop_trigger\Form\TriggerSettingsUserForm
 */
final class TriggerSettingsGeneralForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return "flowdrop_trigger_settings_general";
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ["flowdrop_trigger.settings"];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config("flowdrop_trigger.settings");

    $form["description"] = [
      "#type" => "markup",
      "#markup" => "<p>" . $this->t("Configure general settings for FlowDrop triggers. Category-specific settings (Entity, Form, User events) are available in their respective tabs.") . "</p>",
    ];

    // =========================================================================
    // Orchestrator Settings.
    // =========================================================================
    $form["orchestrator"] = [
      "#type" => "details",
      "#title" => $this->t("Orchestrator Settings"),
      "#open" => TRUE,
    ];

    $form["orchestrator"]["default_orchestrator"] = [
      "#type" => "radios",
      "#title" => $this->t("Default Orchestrator"),
      "#description" => $this->t("Select the default orchestrator type for workflow execution. Individual triggers can override this setting."),
      "#options" => [
        "synchronous" => $this->t("Synchronous - Execute immediately in the same request"),
        "synchronous_pipeline" => $this->t("Synchronous Pipeline - Execute synchronously with job tracking"),
        "asynchronous" => $this->t("Asynchronous - Queue for background execution"),
      ],
      "#default_value" => $config->get("default_orchestrator") ?? "asynchronous",
    ];

    // =========================================================================
    // Logging Settings.
    // =========================================================================
    $form["logging"] = [
      "#type" => "details",
      "#title" => $this->t("Logging Settings"),
      "#open" => TRUE,
    ];

    $form["logging"]["log_level"] = [
      "#type" => "select",
      "#title" => $this->t("Log Level"),
      "#description" => $this->t("Set the verbosity of trigger-related log messages."),
      "#options" => [
        "debug" => $this->t("Debug - All messages including detailed debugging"),
        "info" => $this->t("Info - General operational information"),
        "warning" => $this->t("Warning - Potential issues and warnings"),
        "error" => $this->t("Error - Only error messages"),
      ],
      "#default_value" => $config->get("log_level") ?? "info",
    ];

    // =========================================================================
    // Advanced Settings.
    // =========================================================================
    $form["advanced"] = [
      "#type" => "details",
      "#title" => $this->t("Advanced Settings"),
      "#open" => FALSE,
    ];

    $form["advanced"]["info"] = [
      "#type" => "markup",
      "#markup" => "<p>" . $this->t("Advanced settings for fine-tuning trigger behavior. Use with caution.") . "</p>",
    ];

    // Placeholder for future advanced settings.
    $form["advanced"]["placeholder"] = [
      "#type" => "markup",
      "#markup" => "<p><em>" . $this->t("Additional advanced settings will be available in future versions.") . "</em></p>",
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $config = $this->config("flowdrop_trigger.settings");

    $config->set("default_orchestrator", $form_state->getValue("default_orchestrator"));
    $config->set("log_level", $form_state->getValue("log_level"));

    $config->save();

    parent::submitForm($form, $form_state);
  }

}
