<?php

namespace Drupal\flowdrop_trigger\Hook;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\flowdrop_trigger\Service\TriggerManager;

/**
 * Implements entity hooks for FlowDrop ecosystem.
 */
class EntityHooks {

  /**
   * Constructs a new ContentHooks object.
   */
  public function __construct(
    readonly protected TriggerManager $triggerManager,
  ) {}

  /**
   * Implements hook_entity_load().
   */
  #[Hook('entity_load')]
  public function entityLoad(array $entities, string $entity_type_id): void {
    foreach ($entities as $entity) {
      if ($entity instanceof ContentEntityInterface) {
        $this->triggerManager->processEvent('entity.load', $entity);
      }
    }
  }

  /**
   * Implements hook_entity_create().
   */
  #[Hook('entity_create')]
  public function entityCreate(EntityInterface $entity): void {
    if ($entity instanceof ContentEntityInterface) {
      $this->triggerManager->processEvent('entity.create', $entity);
    }
  }

  /**
   * Implements hook_entity_presave().
   */
  #[Hook('entity_presave')]
  public function entityPresave(EntityInterface $entity): void {
    if ($entity instanceof ContentEntityInterface) {
      $this->triggerManager->processEvent('entity.presave', $entity);
    }
  }

  /**
   * Implements hook_entity_insert().
   */
  #[Hook('entity_insert')]
  public function entityInsert(EntityInterface $entity): void {
    if ($entity instanceof ContentEntityInterface) {
      $this->triggerManager->processEvent('entity.insert', $entity);
    }
  }

  /**
   * Implements hook_entity_update().
   */
  #[Hook('entity_update')]
  public function entityUpdate(EntityInterface $entity): void {
    if ($entity instanceof ContentEntityInterface) {
      if ($entity->getEntityType()->hasKey('revision')) {
        // Make sure the subsequent actions will not create another revision
        // when they save this entity again.
        $entity->setNewRevision(FALSE);
        $entity->updateLoadedRevisionId();
      }
      $this->triggerManager->processEvent('entity.update', $entity);
    }
  }

  /**
   * Implements hook_entity_delete().
   */
  #[Hook('entity_delete')]
  public function entityDelete(EntityInterface $entity): void {
    if ($entity instanceof ContentEntityInterface) {
      $this->triggerManager->processEvent('entity.delete', $entity);
    }
  }

  /**
   * Implements hook_entity_revision_create().
   */
  #[Hook('entity_revision_create')]
  public function entityRevisionCreate(EntityInterface $new_revision, EntityInterface $entity, ?bool $keep_untranslatable_fields): void {
    if ($new_revision instanceof ContentEntityInterface && $entity instanceof ContentEntityInterface) {
      $this->triggerManager->processEvent('entity.revision_create', $new_revision, $entity);
    }
  }

  /**
   * Implements hook_entity_revision_delete().
   */
  #[Hook('entity_revision_delete')]
  public function entityRevisionDelete(EntityInterface $entity): void {
    if ($entity instanceof ContentEntityInterface) {
      $this->triggerManager->processEvent('entity.revision_delete', $entity);
    }
  }

  /**
   * Implements hook_entity_translation_create().
   */
  #[Hook('entity_translation_create')]
  public function entityTranslationCreate(EntityInterface $translation): void {
    if ($translation instanceof ContentEntityInterface) {
      $this->triggerManager->processEvent('entity.translation_create', $translation);
    }
  }

  /**
   * Implements hook_entity_translation_insert().
   */
  #[Hook('entity_translation_insert')]
  public function entityTranslationInsert(EntityInterface $translation): void {
    if ($translation instanceof ContentEntityInterface) {
      $this->triggerManager->processEvent('entity.translation_insert', $translation);
    }
  }

  /**
   * Implements hook_entity_translation_delete().
   */
  #[Hook('entity_translation_delete')]
  public function entityTranslationDelete(EntityInterface $translation): void {
    if ($translation instanceof ContentEntityInterface) {
      $this->triggerManager->processEvent('entity.translation_delete', $translation);
    }
  }

}
