<?php

declare(strict_types=1);

namespace Drupal\flowdrop_trigger\Plugin\FlowDropEventType;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\flowdrop_trigger\Attribute\FlowDropEventType;
use Drupal\flowdrop_trigger\FlowDropEventTypeInterface;
use Drupal\flowdrop_trigger\Plugin\FlowDropEventTypeBase;

/**
 * Event type for entity insert events.
 *
 * Triggers when a new entity is created and saved to the database.
 * This event fires after the entity has been successfully inserted.
 */
#[FlowDropEventType(
  id: "entity.insert",
  label: new TranslatableMarkup("Entity Insert"),
  description: new TranslatableMarkup("Triggered when a new entity is created"),
  category: "entity",
  defaultOrchestrator: "asynchronous",
  defaultExecutionMode: "async",
  supportedEntityTypes: [],
)]
class EntityInsert extends FlowDropEventTypeBase implements FlowDropEventTypeInterface {

  /**
   * {@inheritdoc}
   */
  public function extractTriggerData(
    ?EntityInterface $entity,
    ?EntityInterface $originalEntity = NULL,
    array $context = [],
  ): array {
    if ($entity === NULL) {
      throw new \InvalidArgumentException("Entity cannot be NULL for entity insert events.");
    }

    $data = $this->getBaseTriggerData($entity);

    // Add insert-specific data.
    $data["is_new"] = TRUE;
    $data["operation"] = "insert";

    return $data;
  }

  /**
   * {@inheritdoc}
   */
  public function getParameterSchema(): array {
    $baseSchema = parent::getParameterSchema();

    // Entity insert doesn't need additional configuration beyond base.
    return $baseSchema;
  }

}
