<?php

declare(strict_types=1);

namespace Drupal\flowdrop_ui\Service;

use Drupal\Core\Url;

/**
 * Service for generating FlowDrop endpoint configuration.
 */
class FlowDropEndpointConfigService {

  /**
   * Generate the default endpoint configuration for FlowDrop.
   *
   * @param string $base_url
   *   The base URL for the API endpoints.
   * @param array $overrides
   *   Optional overrides for the configuration.
   *
   * @return array
   *   The endpoint configuration array.
   */
  public function generateEndpointConfig(string $base_url, array $overrides = []): array {
    $default_config = [
      'baseUrl' => $base_url,
      'endpoints' => [
        'nodes' => [
          'list' => '/nodes',
          'get' => '/nodes/{id}',
          'byCategory' => '/nodes?category={category}',
          'metadata' => '/nodes/{id}/metadata'
        ],
        'workflows' => [
          'list' => '/workflows',
          'get' => '/workflows/{id}',
          'create' => '/workflows',
          'update' => '/workflows/{id}',
          'delete' => '/workflows/{id}',
          'validate' => '/workflows/validate',
          'export' => '/workflows/{id}/export',
          'import' => '/workflows/import'
        ],
        'executions' => [
          'execute' => '/workflows/{id}/execute',
          'status' => '/executions/{id}',
          'cancel' => '/executions/{id}/cancel',
          'logs' => '/executions/{id}/logs',
          'history' => '/executions'
        ],
        'templates' => [
          'list' => '/templates',
          'get' => '/templates/{id}',
          'create' => '/templates',
          'update' => '/templates/{id}',
          'delete' => '/templates/{id}'
        ],
        'users' => [
          'profile' => '/users/profile',
          'preferences' => '/users/preferences'
        ],
        'system' => [
          'health' => '/system/health',
          'config' => '/system/config',
          'version' => '/system/version'
        ]
      ],
      'timeout' => 30000,
      'retry' => [
        'enabled' => true,
        'maxAttempts' => 3,
        'delay' => 1000,
        'backoff' => 'exponential'
      ],
      'headers' => [
        'Content-Type' => 'application/json',
        'Accept' => 'application/json'
      ]
    ];

    // Merge with overrides
    return array_merge_recursive($default_config, $overrides);
  }

  /**
   * Generate endpoint configuration for Drupal settings.
   *
   * @param \Drupal\Core\Url $base_url
   *   The base URL object.
   * @param array $overrides
   *   Optional overrides for the configuration.
   *
   * @return array
   *   The endpoint configuration array for drupalSettings.
   */
  public function generateDrupalSettings(Url $base_url, array $overrides = []): array {
    $url_options = [
      'absolute' => TRUE,
      'language' => \Drupal::languageManager()->getCurrentLanguage(),
    ];
    
    $base_url_string = $base_url->toString();
    
    return $this->generateEndpointConfig($base_url_string, $overrides);
  }

}
