<?php

declare(strict_types=1);

namespace Drupal\flowdrop_pipeline\Service;

use Drupal\flowdrop_pipeline\Entity\FlowDropPipelineInterface;
use Drupal\flowdrop_job\FlowDropJobInterface;
use Drupal\flowdrop_node_type\FlowDropNodeTypeInterface;
use Drupal\flowdrop_workflow\Exception\WorkflowExecutionException;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\EventDispatcher\GenericEvent;

/**
 * Service for generating jobs from pipeline workflows.
 */
class JobGenerationService {

  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected LoggerChannelFactoryInterface $loggerFactory,
    protected EventDispatcherInterface $eventDispatcher,
  ) {}

  /**
   * Generate jobs for a pipeline based on its workflow.
   *
   * @param \Drupal\flowdrop_pipeline\Entity\FlowDropPipelineInterface $pipeline
   *   The pipeline to generate jobs for.
   *
   * @return array
   *   Array of created job entities.
   *
   * @throws \Exception
   *   If job generation fails.
   */
  public function generateJobs(FlowDropPipelineInterface $pipeline): array {
    $logger = $this->loggerFactory->get('flowdrop_pipeline');

    try {
      $logger->info('Starting job generation for pipeline @id', [
        '@id' => $pipeline->id(),
      ]);

      // Get the workflow.
      $workflow = $pipeline->getWorkflow();
      if (!$workflow) {
        throw new \RuntimeException('Workflow not found for pipeline ' . $pipeline->id());
      }

      // Get nodes and edges from workflow.
      $nodes = $workflow->getNodes();
      $edges = $workflow->getEdges();

      if (empty($nodes)) {
        throw new \RuntimeException('No nodes found in workflow ' . $workflow->id());
      }

      // Build dependency graph.
      $dependency_graph = $this->buildDependencyGraph($nodes, $edges);

      // Validate dependency graph for cycles.
      $this->validateDependencyGraph($dependency_graph);

      // Calculate execution order and priorities.
      $execution_order = $this->calculateExecutionOrder($dependency_graph);

      // Create jobs first (without dependencies).
      $created_jobs = [];
      $job_storage = $this->entityTypeManager->getStorage('flowdrop_job');
      // Map node IDs to job entities.
      $node_to_job_map = [];

      foreach ($nodes as $node) {
        $node_id = $node['id'];
        $node_type_id = $node['data']['metadata']['id'];

        // Calculate priority based on execution order.
        $priority = $this->calculateJobPriority($node_id, $execution_order, $dependency_graph);

        // Prepare job data (dependencies will be added as
        // entity references later).
        $job_data = [
          'label' => $node['data']['label'] ?? 'Job ' . $node_id,
          'bundle' => 'default',
          'metadata' => json_encode([
            'node_id' => $node_id,
            'node_type_id' => $node_type_id,
          ]),
          'node_type_id' => $node_type_id,
          'status' => 'pending',
          'priority' => $priority,
          'input_data' => json_encode($node['data'] ?? []),
          'output_data' => json_encode([]),
          'max_retries' => $this->getMaxRetriesForNode($node),
        ];

        // Create the job.
        $job = $job_storage->create($job_data);
        $job->save();

        // Store mapping for dependency resolution.
        $node_to_job_map[$node_id] = $job;

        // Add job to pipeline's job_id field.
        assert($job instanceof FlowDropJobInterface);
        $pipeline->addJob($job);

        $created_jobs[] = $job;

        // Dispatch job created event.
        $this->eventDispatcher->dispatch(new GenericEvent($job, [
          'pipeline_id' => $pipeline->id(),
          'node_id' => $node_id,
          'dependencies' => $dependency_graph[$node_id] ?? [],
        ]), 'flowdrop.job.created');

        $logger->info('Created job @job_id for node @node_id in pipeline @pipeline_id', [
          '@job_id' => $job->id(),
          '@node_id' => $node_id,
          '@pipeline_id' => $pipeline->id(),
        ]);
      }

      // Now establish entity reference dependencies between jobs.
      foreach ($dependency_graph as $node_id => $dependency_node_ids) {
        if (empty($dependency_node_ids)) {
          // No dependencies for this node.
          continue;
        }

        $job = $node_to_job_map[$node_id];
        assert($job instanceof FlowDropJobInterface);

        // Add each dependency as an entity reference.
        foreach ($dependency_node_ids as $dependency_node_id) {
          if (isset($node_to_job_map[$dependency_node_id])) {
            $dependency_job = $node_to_job_map[$dependency_node_id];
            assert($dependency_job instanceof FlowDropJobInterface);
            $job->addDependentJob($dependency_job);
          }
        }

        // Save the job with its new dependencies.
        $job->save();
      }

      // Save pipeline with all added jobs.
      $pipeline->save();

      // Dispatch pipeline jobs generated event.
      $this->eventDispatcher->dispatch(new GenericEvent($pipeline, [
        'jobs_created' => count($created_jobs),
        'job_ids' => array_map(fn($job) => $job->id(), $created_jobs),
      ]), 'flowdrop.pipeline.jobs_generated');

      $logger->info('Successfully generated @count jobs for pipeline @id', [
        '@count' => count($created_jobs),
        '@id' => $pipeline->id(),
      ]);

      return $created_jobs;

    }
    catch (\Exception $e) {
      $logger->error('Failed to generate jobs for pipeline @id: @message', [
        '@id' => $pipeline->id(),
        '@message' => $e->getMessage(),
      ]);
      throw $e;
    }
  }

  /**
   * Build dependency graph from workflow nodes and edges.
   *
   * @param array $nodes
   *   The workflow nodes.
   * @param array $edges
   *   The workflow edges.
   *
   * @return array
   *   Dependency graph where keys are node IDs and values are arrays of
   *   dependency node IDs.
   */
  protected function buildDependencyGraph(array $nodes, array $edges): array {
    $dependencies = [];

    // Initialize empty dependencies for all nodes.
    foreach ($nodes as $node) {
      $dependencies[$node['id']] = [];
    }

    // Build dependencies from edges.
    foreach ($edges as $edge) {
      $source_id = $edge['source'];
      $target_id = $edge['target'];

      // Target depends on source.
      if (isset($dependencies[$target_id])) {
        $dependencies[$target_id][] = $source_id;
      }
    }

    return $dependencies;
  }

  /**
   * Validate dependency graph for cycles.
   *
   * @param array $dependency_graph
   *   The dependency graph to validate.
   *
   * @throws \RuntimeException
   *   If a cycle is detected.
   */
  protected function validateDependencyGraph(array $dependency_graph): void {
    $visited = [];
    $rec_stack = [];

    foreach (array_keys($dependency_graph) as $node_id) {
      if (!isset($visited[$node_id])) {
        if ($this->hasCycleDfs($dependency_graph, $node_id, $visited, $rec_stack)) {
          throw new \RuntimeException('Circular dependency detected in workflow');
        }
      }
    }
  }

  /**
   * Depth-first search to detect cycles.
   *
   * @param array $dependency_graph
   *   The dependency graph.
   * @param string $node_id
   *   Current node ID.
   * @param array $visited
   *   Visited nodes tracker.
   * @param array $rec_stack
   *   Recursion stack tracker.
   *
   * @return bool
   *   TRUE if cycle is detected.
   */
  protected function hasCycleDfs(array $dependency_graph, string $node_id, array &$visited, array &$rec_stack): bool {
    $visited[$node_id] = TRUE;
    $rec_stack[$node_id] = TRUE;

    // Check all dependencies (reverse direction for cycle detection)
    foreach ($dependency_graph as $target_id => $dependencies) {
      if (in_array($node_id, $dependencies, TRUE)) {
        if (!isset($visited[$target_id])) {
          if ($this->hasCycleDfs($dependency_graph, $target_id, $visited, $rec_stack)) {
            return TRUE;
          }
        }
        elseif (isset($rec_stack[$target_id]) && $rec_stack[$target_id]) {
          return TRUE;
        }
      }
    }

    $rec_stack[$node_id] = FALSE;
    return FALSE;
  }

  /**
   * Calculate execution order using topological sort.
   *
   * @param array $dependency_graph
   *   The dependency graph.
   *
   * @return array
   *   Execution order where keys are node IDs and values are order indices.
   */
  protected function calculateExecutionOrder(array $dependency_graph): array {
    $in_degree = [];
    $execution_order = [];
    $order_index = 0;

    // Calculate in-degree for each node.
    foreach (array_keys($dependency_graph) as $node_id) {
      $in_degree[$node_id] = count($dependency_graph[$node_id]);
    }

    // Queue nodes with no dependencies.
    $queue = [];
    foreach ($in_degree as $node_id => $degree) {
      if ($degree === 0) {
        $queue[] = $node_id;
      }
    }

    // Process nodes in topological order.
    while (!empty($queue)) {
      $current_node = array_shift($queue);
      $execution_order[$current_node] = $order_index++;

      // Reduce in-degree for nodes that depend on current node.
      foreach ($dependency_graph as $node_id => $dependencies) {
        if (in_array($current_node, $dependencies, TRUE)) {
          $in_degree[$node_id]--;
          if ($in_degree[$node_id] === 0) {
            $queue[] = $node_id;
          }
        }
      }
    }

    return $execution_order;
  }

  /**
   * Calculate job priority based on execution order and dependencies.
   *
   * @param string $node_id
   *   The node ID.
   * @param array $execution_order
   *   The execution order.
   * @param array $dependency_graph
   *   The dependency graph.
   *
   * @return int
   *   The calculated priority (lower number = higher priority).
   */
  protected function calculateJobPriority(string $node_id, array $execution_order, array $dependency_graph): int {
    $base_priority = ($execution_order[$node_id] ?? 1000) * 10;

    // Adjust priority based on number of dependencies.
    $dependency_count = count($dependency_graph[$node_id] ?? []);
    $dependency_adjustment = $dependency_count * 5;

    return $base_priority + $dependency_adjustment;
  }

  /**
   * Get the executor plugin ID for a node.
   *
   * @param array $node
   *   The workflow node.
   *
   * @return string
   *   The executor plugin ID.
   */
  protected function getExecutorPluginIdForNode(array $node): string {
    $node_type_id = $node['data']['metadata']['id'] ?? 'unknown';

    // Try to load the node type entity to get the executor plugin.
    $node_type = $this->entityTypeManager->getStorage('flowdrop_node_type')->load($node_type_id);

    if ($node_type instanceof FlowDropNodeTypeInterface && $node_type->getExecutorPlugin()) {
      return $node_type->getExecutorPlugin();
    }

    // Fall back to a default or throw exception.
    throw new WorkflowExecutionException(
      "No executor plugin ID found for node type: $node_type_id"
    );
  }

  /**
   * Get max retries for a node based on its configuration.
   *
   * @param array $node
   *   The workflow node.
   *
   * @return int
   *   The maximum number of retries.
   */
  protected function getMaxRetriesForNode(array $node): int {
    // Check if the node has a specific retry configuration.
    if (isset($node['data']['config']['max_retries'])) {
      return (int) $node['data']['config']['max_retries'];
    }

    // Default retry count.
    return 3;
  }

  /**
   * Get ready jobs for a pipeline (jobs whose dependencies are completed).
   *
   * @param \Drupal\flowdrop_pipeline\Entity\FlowDropPipelineInterface $pipeline
   *   The pipeline.
   *
   * @return array
   *   Array of ready job entities.
   */
  public function getReadyJobs(FlowDropPipelineInterface $pipeline): array {
    $all_jobs = $pipeline->getJobs();
    $pending_jobs = $pipeline->getJobsByStatus('pending');
    $completed_jobs = $pipeline->getJobsByStatus('completed');

    // Get list of completed node IDs.
    $completed_node_ids = [];
    foreach ($completed_jobs as $job) {
      $completed_node_ids[] = $job->getNodeId();
    }

    $ready_jobs = [];
    foreach ($pending_jobs as $job) {
      if ($this->areJobDependenciesMet($job, $completed_node_ids)) {
        $ready_jobs[] = $job;
      }
    }

    // Sort by priority (lower number = higher priority)
    usort($ready_jobs, function ($a, $b) {
      return $a->getPriority() <=> $b->getPriority();
    });

    return $ready_jobs;
  }

  /**
   * Check if a job's dependencies are met.
   *
   * @param \Drupal\flowdrop_job\FlowDropJobInterface $job
   *   The job to check.
   * @param array $completed_node_ids
   *   Array of completed node IDs.
   *
   * @return bool
   *   TRUE if all dependencies are met.
   */
  protected function areJobDependenciesMet(FlowDropJobInterface $job, array $completed_node_ids): bool {
    $dependent_jobs = $job->getDependentJobs();

    if (empty($dependent_jobs)) {
      // No dependencies.
      return TRUE;
    }

    // Check if all dependency jobs' node IDs are in the completed list.
    foreach ($dependent_jobs as $dependent_job) {
      $dependency_node_id = $dependent_job->getNodeId();
      if (!in_array($dependency_node_id, $completed_node_ids, TRUE)) {
        return FALSE;
      }
    }

    return TRUE;
  }

  /**
   * Clear all jobs for a pipeline.
   *
   * @param \Drupal\flowdrop_pipeline\Entity\FlowDropPipelineInterface $pipeline
   *   The pipeline to clear jobs for.
   *
   * @return int
   *   Number of jobs deleted.
   */
  public function clearJobs(FlowDropPipelineInterface $pipeline): int {
    $jobs = $pipeline->getJobs();
    $count = count($jobs);

    if ($count > 0) {
      $job_storage = $this->entityTypeManager->getStorage('flowdrop_job');
      $job_storage->delete($jobs);

      // Clear jobs from pipeline's job_id field.
      $pipeline->clearJobs();
      $pipeline->save();

      $this->loggerFactory->get('flowdrop_pipeline')->info('Cleared @count jobs for pipeline @id', [
        '@count' => $count,
        '@id' => $pipeline->id(),
      ]);
    }

    return $count;
  }

}
