<?php

declare(strict_types=1);

namespace Drupal\flowdrop_runtime\Service\Orchestrator;

use Psr\Log\LoggerInterface;
use Drupal\flowdrop\DTO\Input;
use Drupal\flowdrop\DTO\Config;
use Drupal\flowdrop_runtime\DTO\Runtime\NodeExecutionContext;
use Drupal\flowdrop_workflow\FlowDropWorkflowInterface;
use Drupal\flowdrop_pipeline\Entity\FlowDropPipelineInterface;
use Drupal\flowdrop_job\FlowDropJobInterface;
use Drupal\flowdrop_runtime\Service\Runtime\NodeRuntimeService;
use Drupal\flowdrop_runtime\Service\Runtime\ExecutionContext;
use Drupal\flowdrop_runtime\Service\RealTime\RealTimeManager;
use Drupal\flowdrop_runtime\Service\Compiler\WorkflowCompiler;
use Drupal\flowdrop_runtime\DTO\Orchestrator\OrchestrationRequest;
use Drupal\flowdrop_runtime\DTO\Orchestrator\OrchestrationResponse;
use Drupal\flowdrop_runtime\Exception\OrchestrationException;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\EventDispatcher\GenericEvent;

/**
 * Synchronous workflow orchestrator with real-time updates.
 */
class SynchronousOrchestrator implements OrchestratorInterface {

  /**
   * Logger channel for this service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  private readonly LoggerInterface $logger;

  public function __construct(
    private readonly NodeRuntimeService $nodeRuntime,
    private readonly ExecutionContext $executionContext,
    private readonly RealTimeManager $realTimeManager,
    private readonly WorkflowCompiler $workflowCompiler,
    LoggerChannelFactoryInterface $loggerFactory,
    private readonly EventDispatcherInterface $eventDispatcher,
  ) {
    $this->logger = $loggerFactory->get('flowdrop_runtime');
  }

  /**
   * {@inheritdoc}
   */
  public function getType(): string {
    return 'synchronous';
  }

  /**
   * Execute a workflow using WorkflowInterface.
   *
   * @param \Drupal\flowdrop_workflow\FlowDropWorkflowInterface $workflow
   *   The workflow to execute.
   * @param array $initialContext
   *   Initial context data for the workflow execution.
   *
   * @return \Drupal\flowdrop_runtime\DTO\Orchestrator\OrchestrationResponse
   *   The orchestration response.
   *
   * @throws \Drupal\flowdrop_runtime\Exception\OrchestrationException
   *   When orchestration fails.
   */
  public function executeWorkflow(FlowDropWorkflowInterface $workflow, array $initialContext = []): OrchestrationResponse {
    $this->logger->info('Starting workflow execution for @workflow_id', [
      '@workflow_id' => $workflow->id(),
      '@workflow_label' => $workflow->getLabel(),
    ]);

    // Convert workflow interface to array format for orchestration.
    $workflowData = [
      'id' => $workflow->id(),
      'label' => $workflow->getLabel(),
      'description' => $workflow->getDescription(),
      'nodes' => $workflow->getNodes(),
      'edges' => $workflow->getEdges(),
      'metadata' => $workflow->getMetadata(),
    ];

    // Create orchestration request.
    $request = new OrchestrationRequest(
      workflowId: $workflow->id(),
      pipelineId: 'pipeline_' . $workflow->id() . '_' . time(),
      workflow: $workflowData,
      initialData: $initialContext,
      options: [],
    );

    // Execute the workflow using the existing orchestrate method.
    return $this->orchestrate($request);
  }

  /**
   * {@inheritdoc}
   */
  public function orchestrate(OrchestrationRequest $request): OrchestrationResponse {
    $workflow = $request->getWorkflow();
    $executionId = $this->generateExecutionId();

    $this->logger->info('Starting synchronous orchestration for workflow @workflow_id', [
      '@workflow_id' => $request->getWorkflowId(),
      '@execution_id' => $executionId,
    ]);

    try {
      // Compile workflow using WorkflowCompiler.
      $compiledWorkflow = $this->workflowCompiler->compile($workflow);
      $executionPlan = $compiledWorkflow->getExecutionPlan();
      $nodeMappings = $compiledWorkflow->getNodeMappings();

      $this->logger->info('Workflow compiled successfully with @nodes nodes', [
        '@nodes' => count($nodeMappings),
        '@execution_id' => $executionId,
      ]);

      // Start real-time monitoring.
      $this->realTimeManager->startMonitoring($executionId, $workflow);

      // Update execution status to running.
      $this->realTimeManager->updateExecutionStatus($executionId, 'running', [
        'workflow_id' => $request->getWorkflowId(),
        'pipeline_id' => $request->getPipelineId(),
        'node_count' => count($nodeMappings),
        'start_time' => time(),
      ]);

      // Create execution context.
      $context = $this->executionContext->createContext(
        $request->getWorkflowId(),
        $request->getPipelineId(),
        $request->getInitialData()
      );

      // Get execution order from compiled workflow.
      $executionOrder = $executionPlan->getExecutionOrder();

      $results = [];
      $startTime = microtime(TRUE);

      // Execute nodes in compiled execution order with real-time updates.
      foreach ($executionOrder as $nodeId) {
        $nodeMapping = $nodeMappings[$nodeId];
        $node = $this->findNodeById($workflow['nodes'], $nodeId);

        if (!$node) {
          throw new OrchestrationException("Node $nodeId not found in workflow");
        }

        $nodeType = $nodeMapping->getProcessorId();

        // Log node start.
        $this->logger->info('Starting execution of node @node_id (@node_type)', [
          '@node_id' => $nodeId,
          '@node_type' => $nodeType,
          '@execution_id' => $executionId,
        ]);

        // Emit pre-execute event.
        $this->eventDispatcher->dispatch(
          new GenericEvent($node, [
            'execution_id' => $executionId,
            'workflow_id' => $request->getWorkflowId(),
            'context' => $context,
          ]),
          'flowdrop.node.pre_execute'
        );

        try {
          $nodeInputs = $this->prepareNodeInputsFromPlan($context, $nodeId, $executionPlan);

          // Create proper Input and Config objects.
          $inputs = new Input();
          $inputs->fromArray($nodeInputs);

          $config = new Config();
          $config->fromArray($nodeMapping->getConfig());

          $result = $this->nodeRuntime->executeNode(
            $executionId,
            $nodeId,
            $nodeType,
            $inputs,
            $config,
            $context
          );

          $results[$nodeId] = $result;

          // Update context with result.
          $this->executionContext->updateContext($context, $nodeId, $result->getOutput());

          // Log node success.
          $this->logger->info('Node @node_id (@node_type) completed successfully', [
            '@node_id' => $nodeId,
            '@node_type' => $nodeType,
            '@execution_id' => $executionId,
            '@execution_time' => $result->getExecutionTime(),
          ]);

          // Emit post-execute event.
          $this->eventDispatcher->dispatch(
            new GenericEvent($node, [
              'execution_id' => $executionId,
              'workflow_id' => $request->getWorkflowId(),
              'context' => $context,
              'result' => $result,
            ]),
            'flowdrop.node.post_execute'
          );

        }
        catch (\Exception $e) {
          // Log node failure.
          $this->logger->error('Node @node_id (@node_type) execution failed: @error', [
            '@node_id' => $nodeId,
            '@node_type' => $nodeType,
            '@execution_id' => $executionId,
            '@error' => $e->getMessage(),
          ]);

          // Emit post-execute event with error.
          $this->eventDispatcher->dispatch(
            new GenericEvent($node, [
              'execution_id' => $executionId,
              'workflow_id' => $request->getWorkflowId(),
              'context' => $context,
              'error' => $e->getMessage(),
            ]),
            'flowdrop.node.post_execute'
          );

          throw $e;
        }
      }

      $totalTime = microtime(TRUE) - $startTime;

      // Update execution status to completed.
      $this->realTimeManager->updateExecutionStatus($executionId, 'completed', [
        'total_execution_time' => $totalTime,
        'nodes_executed' => count($results),
        'end_time' => time(),
      ]);

      $this->logger->info('Synchronous orchestration completed in @time seconds', [
        '@time' => round($totalTime, 3),
        '@execution_id' => $executionId,
      ]);

      // Emit workflow completed event.
      $this->eventDispatcher->dispatch(
        new GenericEvent($workflow, [
          'execution_id' => $executionId,
          'workflow_id' => $request->getWorkflowId(),
          'results' => $results,
          'execution_time' => $totalTime,
          'context' => $context,
        ]),
        'flowdrop.workflow.completed'
      );

      return new OrchestrationResponse(
        executionId: $executionId,
        status: 'completed',
        results: $results,
        executionTime: $totalTime,
        metadata: [
          'orchestrator_type' => $this->getType(),
          'nodes_executed' => count($results),
          'compiled_workflow_id' => $compiledWorkflow->getWorkflowId(),
        ]
      );

    }
    catch (\Exception $e) {
      // Update execution status to failed.
      $this->realTimeManager->updateExecutionStatus($executionId, 'failed', [
        'error' => $e->getMessage(),
        'end_time' => time(),
      ]);

      $this->logger->error('Synchronous orchestration failed: @error', [
        '@error' => $e->getMessage(),
        '@execution_id' => $executionId,
      ]);

      throw new OrchestrationException(
        "Synchronous orchestration failed: " . $e->getMessage(),
        0,
        $e
      );
    }
  }

  /**
   * {@inheritdoc}
   */
  public function supportsWorkflow(array $workflow): bool {
    // Synchronous orchestrator supports all workflows.
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function getCapabilities(): array {
    return [
      'synchronous_execution' => TRUE,
      'parallel_execution' => FALSE,
      'async_execution' => FALSE,
      'retry_support' => TRUE,
      'error_recovery' => TRUE,
      'workflow_compilation' => TRUE,
    ];
  }

  /**
   * Generate a unique execution ID.
   */
  private function generateExecutionId(): string {
    return 'exec_' . time() . '_' . uniqid();
  }

  /**
   * Find a node by ID in the workflow nodes array.
   *
   * @param array $nodes
   *   The workflow nodes array.
   * @param string $nodeId
   *   The node ID to find.
   *
   * @return array|null
   *   The node array or NULL if not found.
   */
  private function findNodeById(array $nodes, string $nodeId): ?array {
    foreach ($nodes as $node) {
      if ($node['id'] === $nodeId) {
        return $node;
      }
    }
    return NULL;
  }

  /**
   * Prepare inputs for a node based on the execution plan.
   *
   * @param \Drupal\flowdrop_runtime\DTO\Runtime\NodeExecutionContext $context
   *   The execution context.
   * @param string $nodeId
   *   The node ID.
   * @param \Drupal\flowdrop_runtime\DTO\Compiler\ExecutionPlan $executionPlan
   *   The execution plan.
   *
   * @return array
   *   The prepared node inputs.
   */
  private function prepareNodeInputsFromPlan($context, string $nodeId, $executionPlan): array {
    $inputMappings = $executionPlan->getInputMappings();
    $nodeInputMappings = $inputMappings[$nodeId] ?? [];

    $inputData = $context->getInitialData();

    // Add outputs from dependent nodes based on execution plan.
    foreach ($nodeInputMappings as $dependencyId => $mapping) {
      $dependencyOutput = $context->getNodeOutput($dependencyId);
      if ($dependencyOutput) {
        $inputData = array_merge($inputData, $dependencyOutput->toArray());
      }
    }

    return $inputData;
  }

  /**
   * Execute a pipeline by running all ready jobs synchronously.
   *
   * @param \Drupal\flowdrop_pipeline\Entity\FlowDropPipelineInterface $pipeline
   *   The pipeline to execute.
   *
   * @return \Drupal\flowdrop_runtime\DTO\Orchestrator\OrchestrationResponse
   *   The orchestration response.
   *
   * @throws \Drupal\flowdrop_runtime\Exception\OrchestrationException
   *   When execution fails.
   */
  public function executePipeline(FlowDropPipelineInterface $pipeline): OrchestrationResponse {
    $execution_id = $this->generateExecutionId();
    $startTime = microtime(TRUE);

    $this->logger->info('Starting synchronous pipeline execution for pipeline @pipeline_id', [
      '@pipeline_id' => $pipeline->id(),
      '@execution_id' => $execution_id,
    ]);

    try {
      // Mark pipeline as running.
      $pipeline->markAsStarted();
      $pipeline->save();

      $results = [];
      $executed_jobs = [];
      // Prevent infinite loops.
      $max_iterations = 100;
      $iteration = 0;

      // Keep executing ready jobs until none are left or max iterations
      // reached.
      while ($iteration < $max_iterations) {
        $ready_jobs = $pipeline->getReadyJobs();

        if (empty($ready_jobs)) {
          // No more ready jobs, check if we're done.
          break;
        }

        $this->logger->info('Found @count ready jobs in iteration @iteration', [
          '@count' => count($ready_jobs),
          '@iteration' => $iteration,
          '@execution_id' => $execution_id,
        ]);

        // Execute all ready jobs in this iteration.
        foreach ($ready_jobs as $job) {
          $job_result = $this->executeJob($job, $execution_id);
          $results[$job->getNodeId()] = $job_result;
          $executed_jobs[] = $job->id();
        }

        $iteration++;
      }

      // Check final pipeline status.
      if ($pipeline->allJobsCompleted()) {
        $pipeline->markAsCompleted($results);
        $status = 'completed';
      }
      elseif ($pipeline->hasFailedJobs()) {
        $pipeline->markAsFailed('Some jobs failed during execution');
        $status = 'failed';
      }
      else {
        $pipeline->markAsFailed('Pipeline execution incomplete - possible circular dependencies');
        $status = 'failed';
      }

      $pipeline->save();

      $executionTime = microtime(TRUE) - $startTime;

      $this->logger->info('Synchronous pipeline execution completed for pipeline @pipeline_id in @time seconds', [
        '@pipeline_id' => $pipeline->id(),
        '@time' => round($executionTime, 3),
        '@status' => $status,
        '@executed_jobs' => count($executed_jobs),
      ]);

      return new OrchestrationResponse(
        executionId: $execution_id,
        status: $status,
        results: $results,
        executionTime: $executionTime,
        metadata: [
          'pipeline_id' => $pipeline->id(),
          'executed_jobs' => $executed_jobs,
          'iterations' => $iteration,
          'total_jobs' => count($pipeline->getJobs()),
        ]
      );

    }
    catch (\Exception $e) {
      $pipeline->markAsFailed('Pipeline execution failed: ' . $e->getMessage());
      $pipeline->save();

      $this->logger->error('Synchronous pipeline execution failed for pipeline @pipeline_id: @message', [
        '@pipeline_id' => $pipeline->id(),
        '@message' => $e->getMessage(),
        '@execution_id' => $execution_id,
      ]);

      throw new OrchestrationException(
        'Synchronous pipeline execution failed: ' . $e->getMessage(),
        0,
        $e
      );
    }
  }

  /**
   * Execute a single job.
   *
   * @param \Drupal\flowdrop_job\FlowDropJobInterface $job
   *   The job to execute.
   * @param string $execution_id
   *   The execution ID for logging.
   *
   * @return array
   *   The job execution result.
   */
  protected function executeJob(FlowDropJobInterface $job, string $execution_id): array {
    $this->logger->info('Executing job @job_id (node: @node_id)', [
      '@job_id' => $job->id(),
      '@node_id' => $job->getNodeId(),
      '@execution_id' => $execution_id,
    ]);

    try {
      // Mark job as started.
      $job->markAsStarted();
      $job->save();

      // Prepare execution context.
      $input_data = $job->getInputData();
      $input_dto = new Input($input_data);
      $config_dto = new Config([]);

      $context = new NodeExecutionContext(
        workflowId: 'pipeline_' . $job->id(),
        pipelineId: 'pipeline_' . $job->id(),
        initialData: $input_data
      );

      // Get node type for execution.
      $node_type_id = $job->getMetadataValue('node_type_id', 'default');

      // Execute the job using node runtime.
      $result = $this->nodeRuntime->executeNode(
        $execution_id,
        $job->getNodeId(),
        $node_type_id,
        $input_dto,
        $config_dto,
        $context
      );

      // Mark job as completed.
      $output_data = $result->getOutput()->toArray();
      $job->markAsCompleted($output_data);
      $job->save();

      $this->logger->info('Job @job_id completed successfully', [
        '@job_id' => $job->id(),
        '@execution_id' => $execution_id,
      ]);

      return $output_data;

    }
    catch (\Exception $e) {
      // Mark job as failed.
      $job->markAsFailed($e->getMessage());
      $job->save();

      $this->logger->error('Job @job_id failed: @message', [
        '@job_id' => $job->id(),
        '@message' => $e->getMessage(),
        '@execution_id' => $execution_id,
      ]);

      // Re-throw for pipeline-level handling.
      throw $e;
    }
  }

}
