/**
 * Example: Using FlowDrop API Client
 * This demonstrates how to integrate with FlowDrop API from backend applications
 */

import { ApiClient } from '../clients/ApiClient.js';
import type { Workflow } from '../types/index.js';
import type { CreateWorkflowRequest } from '../clients/ApiClient.js';

// Initialize the API client
const apiClient = new ApiClient({
	baseUrl: 'https://api.flowdrop.dev/v1',
	apiKey: 'your-api-key-here',
	timeout: 30000
});

// Example 1: Health check
async function checkApiHealth() {
	try {
		const health = await apiClient.healthCheck();
		console.log('API Status:', health.status);
		console.log('API Version:', health.version);
		return health;
	} catch (error) {
		console.error('API Health Check Failed:', error);
		throw error;
	}
}

// Example 2: Get available node types
async function getNodeTypes() {
	try {
		// Get all node types
		const allNodes = await apiClient.getNodeTypes();
		console.log(`Found ${allNodes.data?.length || 0} node types`);

		// Get node types by category
		const modelNodes = await apiClient.getNodeTypesByCategory('models');
		console.log(`Found ${modelNodes.data?.length || 0} model nodes`);

		// Search for specific nodes
		const searchResults = await apiClient.getNodeTypes({ search: 'openai' });
		console.log(`Found ${searchResults.data?.length || 0} OpenAI-related nodes`);

		return allNodes;
	} catch (error) {
		console.error('Failed to get node types:', error);
		throw error;
	}
}

// Example 3: Create and manage workflows
async function manageWorkflows() {
	try {
		// Create a new workflow
		const createRequest: CreateWorkflowRequest = {
			name: 'Chat Workflow',
			description: 'A simple chat workflow created via API',
			tags: ['chat', 'demo']
		};

		const newWorkflow = await apiClient.createWorkflow(createRequest);
		console.log('Created workflow:', newWorkflow.data?.id);

		// Get all workflows
		const workflows = await apiClient.getWorkflows({
			limit: 10,
			sort: 'created_at',
			order: 'desc'
		});
		console.log(`Found ${workflows.data?.length || 0} workflows`);

		// Get workflows by tag
		const chatWorkflows = await apiClient.getWorkflowsByTag('chat');
		console.log(`Found ${chatWorkflows.data?.length || 0} chat workflows`);

		return newWorkflow;
	} catch (error) {
		console.error('Failed to manage workflows:', error);
		throw error;
	}
}

// Example 4: Execute a workflow
async function executeWorkflow(workflowId: string) {
	try {
		// Start execution
		const execution = await apiClient.executeWorkflow(workflowId, {
			inputs: {
				message: 'Hello, how are you?',
				temperature: 0.7
			},
			options: {
				timeout: 60,
				maxSteps: 100
			}
		});

		console.log('Execution started:', execution.data?.executionId);

		// Wait for completion
		const result = await apiClient.waitForExecution(execution.data!.executionId);

		if (result.data?.status === 'completed') {
			console.log('Execution completed successfully');
			console.log('Result:', result.data?.result);
		} else {
			console.log('Execution failed:', result.data?.error);
		}

		return result;
	} catch (error) {
		console.error('Failed to execute workflow:', error);
		throw error;
	}
}

// Example 5: Import/Export workflows
async function importExportWorkflow(workflowId: string) {
	try {
		// Export workflow
		const exportedWorkflow = await apiClient.exportWorkflow(workflowId, 'json');
		console.log('Exported workflow:', exportedWorkflow.name);

		// Modify the exported workflow
		const modifiedWorkflow: Workflow = {
			...exportedWorkflow,
			id: '', // Will be generated by server
			name: `${exportedWorkflow.name} (Imported)`,
			metadata: {
				...exportedWorkflow.metadata!,
				createdAt: new Date().toISOString(),
				updatedAt: new Date().toISOString()
			}
		};

		// Import the modified workflow
		const importedWorkflow = await apiClient.importWorkflow(modifiedWorkflow);
		console.log('Imported workflow:', importedWorkflow.data?.id);

		return importedWorkflow;
	} catch (error) {
		console.error('Failed to import/export workflow:', error);
		throw error;
	}
}

// Example 6: Validate workflows
async function validateWorkflow(workflow: Workflow) {
	try {
		const validation = await apiClient.validateWorkflow(workflow);

		if (validation.data?.valid) {
			console.log('Workflow is valid');
		} else {
			console.log('Workflow validation failed:');
			console.log('Errors:', validation.data?.errors);
			console.log('Warnings:', validation.data?.warnings);
			console.log('Suggestions:', validation.data?.suggestions);
		}

		return validation;
	} catch (error) {
		console.error('Failed to validate workflow:', error);
		throw error;
	}
}

// Example 7: Error handling
async function handleApiErrors() {
	try {
		// This will fail - non-existent workflow
		await apiClient.getWorkflow('non-existent-id');
	} catch (error) {
		if (error instanceof Error) {
			console.log('Error type:', error.constructor.name);
			console.log('Error message:', error.message);

			// Check if it's an API error
			if ('status' in error) {
				const apiError = error as Error & { status?: number; code?: string; details?: unknown };
				console.log('HTTP status:', apiError.status);
				console.log('Error code:', apiError.code);
				console.log('Error details:', apiError.details);
			}
		}
	}
}

// Example 8: Batch operations
async function batchOperations() {
	try {
		// Get all workflows
		const workflows = await apiClient.getWorkflows({ limit: 100 });

		// Process each workflow
		const results = await Promise.allSettled(
			workflows.data?.map(async (workflow) => {
				try {
					// Validate each workflow
					const validation = await apiClient.validateWorkflow(workflow);
					return {
						id: workflow.id,
						name: workflow.name,
						valid: validation.data?.valid,
						errors: validation.data?.errors?.length || 0
					};
				} catch (error) {
					return {
						id: workflow.id,
						name: workflow.name,
						valid: false,
						error: error instanceof Error ? error.message : 'Unknown error'
					};
				}
			}) || []
		);

		console.log('Batch validation results:', results);
		return results;
	} catch (error) {
		console.error('Failed to perform batch operations:', error);
		throw error;
	}
}

// Example 9: Real-time monitoring
async function monitorExecutions(workflowId: string) {
	try {
		// Start execution
		const execution = await apiClient.executeWorkflow(workflowId, {
			inputs: { message: 'Test message' }
		});

		const executionId = execution.data!.executionId;
		console.log('Monitoring execution:', executionId);

		// Poll for status updates
		const interval = setInterval(async () => {
			try {
				const status = await apiClient.getExecutionStatus(executionId);
				console.log(`Status: ${status.data?.status}`);

				if (
					status.data?.status === 'completed' ||
					status.data?.status === 'failed' ||
					status.data?.status === 'cancelled'
				) {
					clearInterval(interval);
					console.log('Execution finished:', status.data?.status);
				}
			} catch (error) {
				console.error('Failed to get execution status:', error);
				clearInterval(interval);
			}
		}, 1000);

		// Timeout after 5 minutes
		setTimeout(
			() => {
				clearInterval(interval);
				console.log('Monitoring timeout');
			},
			5 * 60 * 1000
		);
	} catch (error) {
		console.error('Failed to monitor execution:', error);
		throw error;
	}
}

// Export all examples
export {
	checkApiHealth,
	getNodeTypes,
	manageWorkflows,
	executeWorkflow,
	importExportWorkflow,
	validateWorkflow,
	handleApiErrors,
	batchOperations,
	monitorExecutions
};
