<?php

declare(strict_types=1);

namespace Drupal\flowdrop_ai_provider\Plugin\FlowDropNodeProcessor;

use Drupal\ai\OperationType\Chat\ChatInput;
use Drupal\ai\OperationType\Chat\ChatMessage;
use Drupal\flowdrop\DTO\ParameterBagInterface;
use Drupal\flowdrop\DTO\ValidationResult;
use Drupal\flowdrop\Plugin\FlowDropNodeProcessor\AbstractFlowDropNodeProcessor;
use Drupal\flowdrop_ai_provider\Service\AiModelService;
use Drupal\flowdrop\Attribute\FlowDropNodeProcessor;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * AI-specific executor for Chat Model nodes.
 */
#[FlowDropNodeProcessor(
  id: "chat_model",
  label: new TranslatableMarkup("Chat Model (Drupal AI)"),
  description: "AI chat model integration for conversational AI",
  version: "1.0.0",
)]
class ChatModel extends AbstractFlowDropNodeProcessor {

  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    protected AiModelService $aiModelService,
    protected $aiProviderManager,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('flowdrop_ai_provider.model_service'),
      $container->get('ai.provider')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function process(ParameterBagInterface $params): array {
    $message = $params->get('message') ?? $params->get('text', '');

    // Get model_id from config, fallback to default configured model.
    $model_id = $params->get('model', '');
    if (empty($model_id)) {
      // Try to get the default model from AI settings.
      $model_id = $this->aiModelService->getDefaultModelForOperationType('chat');

      // If no default is set, use the first available model.
      if (empty($model_id)) {
        $available_models = $this->aiModelService->getAvailableModels();
        $model_ids = array_keys($available_models);
        $model_id = !empty($model_ids) ? $model_ids[0] : '';

        if (empty($model_id)) {
          throw new \Exception('No AI models are configured. Please configure at least one AI provider.');
        }
      }
    }

    $temperature = $params->get('temperature', 0.7);
    $max_tokens = $params->get('maxTokens', 1000);
    $system_prompt = $params->get('systemPrompt', '');

    if (empty($message)) {
      throw new \Exception('No message provided to chat model');
    }

    // Get model configuration.
    $model_config = $this->aiModelService->getModel($model_id);
    if (!$model_config) {
      throw new \Exception("Model {$model_id} not found");
    }

    // Execute AI chat request.
    $response = $this->executeChatRequest(
      $message,
      $model_id,
      $temperature,
      $max_tokens,
      $model_config,
      $system_prompt
    );
    return [
      'response' => $response['content'],
      'model' => $model_id,
      'provider' => $model_config['provider'],
      'tokens_used' => $response['tokens_used'] ?? 0,
      'temperature' => $temperature,
      'max_tokens' => $max_tokens,
    ];
  }

  /**
   * {@inheritDoc}
   */
  public function validateParams(array $params): ValidationResult {
    // Validate model configuration.
    if (!$this->aiModelService->validateModelConfig($params)) {
      throw new \Exception('Invalid model configuration');
    }
    $model_id = $params['model'];
    // Check if model supports chat.
    if (!$this->aiModelService->supportsChat($model_id)) {
      throw new \Exception("Model {$model_id} does not support chat functionality");
    }
    return parent::validateParams($params);
  }

  /**
   * Execute chat request with AI model.
   *
   * @param string $message
   *   The message to send to the AI model.
   * @param string $model_id
   *   The model ID to use.
   * @param float $temperature
   *   The temperature setting.
   * @param int $max_tokens
   *   The maximum tokens to generate.
   * @param array $model_config
   *   The model configuration.
   * @param string $system_prompt
   *   The system prompt to set AI behavior.
   *
   * @return array
   *   The AI response.
   */
  private function executeChatRequest(string $message, string $model_id, float $temperature, int $max_tokens, array $model_config, string $system_prompt = ''): array {
    // Use the Drupal AI module to make the actual request.
    $provider_id = $model_config['provider'];

    // Get the AI provider instance.
    $provider = $this->aiProviderManager->createInstance($provider_id, [
      'temperature' => $temperature,
      'max_tokens' => $max_tokens,
    ]);

    // Set up the chat request with ChatInput object.
    $chat_messages = [];

    // Add system message if provided.
    if (!empty($system_prompt)) {
      $chat_messages[] = new ChatMessage('system', $system_prompt);
    }

    // Add user message.
    $chat_messages[] = new ChatMessage('user', $message);
    $chat_input = new ChatInput($chat_messages);

    // Call the provider's chat method.
    $response = $provider->chat($chat_input, $model_id, []);

    // Extract the response content.
    $content = '';
    $tokens_used = 0;

    // Handle ChatOutput response object (preferred - Drupal AI module standard).
    if (is_object($response) && method_exists($response, 'getNormalized')) {
      $normalized = $response->getNormalized();

      // getNormalized() returns a ChatMessage object, use getText() method.
      if (is_object($normalized) && method_exists($normalized, 'getText')) {
        $content = $normalized->getText();
      }
      elseif (is_array($normalized) && isset($normalized['text'])) {
        // Fallback for array-based normalized response.
        $content = $normalized['text'];
      }

      // Get token usage from ChatOutput object.
      if (method_exists($response, 'getTotalTokenUsage')) {
        $tokens_used = $response->getTotalTokenUsage() ?? 0;
      }
      elseif (method_exists($response, 'getTokenUsage')) {
        $tokenUsage = $response->getTokenUsage();
        $tokens_used = is_object($tokenUsage) && property_exists($tokenUsage, 'total')
          ? ($tokenUsage->total ?? 0)
          : 0;
      }
    }
    // Handle array response (OpenAI-style).
    elseif (is_array($response) && isset($response['choices'][0]['message']['content'])) {
      $content = $response['choices'][0]['message']['content'];
      $tokens_used = $response['usage']['total_tokens'] ?? 0;
    }
    // Handle string response.
    elseif (is_string($response)) {
      $content = $response;
    }
    // Handle response object with getText method directly.
    elseif (is_object($response) && method_exists($response, 'getText')) {
      $content = $response->getText();
    }

    return [
      'content' => $content,
      'tokens_used' => $tokens_used,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getOutputSchema(): array {
    return [
      'type' => 'object',
      'properties' => [
        'response' => [
          'type' => 'string',
          'description' => 'The AI model response',
        ],
        'model' => [
          'type' => 'string',
          'description' => 'The model used',
        ],
        'provider' => [
          'type' => 'string',
          'description' => 'The AI provider',
        ],
        'tokens_used' => [
          'type' => 'integer',
          'description' => 'Number of tokens used',
        ],
        "temperature" => [
          "type" => "number",
          "description" => "The temperature setting",
        ],
        'max_tokens' => [
          'type' => 'integer',
          'description' => 'Maximum tokens setting',
        ],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getParameterSchema(): array {
    $available_models = $this->aiModelService->getAvailableModels();
    $model_ids = array_keys($available_models);
    $model_options = [];

    foreach ($available_models as $model_id => $model_info) {
      $model_options[] = [
        'value' => $model_id,
        'label' => $model_info['name'] . ' (' . $model_info['provider'] . ')',
      ];
    }

    // Get default model from AI settings, or fallback to first available.
    $default_model = $this->aiModelService->getDefaultModelForOperationType('chat');
    if (empty($default_model) && !empty($model_ids)) {
      $default_model = $model_ids[0];
    }

    return [
      'type' => 'object',
      'properties' => [
        'message' => [
          'type' => 'string',
          'title' => 'Message',
          'description' => 'Message to send to AI model',
          'required' => FALSE,
        ],
        'model' => [
          'type' => 'string',
          'title' => 'Model',
          'description' => 'AI model to use for chat',
          'default' => $default_model,
          'enum' => $model_ids,
          'options' => $model_options,
        ],
        "temperature" => [
          "type" => "number",
          "title" => "Temperature",
          "description" => "Model temperature (0.0 to 2.0)",
          "default" => 0.7,
          "minimum" => 0,
          "maximum" => 2,
          "step" => "0.01",
          "format" => "range",
        ],
        'maxTokens' => [
          'type' => 'integer',
          'title' => 'Max Tokens',
          'description' => 'Maximum tokens to generate',
          'default' => 1000,
          'minimum' => 1,
          'maximum' => 4000,
        ],
        'systemPrompt' => [
          'type' => 'string',
          'title' => 'System Prompt',
          'description' => 'System prompt to set the AI behavior and context',
          'default' => '',
          'format' => 'multiline',
        ],
      ],
    ];
  }

}
