# FlowDrop UI

A Drupal module providing the user interface for FlowDrop - a visual workflow editor built with Svelte.

## Overview

FlowDrop UI is a sub-module of the FlowDrop ecosystem that delivers a modern, interactive workflow editor interface. It wraps the [@d34dman/flowdrop](https://www.npmjs.com/package/@d34dman/flowdrop) library as an IIFE (Immediately Invoked Function Expression) bundle for seamless Drupal integration.

## Requirements

- **Drupal**: 11.x
- **PHP**: 8.1+
- **Node.js**: 18+ (for development/building)

## Installation

### 1. Enable the Module

```bash
drush en flowdrop_ui
```

### 2. Build the Frontend Assets

```bash
cd app/flowdrop
npm install
npm run build
```

### 3. Clear Drupal Cache

```bash
drush cr
```

## Project Structure

```
flowdrop_ui/
├── app/
│   └── flowdrop/               # Svelte frontend application
│       ├── src/                # Source files
│       │   ├── lib/            # Library exports
│       │   └── mocks/          # Mock data for development
│       ├── vite.config.ts      # Default Vite configuration
│       ├── vite.config.drupal.ts    # Drupal-specific build config
│       └── vite.config.production.ts # Production build config
├── build/
│   └── flowdrop/               # Compiled assets
│       ├── flowdrop.css        # Compiled styles
│       ├── flowdrop.es.js      # ES module build
│       ├── flowdrop.iife.js    # IIFE build (used by Drupal)
│       └── *.map               # Source maps
├── js/
│   └── integration-test.js     # Integration testing utilities
├── src/
│   └── Service/
│       └── FlowDropEndpointConfigService.php  # API endpoint configuration
├── flowdrop_ui.info.yml        # Module definition
├── flowdrop_ui.libraries.yml   # Drupal library definitions
└── flowdrop_ui.services.yml    # Service definitions
```

## Development

### Available Scripts

Navigate to `app/flowdrop/` and run:

| Command                          | Description                      |
| -------------------------------- | -------------------------------- |
| `npm run build`                  | Production build                 |
| `npm run build:drupal`           | Build for Drupal integration     |
| `npm run build:production`       | Full production build            |
| `npm run watch:build:drupal`     | Watch mode for Drupal builds     |
| `npm run watch:build:production` | Watch mode for production builds |
| `npm run lint`                   | Run ESLint and Prettier checks   |
| `npm run format`                 | Format code with Prettier        |

### Building for Drupal

The module uses Vite to build the Svelte application into different formats:

```bash
# For local Drupal development
npm run build:drupal

# For production deployment
npm run build:production
```

### Watch Mode

For active development, use watch mode to automatically rebuild on changes:

```bash
npm run watch:build:drupal
```

## Usage

### Attaching the Library

In your Drupal render arrays or templates, attach the FlowDrop editor library:

```php
$build['#attached']['library'][] = 'flowdrop_ui/editor';
```

### Mounting the Editor

The library exposes mounting functions on the global `window.FlowDrop` object:

```javascript
// Mount the full FlowDrop App
window.FlowDrop.mountFlowDropApp(container, options);

// Mount just the WorkflowEditor component
window.FlowDrop.mountWorkflowEditor(container, options);
```

### Drupal Behavior Example

```javascript
Drupal.behaviors.flowdropWorkflowEditor = {
  attach: function (context, settings) {
    const container = context.querySelector(".flowdrop-workflow-editor");
    if (container && window.FlowDrop) {
      window.FlowDrop.mountWorkflowEditor(container, {
        endpointConfig: settings.flowdrop.endpointConfig,
        workflow: settings.flowdrop.workflow,
        nodes: settings.flowdrop.nodes
      });
    }
  }
};
```

## Services

### FlowDropEndpointConfigService

Generates API endpoint configuration for the FlowDrop frontend:

```php
/** @var \Drupal\flowdrop_ui\Service\FlowDropEndpointConfigService $service */
$service = \Drupal::service('flowdrop_ui.endpoint_config');
$config = $service->generateEndpointConfig('/api/flowdrop');
```

## Configuration Options

The editor accepts various configuration options:

```javascript
window.FlowDrop.mountWorkflowEditor(container, {
  workflow: myWorkflow,           // Workflow data object
  nodes: availableNodes,          // Available node definitions
  endpointConfig: {
    baseUrl: "/api/flowdrop",
    endpoints: { /* endpoint definitions */ },
    timeout: 30000,
    retry: {
      enabled: true,
      maxAttempts: 3,
      delay: 1000,
      backoff: "exponential"
    }
  }
});
```

## Related Modules

- **flowdrop**: Core FlowDrop module with node processors and runtime
- **flowdrop_workflow**: Workflow entity management
- **flowdrop_runner**: Workflow execution engine

## License

This project is part of the FlowDrop ecosystem for Drupal.

## Support

For issues and questions, please refer to the main FlowDrop documentation or create an issue in the project repository.

