/**
 * @file
 * FlowDrop Agents Editor initialization for Modeler API.
 *
 * Initializes the FlowDrop UI for editing AI Agents via Modeler API.
 */

(function (once, Drupal, drupalSettings) {
    'use strict';

    /**
     * View settings for the editor (not persisted).
     */
    const viewSettings = {
        nodeLayout: 'default',     // 'default' | 'simple'
        agentExpansion: 'expanded' // 'expanded' | 'grouped' | 'collapsed'
    };

    /**
     * Tracks whether there are unsaved changes.
     */
    let hasUnsavedChanges = false;

    Drupal.behaviors.flowdropAgentsEditor = {
        attach: function (context, settings) {
            once('flowdrop-agents-init', '#flowdrop-agents-editor', context).forEach(function (editorContainer) {
                // Skip if already initialized.
                if (editorContainer.dataset.flowdropInitialized) {
                    return;
                }

                const config = drupalSettings.flowdrop_agents || {};

                // Check if FlowDrop is available.
                if (typeof window.FlowDrop === 'undefined') {
                    editorContainer.innerHTML = `
            <div style="padding: 20px; text-align: center; color: #d32f2f;">
              <h3>FlowDrop Editor Not Available</h3>
              <p>The FlowDrop library could not be loaded. Please ensure the flowdrop_ui module is enabled.</p>
            </div>
          `;
                    console.error('FlowDrop Agents: FlowDrop library not loaded');
                    return;
                }

                // Prepare workflow data.
                const workflowData = config.workflow || {
                    id: config.workflowId || 'new_agent',
                    name: 'New AI Agent',
                    description: '',
                    nodes: [],
                    edges: [],
                    metadata: {
                        version: '1.0.0',
                        aiAgentMode: true,
                    },
                };

                // Create endpoint configuration for FlowDrop Agents API.
                // Uses our custom endpoints that return AI tools, agents, and assistants.
                const endpointConfig = {
                    baseUrl: '/api/flowdrop-agents',
                    endpoints: {
                        nodes: {
                            list: '/nodes',
                            get: '/nodes/{id}/metadata',
                            byCategory: '/nodes/by-category',
                            metadata: '/nodes/{id}/metadata',
                        },
                        workflows: {
                            list: '/workflows',
                            get: '/workflows/{id}',
                            create: '/workflows',
                            update: '/workflows/{id}',
                            delete: '/workflows/{id}',
                            validate: '/workflows/{id}/validate',
                            export: '/workflows/{id}',
                            import: '/workflows',
                        },
                        portConfig: '/port-config',
                    },
                    timeout: 30000,
                    headers: {
                        'Content-Type': 'application/json',
                        'Accept': 'application/json',
                    },
                };

                // Get the modeler save URL from Modeler API settings.
                const modelerApiSettings = drupalSettings.modeler_api || {};
                const saveUrl = modelerApiSettings.save_url || `/admin/modeler_api/ai_agent/${config.modeler || 'flowdrop_agents'}/save`;
                let isNew = config.isNew || false;

                function validateWorkflow(workflow) {
                    const warnings = [];
                    const nodes = workflow.nodes || [];
                    const edges = workflow.edges || [];

                    const chatbotNodes = nodes.filter(function (n) {
                        return n.data && n.data.nodeType === 'chatbot';
                    });

                    if (chatbotNodes.length === 0) {
                        return warnings;
                    }

                    const connectedChatbots = new Set();
                    edges.forEach(function (edge) {
                        if (edge.source && edge.source.startsWith('chatbot_')) {
                            connectedChatbots.add(edge.source);
                        }
                    });

                    chatbotNodes.forEach(function (chatbot) {
                        if (!connectedChatbots.has(chatbot.id)) {
                            const label = chatbot.data.label || chatbot.data.config?.botName || chatbot.id;
                            warnings.push(Drupal.t('Chatbot "@name" is not connected to the assistant', { '@name': label }));
                        }
                    });

                    return warnings;
                }

                async function saveAgent() {
                    try {
                        const app = editorContainer.flowdropApp || window.currentFlowDropApp;
                        let currentWorkflow;

                        if (app && typeof app.getWorkflow === 'function') {
                            currentWorkflow = app.getWorkflow();
                        } else {
                            console.warn('FlowDrop Agents: No app reference, using initial workflowData');
                            currentWorkflow = workflowData;
                        }

                        const validationWarnings = validateWorkflow(currentWorkflow);
                        if (validationWarnings.length > 0) {
                            validationWarnings.forEach(function (warning) {
                                showNotification(warning, 'warning');
                            });
                        }

                        // CRITICAL: Ensure the workflow ID matches the entity being edited.
                        // This prevents saving to the wrong entity when sub-agents are expanded.
                        // The config.workflowId is the authoritative ID of the entity being edited.
                        const originalId = config.workflowId;
                        if (originalId && currentWorkflow.id !== originalId) {
                            console.warn('FlowDrop Agents: Correcting workflow ID from', currentWorkflow.id, 'to', originalId);
                            currentWorkflow = {
                                ...currentWorkflow,
                                id: originalId
                            };
                        }

                        // Also ensure we preserve the original label if workflow label was corrupted
                        // (e.g., by FlowDrop using first node's label)
                        const originalLabel = workflowData.label || workflowData.name;
                        if (originalLabel && !currentWorkflow.label) {
                            currentWorkflow.label = originalLabel;
                        }

                        // Get CSRF token.
                        const tokenUrl = modelerApiSettings.token_url || '/session/token';
                        const tokenResponse = await fetch(tokenUrl);
                        const csrfToken = await tokenResponse.text();

                        // Save via Modeler API.
                        const response = await fetch(saveUrl, {
                            method: 'POST',
                            headers: {
                                'Content-Type': 'application/json',
                                'Accept': 'application/json',
                                'X-CSRF-Token': csrfToken,
                                'X-Modeler-API-isNew': isNew ? 'true' : 'false',
                            },
                            body: JSON.stringify(currentWorkflow),
                        });

                        if (!response.ok) {
                            throw new Error('Save failed: ' + response.statusText);
                        }

                        const result = await response.json();

                        if (result.success !== false) {
                            const entityType = config.isAssistantMode ? 'Assistant' : 'Agent';
                            showNotification(Drupal.t('@type saved successfully', { '@type': entityType }), 'success');

                            if (typeof Drupal.announce === 'function') {
                                Drupal.announce(Drupal.t('@type saved successfully', { '@type': entityType }));
                            }

                            // Mark as no longer new after successful save
                            if (config.isNew) {
                                config.isNew = false;
                                // Update the data attribute on the container
                                const container = document.getElementById('flowdrop-agents-editor');
                                if (container) {
                                    container.setAttribute('data-is-new', 'false');
                                }
                            }

                            const savedChatbots = result.saved_chatbots || [];
                            const chatbotNodes = (currentWorkflow.nodes || []).filter(function (n) {
                                return n.data && n.data.nodeType === 'chatbot';
                            });

                            if (chatbotNodes.length > 0) {
                                if (savedChatbots.length === chatbotNodes.length) {
                                    showNotification(Drupal.t('@count chatbot(s) saved', { '@count': savedChatbots.length }), 'info');
                                } else if (savedChatbots.length > 0) {
                                    showNotification(Drupal.t('Warning: Only @saved of @total chatbot(s) saved', {
                                        '@saved': savedChatbots.length,
                                        '@total': chatbotNodes.length
                                    }), 'warning');
                                } else {
                                    showNotification(Drupal.t('Warning: Chatbots could not be saved'), 'warning');
                                }
                            }

                            markAsSaved();

                            if (app && typeof app.markAsSaved === 'function') {
                                app.markAsSaved();
                            }

                            // Get the label - prioritize the backend response which has the saved label.
                            let entityLabel = result.assistant_label;

                            // If backend didn't provide label, try workflow sources.
                            if (!entityLabel || entityLabel === 'New AI Agent') {
                                // Try workflow label/name first.
                                entityLabel = currentWorkflow.label || currentWorkflow.name;

                                // If still no label, try to get it from the main assistant node.
                                if (!entityLabel || entityLabel === 'New AI Agent') {
                                    const assistantNode = currentWorkflow.nodes?.find(n =>
                                        n.data?.nodeType === 'assistant' ||
                                        n.id === 'agent_' + currentWorkflow.id
                                    );
                                    if (assistantNode) {
                                        entityLabel = assistantNode.data?.config?.label ||
                                                     assistantNode.data?.label ||
                                                     entityLabel;
                                    }
                                }
                            }

                            // Final fallback.
                            entityLabel = entityLabel || 'Agent';

                            // Use the backend's is_new flag, fallback to frontend isNew.
                            const wasNew = result.is_new !== undefined ? result.is_new : isNew;

                            // Use appropriate message based on whether this is new or existing.
                            const messageKey = wasNew ?
                                'Created @label.' :
                                'Updated @label.';

                            sessionStorage.setItem('flowdrop_save_message', JSON.stringify({
                                type: 'status',
                                message: Drupal.t(messageKey, { '@label': entityLabel })
                            }));

                            // After first save, mark as no longer new for subsequent saves in same session
                            if (wasNew) {
                                isNew = false;
                            }

                            // Redirect to appropriate list page based on whether this is an assistant or agent.
                            const redirectUrl = config.isAssistantMode
                                ? '/admin/config/ai/ai-assistant'
                                : '/admin/config/ai/agents';
                            window.location.href = redirectUrl;
                        } else {
                            throw new Error(result.message || 'Save failed');
                        }

                        return result;
                    } catch (error) {
                        console.error('FlowDrop Agents: Save error', error);
                        showNotification(Drupal.t('Failed to save: @error', { '@error': error.message }), 'error');
                        throw error;
                    }
                }

                /**
                 * Marks the workflow as having unsaved changes.
                 */
                function markAsUnsaved() {
                    if (!hasUnsavedChanges) {
                        hasUnsavedChanges = true;
                        updateSaveButtonState();
                    }
                }

                /**
                 * Marks the workflow as saved (no unsaved changes).
                 */
                function markAsSaved() {
                    hasUnsavedChanges = false;
                    updateSaveButtonState();
                }

                /**
                 * Updates the save button to show unsaved state.
                 */
                function updateSaveButtonState() {
                    const saveBtn = editorContainer.querySelector('.flowdrop-navbar__actions .btn-primary, .flowdrop-navbar__actions [data-action="save"]');
                    const indicator = document.getElementById('flowdrop-unsaved-indicator');

                    if (hasUnsavedChanges) {
                        // Add/update unsaved indicator
                        if (!indicator) {
                            const newIndicator = document.createElement('div');
                            newIndicator.id = 'flowdrop-unsaved-indicator';
                            newIndicator.className = 'flowdrop-unsaved-indicator';
                            newIndicator.innerHTML = '<span class="flowdrop-unsaved-dot"></span> Unsaved changes';
                            // Insert at the start of the navbar actions
                            const navbar = editorContainer.querySelector('.flowdrop-navbar__actions');
                            if (navbar) {
                                navbar.insertBefore(newIndicator, navbar.firstChild);
                            }
                        }
                        if (saveBtn) {
                            saveBtn.classList.add('flowdrop-btn-unsaved');
                        }
                    } else {
                        // Remove indicator
                        if (indicator) {
                            indicator.remove();
                        }
                        if (saveBtn) {
                            saveBtn.classList.remove('flowdrop-btn-unsaved');
                        }
                    }
                }

                /**
                 * Sets up change detection listeners.
                 */
                function setupChangeDetection() {
                    // Listen for FlowDrop workflow changes
                    if (window.FlowDrop && window.FlowDrop.events) {
                        window.FlowDrop.events.on('workflow:changed', markAsUnsaved);
                        window.FlowDrop.events.on('node:changed', markAsUnsaved);
                        window.FlowDrop.events.on('edge:changed', markAsUnsaved);
                    }

                    // Fallback: Listen for any changes in the editor container
                    const observer = new MutationObserver(function (mutations) {
                        // Ignore mutations from our own indicator
                        const isOurMutation = mutations.some(function (m) {
                            return m.target.id === 'flowdrop-unsaved-indicator' ||
                                m.target.classList?.contains('flowdrop-notification');
                        });
                        if (!isOurMutation && !hasUnsavedChanges) {
                            // Only mark as unsaved for significant mutations
                            const hasSignificantChange = mutations.some(function (m) {
                                return m.type === 'childList' && m.addedNodes.length > 0;
                            });
                            if (hasSignificantChange) {
                                markAsUnsaved();
                            }
                        }
                    });

                    // Observe the React Flow container for changes
                    const flowContainer = editorContainer.querySelector('.react-flow');
                    if (flowContainer) {
                        observer.observe(flowContainer, {
                            childList: true,
                            subtree: true,
                            attributes: true,
                            attributeFilter: ['data-id', 'transform']
                        });
                    }

                    // Store observer for cleanup
                    editorContainer.changeObserver = observer;
                }

                /**
                 * Shows a toast notification message.
                 *
                 * @param {string} message - The message to display.
                 * @param {string} type - The type: 'success', 'error', 'warning', 'info'.
                 */
                function showNotification(message, type) {
                    // Remove any existing notifications first.
                    const existing = document.querySelectorAll('.flowdrop-notification');
                    existing.forEach(function (el) { el.remove(); });

                    // Create notification element.
                    const notification = document.createElement('div');
                    notification.className = 'flowdrop-notification flowdrop-notification--' + type;
                    notification.setAttribute('role', 'alert');
                    notification.innerHTML = `
            <span class="flowdrop-notification__icon"></span>
            <span class="flowdrop-notification__message">${message}</span>
            <button class="flowdrop-notification__close" aria-label="Close">&times;</button>
          `;

                    // Style the notification.
                    Object.assign(notification.style, {
                        position: 'fixed',
                        top: '20px',
                        right: '20px',
                        padding: '12px 40px 12px 16px',
                        borderRadius: '6px',
                        boxShadow: '0 4px 12px rgba(0, 0, 0, 0.15)',
                        zIndex: '10000',
                        display: 'flex',
                        alignItems: 'center',
                        gap: '10px',
                        fontSize: '14px',
                        fontWeight: '500',
                        maxWidth: '400px',
                        animation: 'flowdrop-slide-in 0.3s ease-out',
                    });

                    // Set colors based on type.
                    const colors = {
                        success: { bg: '#d4edda', border: '#28a745', text: '#155724' },
                        error: { bg: '#f8d7da', border: '#dc3545', text: '#721c24' },
                        warning: { bg: '#fff3cd', border: '#ffc107', text: '#856404' },
                        info: { bg: '#d1ecf1', border: '#17a2b8', text: '#0c5460' },
                    };
                    const color = colors[type] || colors.info;

                    notification.style.backgroundColor = color.bg;
                    notification.style.borderLeft = '4px solid ' + color.border;
                    notification.style.color = color.text;

                    // Add close button styling.
                    const closeBtn = notification.querySelector('.flowdrop-notification__close');
                    Object.assign(closeBtn.style, {
                        position: 'absolute',
                        right: '8px',
                        top: '50%',
                        transform: 'translateY(-50%)',
                        background: 'none',
                        border: 'none',
                        fontSize: '20px',
                        cursor: 'pointer',
                        color: color.text,
                        opacity: '0.7',
                    });

                    // Close on click.
                    closeBtn.addEventListener('click', function () {
                        notification.remove();
                    });

                    // Add to page.
                    document.body.appendChild(notification);

                    // Auto-remove after 4 seconds.
                    setTimeout(function () {
                        if (notification.parentNode) {
                            notification.style.animation = 'flowdrop-slide-out 0.3s ease-in forwards';
                            setTimeout(function () { notification.remove(); }, 300);
                        }
                    }, 4000);
                }

                // Make save function available globally.
                window.flowdropSave = saveAgent;

                /**
                 * Updates all nodes' nodeType to switch between visual styles.
                 * 'default' = WorkflowNode, 'simple' = SimpleNode (compact)
                 */
                function updateAllNodesLayout(layout) {
                    const app = editorContainer.flowdropApp;
                    if (!app || typeof app.getWorkflow !== 'function') {
                        console.warn('FlowDrop Agents: Cannot update layout - no app reference');
                        return;
                    }

                    const workflow = app.getWorkflow();
                    if (!workflow || !workflow.nodes) {
                        return;
                    }

                    // Create new node objects with updated nodeType
                    const updatedNodes = workflow.nodes.map(function (node) {
                        const newConfig = { ...(node.data?.config || {}) };
                        // Set nodeType to 'simple' for compact, or remove for default
                        if (layout === 'simple') {
                            newConfig.nodeType = 'simple';
                        } else {
                            delete newConfig.nodeType;
                        }
                        return {
                            ...node,
                            data: {
                                ...node.data,
                                config: newConfig
                            }
                        };
                    });

                    // Create new workflow object with updated nodes
                    const updatedWorkflow = {
                        ...workflow,
                        nodes: updatedNodes
                    };

                    // Use FlowDrop's workflowActions.initialize to fully reset with new data
                    if (window.FlowDrop && window.FlowDrop.workflowActions) {
                        window.FlowDrop.workflowActions.initialize(updatedWorkflow);
                    }

                    viewSettings.nodeLayout = layout;
                    console.log('FlowDrop Agents: Layout changed to', layout);
                }

                /**
                 * Reloads workflow with new expansion mode via AJAX.
                 */
                async function reloadWorkflowWithExpansion(expansionMode) {
                    const agentId = config.workflowId;
                    if (!agentId) {
                        console.warn('FlowDrop Agents: No agent ID for expansion reload');
                        return;
                    }

                    const url = '/api/flowdrop-agents/workflow/' + agentId + '?expansion=' + expansionMode;

                    try {
                        const response = await fetch(url, {
                            headers: { 'Accept': 'application/json' }
                        });
                        const result = await response.json();

                        if (result.success && result.data) {
                            // Use FlowDrop's workflowActions.initialize to load the new workflow
                            if (window.FlowDrop && window.FlowDrop.workflowActions) {
                                window.FlowDrop.workflowActions.initialize(result.data);
                                viewSettings.agentExpansion = expansionMode;
                                console.log('FlowDrop Agents: Expansion mode changed to', expansionMode);
                            } else {
                                console.error('FlowDrop Agents: workflowActions not available');
                            }
                        } else {
                            console.error('FlowDrop Agents: Failed to load workflow with expansion', result.error);
                        }
                    } catch (error) {
                        console.error('FlowDrop Agents: Error reloading workflow:', error);
                    }
                }

                /**
                 * Creates the view controls toolbar.
                 */
                function createViewControls() {
                    const toolbar = document.createElement('div');
                    toolbar.className = 'flowdrop-agents-view-controls';
                    toolbar.innerHTML = `
            <div class="flowdrop-agents-control">
              <label for="layout-select">Layout:</label>
              <select id="layout-select" class="flowdrop-agents-select">
                <option value="default">Default</option>
                <option value="simple">Compact</option>
              </select>
            </div>
            <div class="flowdrop-agents-control">
              <label for="expansion-select">Agents:</label>
              <select id="expansion-select" class="flowdrop-agents-select">
                <option value="expanded">Expanded</option>
                <option value="grouped">Grouped</option>
                <option value="collapsed">Collapsed</option>
              </select>
            </div>
          `;

                    // Add event listener for layout mode
                    const layoutSelect = toolbar.querySelector('#layout-select');
                    layoutSelect.value = viewSettings.nodeLayout;
                    layoutSelect.addEventListener('change', function (e) {
                        updateAllNodesLayout(e.target.value);
                    });

                    // Add event listener for expansion mode
                    const expansionSelect = toolbar.querySelector('#expansion-select');
                    expansionSelect.value = viewSettings.agentExpansion;
                    expansionSelect.addEventListener('change', function (e) {
                        reloadWorkflowWithExpansion(e.target.value);
                    });

                    return toolbar;
                }

                /**
                 * Injects view controls into the FlowDrop navbar.
                 */
                function injectViewControls() {
                    // Find the navbar actions area
                    const navbar = editorContainer.querySelector('.flowdrop-navbar__actions');
                    if (navbar) {
                        const controls = createViewControls();
                        // Insert at the beginning of actions
                        navbar.insertBefore(controls, navbar.firstChild);
                    } else {
                        // Fallback: add controls above the editor
                        const controls = createViewControls();
                        editorContainer.parentNode.insertBefore(controls, editorContainer);
                    }
                }

                /**
                 * Layout constants matching PHP AgentWorkflowMapper.
                 */
                const LAYOUT = {
                    toolOffsetX: 350,
                    toolSpacingY: 180,
                    toolColWidth: 300,
                    nodeWidth: 250,
                };

                /**
                 * Checks if a node is an agent.
                 */
                function isAgentNode(node) {
                    const cat = node.data?.category || node.category;
                    return cat === 'Agent' || cat === 'agents' || node.type === 'agent' ||
                        node.data?.nodeType === 'agent' || node.data?.metadata?.type === 'agent';
                }

                /**
                 * Checks if a node is a tool (not an agent).
                 */
                function isToolNode(node) {
                    const cat = node.data?.category || node.category;
                    if (cat === 'Agent' || cat === 'agents' || node.type === 'agent') return false;
                    return node.data?.nodeType === 'tool' || node.data?.metadata?.type === 'tool' || true;
                }

                /**
                 * Gets tools connected to an agent.
                 */
                function getAgentTools(agentId, nodes, edges) {
                    const toolIds = edges
                        .filter(e => e.source === agentId)
                        .map(e => e.target);
                    return nodes.filter(n => toolIds.includes(n.id) && isToolNode(n));
                }

                /**
                 * Calculates grid position for a new tool attached to an agent.
                 * Matches PHP AgentWorkflowMapper layout algorithm.
                 */
                function calculateToolPosition(agent, existingTools) {
                    const agentX = agent.position.x;
                    const agentY = agent.position.y;
                    const toolCount = existingTools.length + 1; // +1 for new tool

                    // Dynamic columns: sqrt approximation, min 2 cols
                    const cols = Math.max(2, Math.ceil(Math.sqrt(toolCount)));
                    const rows = Math.ceil(toolCount / cols);

                    // Calculate grid height to center vertically relative to agent
                    const gridHeight = (rows - 1) * LAYOUT.toolSpacingY;
                    const startToolY = (agentY + 100) - (gridHeight / 2);
                    const toolX = agentX + LAYOUT.toolOffsetX;

                    // New tool goes at the end
                    const newIndex = existingTools.length;
                    const col = newIndex % cols;
                    const row = Math.floor(newIndex / cols);

                    return {
                        x: toolX + (col * LAYOUT.toolColWidth),
                        y: startToolY + (row * LAYOUT.toolSpacingY)
                    };
                }

                /**
                 * Sets up drag-over highlighting for agent nodes.
                 */
                function setupDragOverHighlight() {
                    let currentHighlight = null;

                    // Add CSS for highlight effect
                    const style = document.createElement('style');
                    style.textContent = `
                        .flowdrop-agent-drop-target {
                            outline: 3px dashed var(--color-ref-teal-500, #14b8a6) !important;
                            outline-offset: 4px;
                            box-shadow: 0 0 20px rgba(20, 184, 166, 0.4) !important;
                            transition: outline 0.15s ease, box-shadow 0.15s ease;
                        }
                        .flowdrop-agent-drop-target::after {
                            content: 'Drop to add tool';
                            position: absolute;
                            top: -28px;
                            left: 50%;
                            transform: translateX(-50%);
                            background: var(--color-ref-teal-500, #14b8a6);
                            color: white;
                            padding: 4px 12px;
                            border-radius: 4px;
                            font-size: 12px;
                            font-weight: 500;
                            white-space: nowrap;
                            z-index: 1000;
                        }
                    `;
                    document.head.appendChild(style);

                    // Track drag state
                    let isDraggingTool = false;

                    // Listen for sidebar drag start
                    document.addEventListener('dragstart', function (e) {
                        const target = e.target.closest('.flowdrop-node-item, [draggable="true"]');
                        if (target) {
                            // Check if it's a tool (not an agent) being dragged
                            const nodeData = target.dataset.nodeType || target.dataset.category || '';
                            isDraggingTool = !nodeData.toLowerCase().includes('agent');
                        }
                    });

                    document.addEventListener('dragend', function () {
                        isDraggingTool = false;
                        clearHighlight();
                    });

                    // Listen for drag over on the flow container
                    editorContainer.addEventListener('dragover', function (e) {
                        if (!isDraggingTool) return;

                        // Find agent node under cursor
                        const flowContainer = editorContainer.querySelector('.react-flow');
                        if (!flowContainer) return;

                        const agentNode = findAgentNodeAtPoint(e.clientX, e.clientY);

                        if (agentNode && agentNode !== currentHighlight) {
                            clearHighlight();
                            agentNode.classList.add('flowdrop-agent-drop-target');
                            currentHighlight = agentNode;
                        } else if (!agentNode && currentHighlight) {
                            clearHighlight();
                        }
                    });

                    function clearHighlight() {
                        if (currentHighlight) {
                            currentHighlight.classList.remove('flowdrop-agent-drop-target');
                            currentHighlight = null;
                        }
                        // Also clear any other highlights
                        document.querySelectorAll('.flowdrop-agent-drop-target').forEach(el => {
                            el.classList.remove('flowdrop-agent-drop-target');
                        });
                    }

                    function findAgentNodeAtPoint(x, y) {
                        // Get all elements at point
                        const elements = document.elementsFromPoint(x, y);

                        for (const el of elements) {
                            // Look for React Flow node wrapper
                            const nodeWrapper = el.closest('.react-flow__node');
                            if (!nodeWrapper) continue;

                            // Check if it's an agent node by data attributes or content
                            const nodeId = nodeWrapper.dataset.id;
                            if (!nodeId) continue;

                            // Get workflow data to check node type
                            const app = editorContainer.flowdropApp;
                            if (!app) continue;

                            let nodes = [];
                            if (typeof app.getWorkflow === 'function') {
                                const wf = app.getWorkflow();
                                nodes = wf ? (wf.nodes || []) : [];
                            }

                            const node = nodes.find(n => n.id === nodeId);
                            if (node && isAgentNode(node)) {
                                return nodeWrapper;
                            }
                        }
                        return null;
                    }

                    // Store reference for cleanup
                    editorContainer.dragHighlightStyle = style;
                }

                /**
                 * Sets up auto-attach logic for dropped tools with grid snapping.
                 */
                function setupAutoAttach() {
                    // Track which nodes we've already processed for auto-attachment
                    const processedNodes = new Set();
                    let previousNodeCount = 0;

                    const autoAttachHandler = function () {
                        // Check if app is ready
                        if (!editorContainer.flowdropApp) return;
                        const app = editorContainer.flowdropApp;

                        let nodes = [];
                        let edges = [];

                        if (typeof app.getWorkflow === 'function') {
                            const wf = app.getWorkflow();
                            if (!wf) return; // Workflow not ready yet
                            nodes = wf.nodes || [];
                            edges = wf.edges || [];
                        } else if (typeof app.getNodes === 'function') {
                            nodes = app.getNodes();
                            if (typeof app.getEdges === 'function') {
                                edges = app.getEdges();
                            }
                        }

                        // Optimization: Only check if node count increased
                        if (nodes.length <= previousNodeCount) {
                            previousNodeCount = nodes.length;
                            return;
                        }
                        previousNodeCount = nodes.length;

                        // Identify new nodes (tools only)
                        const newTools = nodes.filter(node => {
                            if (processedNodes.has(node.id)) return false;

                            // Mark as processed immediately so we don't try again
                            processedNodes.add(node.id);

                            // Ignore agents
                            if (isAgentNode(node)) return false;

                            // Check if already connected (has incoming edge)
                            const hasIncoming = edges.some(e => e.target === node.id);
                            if (hasIncoming) return false;

                            return true;
                        });

                        if (newTools.length === 0) return;

                        console.log('FlowDrop Agents: Found', newTools.length, 'new tools to auto-attach.');

                        // Find agents
                        const agents = nodes.filter(n => isAgentNode(n));

                        if (agents.length === 0) return;

                        newTools.forEach(tool => {
                            let nearest = null;
                            let minDist = Infinity;
                            const nodeX = tool.position.x;
                            const nodeY = tool.position.y;

                            agents.forEach(agent => {
                                const ax = agent.position.x;
                                const ay = agent.position.y;
                                const dist = Math.sqrt(Math.pow(nodeX - ax, 2) + Math.pow(nodeY - ay, 2));

                                if (dist < minDist) {
                                    minDist = dist;
                                    nearest = agent;
                                }
                            });

                            // Threshold: 1000px
                            if (nearest && minDist < 1000) {
                                console.log('FlowDrop Agents: Auto-attaching tool', tool.id, 'to agent', nearest.id);

                                // Calculate grid position for the new tool
                                const existingTools = getAgentTools(nearest.id, nodes, edges);
                                const snapPosition = calculateToolPosition(nearest, existingTools);

                                // Move tool to calculated position
                                if (window.FlowDrop.workflowActions && typeof window.FlowDrop.workflowActions.updateNodePosition === 'function') {
                                    window.FlowDrop.workflowActions.updateNodePosition(tool.id, snapPosition);
                                } else {
                                    tool.position = snapPosition;
                                }

                                // Create edge
                                const edge = {
                                    id: 'e-' + nearest.id + '-' + tool.id + '-' + Date.now(),
                                    source: nearest.id,
                                    target: tool.id,
                                    sourceHandle: nearest.id + '-output-tools',
                                    targetHandle: tool.id + '-input-tool',
                                    type: 'default',
                                    data: { dataType: 'tool' },
                                };

                                // Add edge via global action helper or app method
                                if (window.FlowDrop.workflowActions && typeof window.FlowDrop.workflowActions.addEdge === 'function') {
                                    window.FlowDrop.workflowActions.addEdge(edge);
                                    showNotification(Drupal.t('Added tool to @agent', { '@agent': nearest.data?.label || 'Agent' }), 'success');
                                } else if (typeof app.addEdge === 'function') {
                                    app.addEdge(edge);
                                    showNotification(Drupal.t('Added tool to @agent', { '@agent': nearest.data?.label || 'Agent' }), 'success');
                                }
                            }
                        });
                    };

                    // Initial population of processed nodes so we don't attach to existing ones
                    if (editorContainer.flowdropApp) {
                        const app = editorContainer.flowdropApp;
                        let initialNodes = [];
                        if (typeof app.getWorkflow === 'function') {
                            const wf = app.getWorkflow();
                            initialNodes = wf ? (wf.nodes || []) : [];
                        }
                        initialNodes.forEach(n => processedNodes.add(n.id));
                        previousNodeCount = initialNodes.length;
                    }

                    // Set up drag-over highlighting
                    setupDragOverHighlight();

                    // Use polling since events are not reliably exposed
                    console.log('FlowDrop Agents: Starting auto-attach with grid snapping');
                    const intervalId = setInterval(autoAttachHandler, 500);
                    editorContainer.autoAttachInterval = intervalId;
                }

                /**
                 * Sets up magnetic parent dragging (moving subtree with parent).
                 */
                /**
                 * Sets up magnetic parent dragging (moving subtree with parent).
                 */
                function setupMagneticDrag() {
                    // Safety check for FlowDrop events
                    if (!window.FlowDrop || !window.FlowDrop.events) {
                        console.warn('FlowDrop Agents: FlowDrop events not available for magnetic drag');
                        return;
                    }

                    const nodePositions = new Map();

                    // Track drag start to initialize offsets
                    window.FlowDrop.events.on('node:dragstart', function (data) {
                        const node = data.node || data;
                        // console.log('FlowDrop Agents: Drag Start', node.id);
                        if (node && node.id && node.position) {
                            nodePositions.set(node.id, { x: node.position.x, y: node.position.y });
                        }
                    });

                    // Handle dragging
                    window.FlowDrop.events.on('node:drag', function (data) {
                        const node = data.node || data;
                        if (!node || !node.id || !node.position) return;

                        const lastPos = nodePositions.get(node.id);
                        if (!lastPos) {
                            nodePositions.set(node.id, { x: node.position.x, y: node.position.y });
                            return;
                        }

                        const dx = node.position.x - lastPos.x;
                        const dy = node.position.y - lastPos.y;

                        // Only process significant movement
                        if (Math.abs(dx) < 0.1 && Math.abs(dy) < 0.1) return;

                        // Check if node is an Agent (Parent)
                        // Agents pull their children. Tools typically don't pull structure unless they are parents (unlikely).
                        // console.log('FlowDrop Agents: Dragging', node.id, node.type, node.data?.metadata?.type);

                        const isAgent = node.type === 'agent' ||
                            (node.data && node.data.type === 'agent') ||
                            (node.data && node.data.metadata && node.data.metadata.type === 'agent') ||
                            (node.data && node.data.nodeType && node.data.nodeType.indexOf('agent') !== -1) ||
                            (node.data && node.data.config && node.data.config.agent_id);

                        if (isAgent) {
                            const app = editorContainer.flowdropApp;
                            let edges = [];
                            // Safe retrieval of edges
                            if (typeof app.getWorkflow === 'function') {
                                edges = app.getWorkflow().edges || [];
                            } else if (typeof app.getEdges === 'function') {
                                edges = app.getEdges();
                            }

                            // Initial children of the dragged node
                            const childrenIds = edges
                                .filter(e => e.source === node.id)
                                .map(e => e.target);

                            if (childrenIds.length > 0) {
                                // console.log('FlowDrop Agents: Moving subtree of', node.id, 'children:', childrenIds);
                                // Use a Set to prevent cyclic infinite loops
                                const movedNodes = new Set();
                                movedNodes.add(node.id); // Parent already moved by user

                                childrenIds.forEach(childId => {
                                    moveSubtree(childId, dx, dy, edges, movedNodes);
                                });
                            }
                        }

                        // Update 'last' position for next frame
                        nodePositions.set(node.id, { x: node.position.x, y: node.position.y });
                    });

                    // Recursive helper to move node and its children
                    function moveSubtree(nodeId, dx, dy, allEdges, movedNodes) {
                        if (movedNodes.has(nodeId)) return;
                        movedNodes.add(nodeId);

                        const app = editorContainer.flowdropApp;
                        let allNodes = [];
                        if (typeof app.getWorkflow === 'function') {
                            const wf = app.getWorkflow();
                            allNodes = wf ? (wf.nodes || []) : [];
                        } else if (typeof app.getNodes === 'function') {
                            allNodes = app.getNodes();
                        }

                        const targetNode = allNodes.find(n => n.id === nodeId);
                        if (!targetNode) return;

                        const newPos = {
                            x: targetNode.position.x + dx,
                            y: targetNode.position.y + dy
                        };

                        // Apply move using FlowDrop API if available to update ReactFlow state
                        if (window.FlowDrop.workflowActions && typeof window.FlowDrop.workflowActions.updateNodePosition === 'function') {
                            window.FlowDrop.workflowActions.updateNodePosition(nodeId, newPos);
                        } else {
                            // Fallback: mutate directly (might not trigger render immediately)
                            targetNode.position = newPos;
                            // Force trigger change event if we have direct access?
                            if (window.FlowDrop.events) {
                                window.FlowDrop.events.emit('node:changed', { node: targetNode });
                            }
                        }

                        // Recurse to children
                        const kids = allEdges.filter(e => e.source === nodeId).map(e => e.target);
                        kids.forEach(kidId => moveSubtree(kidId, dx, dy, allEdges, movedNodes));
                    }
                }

                function setupSidebarStyling() {
                    var agentKeywords = ['agents', 'sub-agent'];
                    var wrenchSvg = '<svg xmlns="http://www.w3.org/2000/svg" aria-hidden="true" role="img" width="1em" height="1em" viewBox="0 0 24 24" class="iconify iconify--mdi"><path fill="currentColor" d="m22.7 19l-9.1-9.1c.9-2.3.4-5-1.5-6.9c-2-2-5-2.4-7.4-1.3L9 6L6 9L1.6 4.7C.4 7.1.9 10.1 2.9 12.1c1.9 1.9 4.6 2.4 6.9 1.5l9.1 9.1c.4.4 1 .4 1.4 0l2.3-2.3c.5-.4.5-1.1.1-1.4"></path></svg>';

                    var handler = function () {
                        // Style category containers
                        var categories = document.querySelectorAll('.flowdrop-details');
                        categories.forEach(function (details) {
                            if (details.dataset.sidebarStyled) return;

                            var summary = details.querySelector('.flowdrop-details__summary');
                            if (!summary) return;

                            var categoryText = summary.textContent.trim().toLowerCase();
                            var isAgentCategory = agentKeywords.some(function (kw) {
                                return categoryText.includes(kw);
                            });

                            if (isAgentCategory) {
                                var spans = summary.querySelectorAll('span');
                                spans.forEach(function (span) {
                                    if (span.textContent.trim() === 'Agents') {
                                        span.textContent = 'Sub-Agent Tools';
                                    }
                                });
                                // Mark as agent so CSS can style differently
                                details.dataset.isAgent = 'true';
                            } else {
                                // Mark as tool (not agent)
                                details.dataset.isAgent = 'false';
                                details.classList.add('flowdrop-tool-category');
                                var iconEl = summary.querySelector('.flowdrop-node-icon');
                                if (iconEl && !iconEl.dataset.wrenchApplied) {
                                    iconEl.innerHTML = wrenchSvg;
                                    iconEl.dataset.wrenchApplied = 'true';
                                }
                            }

                            details.dataset.sidebarStyled = 'true';
                        });

                        // Style individual tool nodes (e.g., in search results)
                        var toolNodes = document.querySelectorAll('.flowdrop-node-item:not([data-tool-styled])');
                        toolNodes.forEach(function (toolNode) {
                            // Check if this is an agent node by looking at data attributes or category
                            var nodeCategory = toolNode.dataset.category || '';
                            var nodeType = toolNode.dataset.nodeType || '';
                            var nodeLabel = toolNode.textContent.trim().toLowerCase();

                            // More robust agent detection
                            var isAgent = agentKeywords.some(function (kw) {
                                return nodeCategory.toLowerCase().includes(kw) ||
                                       nodeType.toLowerCase().includes('agent') ||
                                       nodeLabel.includes('agent');
                            });

                            if (!isAgent) {
                                // This is a tool node
                                toolNode.dataset.isAgent = 'false';
                                var iconEl = toolNode.querySelector('.flowdrop-node-icon');
                                if (iconEl && !iconEl.dataset.wrenchApplied) {
                                    iconEl.innerHTML = wrenchSvg;
                                    iconEl.dataset.wrenchApplied = 'true';
                                }
                                // No need to apply inline color - CSS handles it now
                            } else {
                                // This is an agent - mark it so CSS can override
                                toolNode.dataset.isAgent = 'true';
                                // Don't apply wrench icon to agents
                            }

                            toolNode.dataset.toolStyled = 'true';
                        });

                        removeTruncationFromLabels();
                    };

                    function removeTruncationFromLabels() {
                        var sidebar = document.querySelector('[aria-label="Components sidebar"]');
                        if (!sidebar) return;
                        var truncatedElements = sidebar.querySelectorAll('.flowdrop-truncate');
                        truncatedElements.forEach(function (el) {
                            if (el.dataset.truncateRemoved) return;
                            el.classList.remove('flowdrop-truncate');
                            el.style.whiteSpace = 'normal';
                            el.style.overflow = 'visible';
                            el.style.textOverflow = 'clip';
                            el.dataset.truncateRemoved = 'true';
                        });
                    }

                    // Initial styling
                    handler();

                    // Reduced interval for faster updates (200ms instead of 2s)
                    setInterval(handler, 200);

                    // Add MutationObserver for immediate styling of new search results
                    var sidebar = document.querySelector('[aria-label="Components sidebar"]');
                    if (sidebar) {
                        var observer = new MutationObserver(function(mutations) {
                            // Only run handler if new nodes were actually added
                            var hasNewNodes = mutations.some(function(m) {
                                return m.addedNodes.length > 0;
                            });
                            if (hasNewNodes) {
                                handler();
                            }
                        });
                        observer.observe(sidebar, { childList: true, subtree: true });

                        // Store observer reference for cleanup if needed
                        editorContainer.sidebarStylingObserver = observer;
                    }
                }

                /**
                 * Sets up chatbot node styling to match tool node appearance.
                 * Adds purple color scheme and "CHAT" badge.
                 */
                function setupChatbotNodeStyling() {
                    const style = document.createElement('style');
                    style.id = 'flowdrop-chatbot-node-styling';
                    style.textContent = `
                        /* Chatbot node styling - purple theme from POC */
                        
                        /* 1. Dimensions */
                        .svelte-flow__node[data-id^="chatbot_"] {
                            width: 288px !important;
                            min-width: 288px !important;
                            max-width: 288px !important;
                        }

                        /* 2. Reset Universal Node */
                        .svelte-flow__node[data-id^="chatbot_"] .universal-node {
                            background: transparent !important;
                            border: none !important;
                            box-shadow: none !important;
                        }

                        /* 3. Main Card Styling */
                        .svelte-flow__node[data-id^="chatbot_"] .flowdrop-workflow-node {
                            background: #f3e8ff !important; /* Light purple fill */
                            border-radius: 12px !important;
                            padding: 0 !important;
                            color: #6b21a8 !important; /* Dark purple */
                            display: flex !important;
                            flex-direction: column !important;
                            box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06) !important;
                            border: 2px solid #a855f7 !important; /* Purple 500 border */
                            width: 100% !important;
                            min-height: unset !important;
                            height: auto !important;
                            position: relative;
                        }

                        /* 4. Hide Ports / White Space - BUT KEEP HANDLES */
                        /* We must NOT use display:none on the container, or handles disappear */
                        .svelte-flow__node[data-id^="chatbot_"] .flowdrop-workflow-node__ports {
                            display: block !important;
                            height: 0 !important;
                            min-height: 0 !important;
                            padding: 0 !important;
                            margin: 0 !important;
                            background: transparent !important;
                            border: none !important;
                            overflow: visible !important; /* Allow handles to stick out */
                        }

                        /* Show configuration button on hover */
                        .svelte-flow__node[data-id^="chatbot_"] .flowdrop-workflow-node__config-btn {
                            display: block !important;
                            opacity: 0;
                            transition: opacity 0.2s ease;
                            z-index: 200 !important;
                        }

                        /* Hide text content INSIDE the ports container */
                        .svelte-flow__node[data-id^="chatbot_"] .flowdrop-workflow-node__ports-header,
                        .svelte-flow__node[data-id^="chatbot_"] .flowdrop-workflow-node__ports-list .flowdrop-workflow-node__port > div {
                            display: none !important;
                        }
                        
                        /* Ensure the handle wrapper itself doesn't take up space but exists */
                        .svelte-flow__node[data-id^="chatbot_"] .flowdrop-workflow-node__ports-list,
                        .svelte-flow__node[data-id^="chatbot_"] .flowdrop-workflow-node__port {
                            height: 0 !important;
                            min-height: 0 !important;
                            padding: 0 !important;
                            margin: 0 !important;
                            display: block !important;
                        }

                        /* 5. Header Styling */
                        .svelte-flow__node[data-id^="chatbot_"] .flowdrop-workflow-node__header {
                            background: transparent !important;
                            padding: 12px 16px 4px 16px !important;
                            border: none !important;
                            color: #4c1d95 !important;
                            margin: 0 !important;
                            width: 100% !important;
                            display: flex !important;
                            gap: 12px !important;
                        }

                        .svelte-flow__node[data-id^="chatbot_"] .flowdrop-workflow-node__header > div {
                            border: none !important;
                            padding: 0 !important;
                            margin: 0 !important;
                        }

                        /* Force Title Color to Dark Purple */
                        .svelte-flow__node[data-id^="chatbot_"] h3,
                        .svelte-flow__node[data-id^="chatbot_"] .flowdrop-workflow-node__header h3,
                        .svelte-flow__node[data-id^="chatbot_"] .flowdrop-text--sm {
                            color: #4c1d95 !important;
                            font-size: 14px !important;
                            font-weight: 700 !important;
                            margin: 0 !important;
                            opacity: 1 !important;
                        }

                        /* Hide standard subtitle */
                        .svelte-flow__node[data-id^="chatbot_"] .flowdrop-workflow-node__header p {
                             display: none !important;
                        }

                        /* 6. Icon Styling - Target CONTAINER and SVG */
                        .svelte-flow__node[data-id^="chatbot_"] .flowdrop-workflow-node__icon,
                        .svelte-flow__node[data-id^="chatbot_"] .flowdrop-workflow-node__icon svg {
                            background-color: #a855f7 !important;
                            border-radius: 8px !important;
                            color: white !important;
                            width: 32px !important; 
                            height: 32px !important;
                            display: flex !important;
                            align-items: center !important;
                            justify-content: center !important;
                            box-shadow: none !important;
                            border: none !important;
                            margin: 0 !important;
                        }
                        
                        /* Inner SVG sizing */
                        .svelte-flow__node[data-id^="chatbot_"] .flowdrop-workflow-node__icon svg {
                            width: 20px !important;
                            height: 20px !important;
                            background-color: transparent !important; /* SVG itself transparent on purple bg */
                            padding: 0 !important;
                        }

                        /* 7. Custom Body Text */
                        .flowdrop-chatbot-custom-body {
                            padding: 0 16px 16px 16px;
                            color: #581c87; /* Darker purple text */
                            font-size: 12px;
                            line-height: 1.4;
                            background: transparent;
                        }

                        /* 8. Badge */
                        .flowdrop-chat-badge {
                            position: absolute;
                            top: 16px; 
                            right: 16px;
                            background: #a855f7;
                            color: white;
                            font-size: 10px;
                            font-weight: 800;
                            padding: 2px 6px;
                            border-radius: 4px;
                            text-transform: uppercase;
                            letter-spacing: 0.5px;
                            z-index: 101;
                            box-shadow: none;
                            border: none;
                        }

                        /* 9. Force Handle Positioning - MATCH TOOL NODE */
                        .svelte-flow__node[data-id^="chatbot_"] .svelte-flow__handle {
                            position: absolute !important;
                            top: 50% !important; /* Center vertically on the card body */
                            right: -9px !important; /* Push out by half width (18px/2) to sit on edge */
                            transform: translateY(-50%) !important; /* Center vertically specifically */
                            width: 18px !important; /* Match Tool size */
                            height: 18px !important; /* Match Tool size */
                            background: #a855f7 !important;
                            border: 2px solid white !important; /* Match Tool border */
                            border-radius: 50% !important;
                            display: block !important;
                            visibility: visible !important;
                            opacity: 1 !important;
                            z-index: 100 !important;
                            margin: 0 !important; /* Reset any negative margins */
                        }

                        /* Ensure Parent allows overflow for handles */
                        .svelte-flow__node[data-id^="chatbot_"] .universal-node,
                        .svelte-flow__node[data-id^="chatbot_"] .flowdrop-workflow-node {
                            overflow: visible !important;
                        }
                    `;
                    if (!document.getElementById('flowdrop-chatbot-node-styling')) {
                        document.head.appendChild(style);
                    }

                    function setupChatbotContent() {
                        const chatbotNodes = document.querySelectorAll('.svelte-flow__node[data-id^="chatbot_"], [data-id^="chatbot_"].svelte-flow__node');
                        chatbotNodes.forEach(function (node) {
                            // Find the best container for badge injection
                            let nodeContainer = node.querySelector('.universal-node');
                            if (!nodeContainer) {
                                nodeContainer = node.querySelector('.flowdrop-workflow-node');
                                if (nodeContainer) nodeContainer = nodeContainer.parentNode;
                                else nodeContainer = node.querySelector(':scope > div');
                            }
                            if (!nodeContainer) return;

                            // Ensure relative positioning for badge
                            if (nodeContainer.style.position !== 'relative') {
                                nodeContainer.style.position = 'relative';
                            }

                            // Inject Badge
                            if (!node.querySelector('.flowdrop-chat-badge')) {
                                const badge = document.createElement('div');
                                badge.className = 'flowdrop-chat-badge';
                                badge.textContent = 'CHAT';
                                nodeContainer.appendChild(badge);
                            }

                            // Inject Body Text
                            if (!node.querySelector('.flowdrop-chatbot-custom-body')) {
                                const header = node.querySelector('.flowdrop-workflow-node__header');
                                if (header) {
                                    const customBody = document.createElement('div');
                                    customBody.className = 'flowdrop-chatbot-custom-body';
                                    customBody.textContent = 'Chat Message Input on Gin theme in Content Region.';
                                    // Insert after header
                                    header.parentNode.insertBefore(customBody, header.nextSibling);
                                }
                            }

                            // --- FORCE HANDLE VISIBILITY (JS Enforcement) ---
                            // Svelte Flow often overwrites inline styles, so we overwrite back.
                            const ports = node.querySelector('.flowdrop-workflow-node__ports');
                            if (ports) {
                                ports.style.setProperty('display', 'block', 'important');
                                ports.style.setProperty('height', '0px', 'important');
                                ports.style.setProperty('min-height', '0px', 'important');
                                ports.style.setProperty('overflow', 'visible', 'important');
                                ports.style.setProperty('position', 'static', 'important');
                                ports.style.setProperty('width', '0px', 'important');
                                ports.style.setProperty('margin', '0', 'important');
                                ports.style.setProperty('padding', '0', 'important');
                            }

                            var portsList = node.querySelector('.flowdrop-workflow-node__ports-list');
                            if (portsList) {
                                portsList.style.setProperty('position', 'static', 'important');
                            }

                            var portElements = node.querySelectorAll('.flowdrop-workflow-node__port');
                            portElements.forEach(function (port) {
                                port.style.setProperty('position', 'static', 'important');
                            });

                            const handles = node.querySelectorAll('.svelte-flow__handle');
                            handles.forEach(function (handle) {
                                handle.style.setProperty('display', 'block', 'important');
                                handle.style.setProperty('visibility', 'visible', 'important');
                                handle.style.setProperty('opacity', '1', 'important');
                                handle.style.setProperty('position', 'absolute', 'important');
                                handle.style.setProperty('top', '50%', 'important');
                                handle.style.setProperty('right', '-9px', 'important');
                                handle.style.setProperty('transform', 'translateY(-50%)', 'important');
                                handle.style.setProperty('z-index', '1000', 'important');
                                handle.style.setProperty('width', '18px', 'important');
                                handle.style.setProperty('height', '18px', 'important');
                                handle.style.setProperty('background-color', '#a855f7', 'important');
                                handle.style.setProperty('border', '2px solid white', 'important');
                                handle.style.setProperty('border-radius', '50%', 'important');
                                handle.style.setProperty('margin', '0', 'important');
                            });

                            var configBtn = node.querySelector('.flowdrop-workflow-node__config-btn');
                            if (configBtn) {
                                configBtn.style.setProperty('display', 'block', 'important');
                                configBtn.style.setProperty('transition', 'opacity 0.2s ease', 'important');
                                var isHovered = node.matches(':hover');
                                if (!isHovered) {
                                    configBtn.style.setProperty('opacity', '0', 'important');
                                }

                                if (!node.hasAttribute('data-hover-setup')) {
                                    node.setAttribute('data-hover-setup', 'true');
                                    node.addEventListener('mouseenter', function() {
                                        var btn = this.querySelector('.flowdrop-workflow-node__config-btn');
                                        if (btn) btn.style.setProperty('opacity', '1', 'important');
                                    });
                                    node.addEventListener('mouseleave', function() {
                                        var btn = this.querySelector('.flowdrop-workflow-node__config-btn');
                                        if (btn) btn.style.setProperty('opacity', '0', 'important');
                                    });
                                }
                            }
                        });
                    }

                    setupChatbotContent();
                    const intervalId = setInterval(setupChatbotContent, 500);
                    if (editorContainer.chatbotStylingInterval) clearInterval(editorContainer.chatbotStylingInterval);
                    editorContainer.chatbotStylingInterval = intervalId;
                }

                function setupConfigPanelEnhancements() {
                    const style = document.createElement('style');
                    style.id = 'flowdrop-config-panel-enhancements';
                    style.textContent = `
                        .flowdrop-chatbot-panel-fix,
                        .flowdrop-chatbot-panel-fix * {
                            box-sizing: border-box !important;
                        }
                        .flowdrop-chatbot-panel-fix {
                            overflow-x: hidden !important;
                            min-width: 380px !important;
                            width: 400px !important;
                        }
                        .flowdrop-chatbot-panel-fix input[type="text"],
                        .flowdrop-chatbot-panel-fix input:not([type]),
                        .flowdrop-chatbot-panel-fix textarea,
                        .flowdrop-chatbot-panel-fix select {
                            max-width: 100% !important;
                            width: calc(100% - 40px) !important;
                        }
                        .flowdrop-chatbot-panel-fix > div {
                            max-width: 100% !important;
                            padding-right: 12px !important;
                        }
                        .flowdrop-visibility-duplicate-field {
                            display: none !important;
                        }
                        .flowdrop-visibility-subgroup-content textarea {
                            width: 100% !important;
                        }
                        
                        /* Toggle Switch Styling */
                        .flowdrop-toggle-switch {
                            position: relative;
                            display: inline-flex;
                            align-items: center;
                            cursor: pointer;
                        }
                        .flowdrop-toggle-switch input[type="checkbox"] {
                            opacity: 0;
                            width: 0;
                            height: 0;
                            position: absolute;
                        }
                        .flowdrop-toggle-slider {
                            position: relative;
                            width: 44px;
                            height: 24px;
                            background-color: #ccc;
                            border-radius: 24px;
                            transition: background-color 0.2s;
                            margin-right: 8px;
                            flex-shrink: 0;
                        }
                        .flowdrop-toggle-slider::before {
                            content: '';
                            position: absolute;
                            width: 20px;
                            height: 20px;
                            left: 2px;
                            top: 2px;
                            background-color: white;
                            border-radius: 50%;
                            transition: transform 0.2s;
                            box-shadow: 0 1px 3px rgba(0,0,0,0.2);
                        }
                        .flowdrop-toggle-switch input:checked + .flowdrop-toggle-slider {
                            background-color: #14b8a6;
                        }
                        .flowdrop-toggle-switch input:checked + .flowdrop-toggle-slider::before {
                            transform: translateX(20px);
                        }
                        .flowdrop-toggle-switch input:focus + .flowdrop-toggle-slider {
                            box-shadow: 0 0 0 2px rgba(20, 184, 166, 0.3);
                        }
                        .flowdrop-toggle-label {
                            font-weight: 500;
                        }

                        /* Property Section Styling */
                        .flowdrop-property-section {
                            border: 1px solid #e0e0e0;
                            border-radius: 8px;
                            margin: 16px 0;
                            background: #fff;
                        }
                        .flowdrop-property-section-header {
                            font-weight: 600;
                            font-size: 14px;
                            color: #333;
                            padding: 12px 16px;
                            cursor: pointer;
                            display: flex;
                            align-items: center;
                            gap: 8px;
                            border-bottom: 1px solid #e0e0e0;
                            background: #f8f9fa;
                            border-radius: 8px 8px 0 0;
                        }
                        .flowdrop-property-section-header::before {
                            content: '▼';
                            font-size: 10px;
                            transition: transform 0.2s;
                        }
                        .flowdrop-property-section.collapsed .flowdrop-property-section-header {
                            border-bottom: none;
                            border-radius: 8px;
                        }
                        .flowdrop-property-section.collapsed .flowdrop-property-section-header::before {
                            transform: rotate(-90deg);
                        }
                        .flowdrop-property-section-content {
                            padding: 16px;
                            border-left: 3px solid #e0e0e0;
                            margin-left: 12px;
                        }
                        .flowdrop-property-section.collapsed .flowdrop-property-section-content {
                            display: none;
                        }

                        /* Property Group Styling (individual properties) */
                        .flowdrop-property-group {
                            border: 1px solid #e8e8e8;
                            border-radius: 6px;
                            padding: 12px;
                            margin: 12px 0;
                            background: #fafafa;
                        }
                        .flowdrop-property-group-header {
                            font-weight: 600;
                            font-size: 13px;
                            color: #333;
                            margin-bottom: 8px;
                            padding-bottom: 8px;
                            border-bottom: 1px solid #e0e0e0;
                            cursor: pointer;
                            display: flex;
                            align-items: center;
                            gap: 6px;
                        }
                        .flowdrop-property-group-header::before {
                            content: '▼';
                            font-size: 10px;
                            transition: transform 0.2s;
                        }
                        .flowdrop-property-group.collapsed .flowdrop-property-group-header::before {
                            transform: rotate(-90deg);
                        }
                        .flowdrop-property-group.collapsed .flowdrop-property-group-content {
                            display: none;
                        }
                        .flowdrop-property-group-content > div {
                            margin-bottom: 8px;
                        }
                        .flowdrop-property-group-content > div:last-child {
                            margin-bottom: 0;
                        }
                        .flowdrop-field-hidden {
                            display: none !important;
                        }
                    `;
                    if (!document.getElementById('flowdrop-config-panel-enhancements')) {
                        document.head.appendChild(style);
                    }

                    // Watch for config panel opening
                    const observer = new MutationObserver(function (mutations) {
                        mutations.forEach(function (mutation) {
                            mutation.addedNodes.forEach(function (node) {
                                if (node.nodeType === 1) {
                                    // Look for config panel dialogs/modals
                                    const panel = node.querySelector ?
                                        (node.querySelector('[class*="config"], [class*="panel"], [class*="modal"]') ||
                                            (node.className && node.className.includes && (node.className.includes('config') || node.className.includes('panel') || node.className.includes('modal')) ? node : null))
                                        : null;
                                    if (panel || (node.className && typeof node.className === 'string' && (node.className.includes('config') || node.className.includes('panel')))) {
                                        setTimeout(function () { enhanceConfigPanel(); }, 100);
                                    }
                                }
                            });
                        });
                    });

                    observer.observe(document.body, { childList: true, subtree: true });
                    editorContainer.configPanelObserver = observer;

                    // Also run periodically to catch panels
                    const intervalId = setInterval(enhanceConfigPanel, 1000);
                    editorContainer.configPanelInterval = intervalId;
                }

                function convertCheckboxToToggle(checkbox, labelText) {
                    if (checkbox.dataset.switchified) return;
                    checkbox.dataset.switchified = 'true';

                    const wrapper = document.createElement('label');
                    wrapper.className = 'flowdrop-toggle-switch';

                    const slider = document.createElement('span');
                    slider.className = 'flowdrop-toggle-slider';

                    const textSpan = document.createElement('span');
                    textSpan.className = 'flowdrop-toggle-label';
                    textSpan.textContent = labelText;

                    checkbox.parentNode.insertBefore(wrapper, checkbox);
                    wrapper.appendChild(checkbox);
                    wrapper.appendChild(slider);
                    wrapper.appendChild(textSpan);
                }

                function enhanceChatbotConfigPanel() {
                    const chatbotGroups = {
                        placement: {
                            title: 'Block Placement',
                            collapsed: false,
                            fields: ['Theme', 'Region']
                        },
                        messages: {
                            title: 'Message Settings',
                            collapsed: false,
                            fields: ['First Message', 'Bot Name', 'Bot Image', 'Default User Name', 'Use Username', 'Default Avatar', 'Use Avatar']
                        },
                        styling: {
                            title: 'Styling Settings',
                            collapsed: true,
                            fields: ['Style', 'Width', 'Height', 'Placement', 'Collapsed Minimal', 'Add Copy Icon']
                        },
                        advanced: {
                            title: 'Advanced Settings',
                            collapsed: true,
                            fields: ['Toggle State', 'Verbose Mode']
                        },
                        visibility: {
                            title: 'Visibility',
                            collapsed: true,
                            fields: [],
                            isCollapsibleGroups: true,
                            subgroups: [
                                { id: 'pages', label: 'Pages', expanded: true },
                                { id: 'response_status', label: 'Response Status', expanded: false },
                                { id: 'roles', label: 'Roles', expanded: false },
                                { id: 'content_types', label: 'Content Types', expanded: false },
                                { id: 'vocabulary', label: 'Vocabulary', expanded: false }
                            ]
                        }
                    };

                    const labelToGroup = new Map();
                    Object.entries(chatbotGroups).forEach(function ([groupKey, groupDef]) {
                        groupDef.fields.forEach(function (fieldLabel) {
                            labelToGroup.set(fieldLabel.toLowerCase(), groupKey);
                        });
                    });

                    const allLabels = document.querySelectorAll('label');
                    const chatbotFields = new Map();
                    let isChatbotPanel = false;
                    let titleField = null;

                    allLabels.forEach(function (label) {
                        const text = (label.textContent || '').trim();
                        const textLower = text.toLowerCase();

                        let container = label.closest('div');
                        let maxUp = 5;
                        while (container && maxUp > 0) {
                            if (container.querySelector('input, select, textarea')) {
                                break;
                            }
                            container = container.parentElement;
                            maxUp--;
                        }
                        if (!container) container = label.parentElement;
                        if (!container) return;

                        if (container.closest('.flowdrop-chatbot-section')) return;
                        if (container.dataset.chatbotGrouped) return;

                        if (text === 'Title') {
                            titleField = container;
                        }

                        const groupKey = labelToGroup.get(textLower);
                        if (groupKey) {
                            isChatbotPanel = true;
                            if (!chatbotFields.has(groupKey)) {
                                chatbotFields.set(groupKey, []);
                            }
                            chatbotFields.get(groupKey).push(container);
                            container.dataset.chatbotGrouped = 'true';
                        }
                    });

                    if (!isChatbotPanel || chatbotFields.size === 0) return;

                    let insertionPoint = titleField;
                    if (!insertionPoint) {
                        const firstGroup = chatbotFields.values().next().value;
                        if (firstGroup && firstGroup.length > 0) {
                            insertionPoint = firstGroup[0];
                        }
                    }
                    if (!insertionPoint) return;

                    const parentContainer = insertionPoint.parentElement;
                    if (!parentContainer) return;

                    const groupOrder = ['placement', 'messages', 'styling', 'advanced', 'visibility'];

                    groupOrder.forEach(function (groupKey) {
                        const groupDef = chatbotGroups[groupKey];
                        const fields = chatbotFields.get(groupKey);

                        if (groupKey !== 'visibility' && (!fields || fields.length === 0)) return;

                        const section = document.createElement('div');
                        section.className = 'flowdrop-chatbot-section flowdrop-property-section';
                        if (groupDef.collapsed) {
                            section.classList.add('collapsed');
                        }

                        const header = document.createElement('div');
                        header.className = 'flowdrop-property-section-header';
                        header.textContent = groupDef.title;
                        header.addEventListener('click', function () {
                            section.classList.toggle('collapsed');
                        });

                        const content = document.createElement('div');
                        content.className = 'flowdrop-property-section-content';

                        section.appendChild(header);
                        section.appendChild(content);

                        if (titleField && titleField.nextSibling) {
                            parentContainer.insertBefore(section, titleField.nextSibling);
                            titleField = section;
                        } else {
                            parentContainer.appendChild(section);
                        }

                        if (groupDef.isCollapsibleGroups && groupDef.subgroups) {
                            renderVisibilityCollapsible(content, fields, groupDef.subgroups);
                        } else {
                            fields.forEach(function (field) {
                                content.appendChild(field);

                                var checkbox = field.querySelector('input[type="checkbox"]');
                                if (checkbox && !checkbox.dataset.switchified) {
                                    var label = field.querySelector('label');
                                    var labelText = label ? label.textContent.trim() : '';
                                    convertCheckboxToToggle(checkbox, labelText);
                                    if (label && !label.classList.contains('flowdrop-toggle-switch')) {
                                        label.style.display = 'none';
                                    }
                                }
                            });
                        }
                    });

                    setupDynamicRegions(parentContainer);
                    fixSelectLabels(parentContainer);
                    hideDuplicateVisibilityFields(parentContainer);

                    var panelContainer = parentContainer;
                    while (panelContainer && panelContainer.parentElement) {
                        if (panelContainer.scrollHeight > panelContainer.clientHeight ||
                            panelContainer.className.indexOf('panel') !== -1 ||
                            panelContainer.className.indexOf('config') !== -1) {
                            panelContainer.classList.add('flowdrop-chatbot-panel-fix');
                            break;
                        }
                        panelContainer = panelContainer.parentElement;
                    }
                }

                function hideDuplicateVisibilityFields(container) {
                    var visibilityLabels = [
                        'pages', 'show for the listed pages', 'response status',
                        'roles', 'content types', 'vocabulary', 'negate'
                    ];

                    var allLabels = container.querySelectorAll('label');
                    allLabels.forEach(function (label) {
                        var text = (label.textContent || '').trim().toLowerCase();
                        if (visibilityLabels.indexOf(text) !== -1) {
                            var fieldContainer = label.closest('div');
                            if (fieldContainer && !fieldContainer.closest('.flowdrop-visibility-subgroup')) {
                                fieldContainer.classList.add('flowdrop-visibility-duplicate-field');
                            }
                        }
                    });
                }

                /**
                 * Gets the live chatbot node from FlowDrop's current workflow.
                 * This ensures we read/write to the actual workflow data, not drupalSettings.
                 */
                function getLiveChatbotNode() {
                    var app = editorContainer.flowdropApp || window.currentFlowDropApp;
                    if (!app || typeof app.getWorkflow !== 'function') {
                        // Fallback to drupalSettings if app not available
                        var config = drupalSettings.flowdrop_agents || {};
                        var workflow = config.workflow || {};
                        var nodes = workflow.nodes || [];
                        return nodes.find(function (n) { return n.data && n.data.nodeType === 'chatbot'; });
                    }
                    var workflow = app.getWorkflow();
                    var nodes = workflow.nodes || [];
                    return nodes.find(function (n) { return n.data && n.data.nodeType === 'chatbot'; });
                }

                function renderVisibilityCollapsible(container, fields, subgroups) {
                    var chatbotNode = getLiveChatbotNode();

                    if (!chatbotNode || !chatbotNode.data) return;

                    // Ensure config object exists on the node
                    if (!chatbotNode.data.config) {
                        chatbotNode.data.config = {};
                    }
                    var chatbotConfig = chatbotNode.data.config;
                    var schema = (chatbotNode.data.metadata && chatbotNode.data.metadata.configSchema) ? chatbotNode.data.metadata.configSchema : {};
                    var properties = schema.properties || {};

                    subgroups.forEach(function (subgroup) {
                        var groupDiv = document.createElement('div');
                        groupDiv.className = 'flowdrop-visibility-subgroup';
                        groupDiv.style.cssText = 'border: 1px solid #e0e0e0; border-radius: 6px; margin-bottom: 8px; overflow: hidden;';
                        if (!subgroup.expanded) {
                            groupDiv.classList.add('collapsed');
                        }

                        var hasRestriction = getSubgroupRestriction(subgroup.id, chatbotConfig);

                        var groupHeader = document.createElement('div');
                        groupHeader.className = 'flowdrop-visibility-subgroup-header';
                        groupHeader.style.cssText = 'padding: 10px 12px; background: #f5f5f5; cursor: pointer; display: flex; justify-content: space-between; align-items: center; font-size: 13px; font-weight: 500;';

                        var headerLeft = document.createElement('span');
                        headerLeft.style.cssText = 'display: flex; align-items: center; gap: 6px;';

                        var arrow = document.createElement('span');
                        arrow.className = 'flowdrop-subgroup-arrow';
                        arrow.textContent = subgroup.expanded ? '▼' : '▶';
                        arrow.style.cssText = 'font-size: 10px;';

                        var labelSpan = document.createElement('span');
                        labelSpan.textContent = subgroup.label;

                        headerLeft.appendChild(arrow);
                        headerLeft.appendChild(labelSpan);

                        var statusSpan = document.createElement('span');
                        statusSpan.className = 'flowdrop-restriction-status';
                        statusSpan.style.cssText = 'font-size: 11px; font-weight: normal; color: ' + (hasRestriction ? '#d32f2f' : '#888') + ';';
                        statusSpan.textContent = hasRestriction ? 'Restricted' : 'Not restricted';

                        groupHeader.appendChild(headerLeft);
                        groupHeader.appendChild(statusSpan);

                        var groupContent = document.createElement('div');
                        groupContent.className = 'flowdrop-visibility-subgroup-content';
                        groupContent.style.cssText = 'padding: 12px; display: ' + (subgroup.expanded ? 'block' : 'none') + ';';

                        groupHeader.addEventListener('click', function () {
                            var isCollapsed = groupDiv.classList.toggle('collapsed');
                            groupContent.style.display = isCollapsed ? 'none' : 'block';
                            arrow.textContent = isCollapsed ? '▶' : '▼';
                        });

                        renderSubgroupContent(groupContent, subgroup.id, chatbotConfig, properties, statusSpan);

                        groupDiv.appendChild(groupHeader);
                        groupDiv.appendChild(groupContent);
                        container.appendChild(groupDiv);
                    });
                }

                function getSubgroupRestriction(subgroupId, config) {
                    if (subgroupId === 'pages') {
                        return config.visibilityPages && config.visibilityPages.trim() !== '';
                    }
                    if (subgroupId === 'response_status') {
                        return config.visibilityResponseStatus && config.visibilityResponseStatus.length > 0;
                    }
                    if (subgroupId === 'roles') {
                        return config.visibilityRoles && config.visibilityRoles.length > 0;
                    }
                    if (subgroupId === 'content_types') {
                        return config.visibilityContentTypes && config.visibilityContentTypes.length > 0;
                    }
                    if (subgroupId === 'vocabulary') {
                        return config.visibilityVocabularies && config.visibilityVocabularies.length > 0;
                    }
                    return false;
                }

                function updateRestrictionStatus(statusSpan, hasRestriction) {
                    statusSpan.style.color = hasRestriction ? '#d32f2f' : '#888';
                    statusSpan.textContent = hasRestriction ? 'Restricted' : 'Not restricted';
                }

                function renderSubgroupContent(container, subgroupId, config, properties, statusSpan) {
                    if (subgroupId === 'pages') {
                        renderPagesSubgroup(container, config, statusSpan);
                    } else if (subgroupId === 'response_status') {
                        renderResponseStatusSubgroup(container, config, statusSpan);
                    } else if (subgroupId === 'roles') {
                        renderCheckboxSubgroup(container, config, properties, 'visibilityRoles', 'visibilityRolesNegate', statusSpan);
                    } else if (subgroupId === 'content_types') {
                        renderCheckboxSubgroup(container, config, properties, 'visibilityContentTypes', 'visibilityContentTypesNegate', statusSpan);
                    } else if (subgroupId === 'vocabulary') {
                        renderCheckboxSubgroup(container, config, properties, 'visibilityVocabularies', 'visibilityVocabulariesNegate', statusSpan);
                    }
                }

                function renderPagesSubgroup(container, config, statusSpan) {
                    var currentPages = config.visibilityPages || '';
                    var currentNegate = config.visibilityPagesNegate || false;

                    var textarea = document.createElement('textarea');
                    textarea.value = currentPages;
                    textarea.placeholder = 'Enter one path per line, e.g.:\n/node/*\n/admin/*';
                    textarea.style.cssText = 'width: 100%; min-height: 80px; padding: 8px; border: 1px solid #ccc; border-radius: 4px; font-family: monospace; font-size: 12px; resize: vertical; box-sizing: border-box;';
                    textarea.addEventListener('input', function () {
                        config.visibilityPages = textarea.value;
                        updateRestrictionStatus(statusSpan, textarea.value.trim() !== '');
                        markAsUnsaved();
                    });

                    var radioGroup = document.createElement('div');
                    radioGroup.style.cssText = 'margin-top: 10px;';

                    var showRadio = createRadioOption('pages_negate', 'show', 'Show for the listed pages', !currentNegate);
                    var hideRadio = createRadioOption('pages_negate', 'hide', 'Hide for the listed pages', currentNegate);

                    showRadio.querySelector('input').addEventListener('change', function () {
                        config.visibilityPagesNegate = false;
                        markAsUnsaved();
                    });
                    hideRadio.querySelector('input').addEventListener('change', function () {
                        config.visibilityPagesNegate = true;
                        markAsUnsaved();
                    });

                    radioGroup.appendChild(showRadio);
                    radioGroup.appendChild(hideRadio);

                    container.appendChild(textarea);
                    container.appendChild(radioGroup);
                }

                function createRadioOption(name, value, label, checked) {
                    var wrapper = document.createElement('label');
                    wrapper.style.cssText = 'display: flex; align-items: center; gap: 6px; padding: 4px 0; cursor: pointer; font-size: 13px;';

                    var input = document.createElement('input');
                    input.type = 'radio';
                    input.name = name;
                    input.value = value;
                    input.checked = checked;

                    var text = document.createElement('span');
                    text.textContent = label;

                    wrapper.appendChild(input);
                    wrapper.appendChild(text);
                    return wrapper;
                }

                function renderResponseStatusSubgroup(container, config, statusSpan) {
                    var currentStatus = config.visibilityResponseStatus || [];
                    var options = [
                        { value: '200', label: 'Success (200)' },
                        { value: '403', label: 'Access denied (403)' },
                        { value: '404', label: 'Page not found (404)' }
                    ];

                    var desc = document.createElement('p');
                    desc.style.cssText = 'margin: 0 0 8px 0; font-size: 12px; color: #666;';
                    desc.textContent = 'Show on pages with selected response status:';
                    container.appendChild(desc);

                    options.forEach(function (opt) {
                        var isChecked = currentStatus.indexOf(opt.value) !== -1;
                        var checkbox = createCheckboxOption('response_status_' + opt.value, opt.label, isChecked);
                        checkbox.querySelector('input').addEventListener('change', function (e) {
                            if (e.target.checked) {
                                if (currentStatus.indexOf(opt.value) === -1) {
                                    currentStatus.push(opt.value);
                                }
                            } else {
                                var idx = currentStatus.indexOf(opt.value);
                                if (idx !== -1) currentStatus.splice(idx, 1);
                            }
                            config.visibilityResponseStatus = currentStatus;
                            updateRestrictionStatus(statusSpan, currentStatus.length > 0);
                            markAsUnsaved();
                        });
                        container.appendChild(checkbox);
                    });
                }

                function renderCheckboxSubgroup(container, config, properties, fieldName, negateFieldName, statusSpan) {
                    var fieldSchema = properties[fieldName];
                    if (!fieldSchema || !fieldSchema.items) return;

                    var enumValues = fieldSchema.items.enum || [];
                    var enumNames = fieldSchema.items.enumNames || enumValues;
                    var currentValues = config[fieldName] || [];
                    var currentNegate = config[negateFieldName] || false;

                    if (enumValues.length === 0) {
                        var noOptions = document.createElement('p');
                        noOptions.style.cssText = 'margin: 0; font-size: 12px; color: #888; font-style: italic;';
                        noOptions.textContent = 'No options available';
                        container.appendChild(noOptions);
                        return;
                    }

                    var checkboxContainer = document.createElement('div');
                    checkboxContainer.style.cssText = 'max-height: 150px; overflow-y: auto; border: 1px solid #e0e0e0; border-radius: 4px; padding: 8px;';

                    enumValues.forEach(function (value, idx) {
                        var label = enumNames[idx] || value;
                        var isChecked = currentValues.indexOf(value) !== -1;
                        var checkbox = createCheckboxOption(fieldName + '_' + value, label, isChecked);
                        checkbox.querySelector('input').addEventListener('change', function (e) {
                            if (e.target.checked) {
                                if (currentValues.indexOf(value) === -1) {
                                    currentValues.push(value);
                                }
                            } else {
                                var idx = currentValues.indexOf(value);
                                if (idx !== -1) currentValues.splice(idx, 1);
                            }
                            config[fieldName] = currentValues;
                            updateRestrictionStatus(statusSpan, currentValues.length > 0);
                            markAsUnsaved();
                        });
                        checkboxContainer.appendChild(checkbox);
                    });

                    container.appendChild(checkboxContainer);

                    var negateWrapper = document.createElement('div');
                    negateWrapper.style.cssText = 'margin-top: 10px; padding-top: 10px; border-top: 1px solid #e0e0e0;';

                    var negateToggle = createToggleSwitch(negateFieldName, 'Negate (exclude selected)', currentNegate);
                    negateToggle.querySelector('input').addEventListener('change', function (e) {
                        config[negateFieldName] = e.target.checked;
                        markAsUnsaved();
                    });
                    negateWrapper.appendChild(negateToggle);
                    container.appendChild(negateWrapper);
                }

                function createCheckboxOption(id, label, checked) {
                    var wrapper = document.createElement('label');
                    wrapper.style.cssText = 'display: flex; align-items: center; gap: 6px; padding: 4px 0; cursor: pointer; font-size: 13px;';

                    var input = document.createElement('input');
                    input.type = 'checkbox';
                    input.id = id;
                    input.checked = checked;

                    var text = document.createElement('span');
                    text.textContent = label;

                    wrapper.appendChild(input);
                    wrapper.appendChild(text);
                    return wrapper;
                }

                function createToggleSwitch(id, label, checked) {
                    var wrapper = document.createElement('label');
                    wrapper.className = 'flowdrop-toggle-switch';
                    wrapper.style.cssText = 'display: inline-flex; align-items: center; cursor: pointer;';

                    var input = document.createElement('input');
                    input.type = 'checkbox';
                    input.id = id;
                    input.checked = checked;
                    input.style.cssText = 'opacity: 0; width: 0; height: 0; position: absolute;';

                    var slider = document.createElement('span');
                    slider.className = 'flowdrop-toggle-slider';

                    var text = document.createElement('span');
                    text.className = 'flowdrop-toggle-label';
                    text.textContent = label;

                    wrapper.appendChild(input);
                    wrapper.appendChild(slider);
                    wrapper.appendChild(text);
                    return wrapper;
                }

                function fixSelectLabels(container) {
                    var config = drupalSettings.flowdrop_agents || {};
                    var workflow = config.workflow || {};
                    var nodes = workflow.nodes || [];
                    var chatbotNode = nodes.find(function (n) { return n.data && n.data.nodeType === 'chatbot'; });

                    if (!chatbotNode || !chatbotNode.data || !chatbotNode.data.metadata || !chatbotNode.data.metadata.configSchema) {
                        return;
                    }

                    var schema = chatbotNode.data.metadata.configSchema;
                    var properties = schema.properties || {};

                    var labelToField = {
                        'theme': 'theme',
                        'style': 'styleFile',
                        'toggle state': 'toggleState',
                        'placement': 'placement'
                    };

                    var allSelects = container.querySelectorAll('select');
                    allSelects.forEach(function (select) {
                        if (select.dataset.labelsFixed) return;

                        var fieldContainer = select.closest('div');
                        if (!fieldContainer) return;

                        var label = fieldContainer.querySelector('label');
                        if (!label) return;

                        var labelText = label.textContent.trim().toLowerCase();
                        var fieldName = labelToField[labelText];

                        if (!fieldName) return;

                        var fieldSchema = properties[fieldName];
                        if (!fieldSchema || !fieldSchema.enum || !fieldSchema.enumNames) return;

                        var enumValues = fieldSchema.enum;
                        var enumNames = fieldSchema.enumNames;

                        var options = select.querySelectorAll('option');
                        options.forEach(function (option) {
                            var idx = enumValues.indexOf(option.value);
                            if (idx !== -1 && enumNames[idx]) {
                                option.textContent = enumNames[idx];
                            }
                        });

                        select.dataset.labelsFixed = 'true';
                    });
                }

                function setupDynamicRegions(container) {
                    var themeSelect = container.querySelector('select');
                    if (!themeSelect) return;

                    var themeLabel = themeSelect.closest('div');
                    if (themeLabel) {
                        var label = themeLabel.querySelector('label');
                        if (label && label.textContent.trim().toLowerCase() !== 'theme') {
                            return;
                        }
                    }

                    var regionSelect = null;
                    var allSelects = container.querySelectorAll('select');
                    allSelects.forEach(function (sel) {
                        if (sel === themeSelect) return;
                        var selLabel = sel.closest('div');
                        if (selLabel) {
                            var lbl = selLabel.querySelector('label');
                            if (lbl && lbl.textContent.trim().toLowerCase() === 'region') {
                                regionSelect = sel;
                            }
                        }
                    });

                    if (!regionSelect) return;

                    var config = drupalSettings.flowdrop_agents || {};
                    var workflow = config.workflow || {};
                    var nodes = workflow.nodes || [];
                    var chatbotNode = nodes.find(function (n) { return n.data && n.data.nodeType === 'chatbot'; });

                    if (!chatbotNode || !chatbotNode.data || !chatbotNode.data.metadata || !chatbotNode.data.metadata.configSchema) {
                        return;
                    }

                    var themeRegions = chatbotNode.data.metadata.configSchema.themeRegions;
                    if (!themeRegions) return;

                    var currentRegion = regionSelect.value;

                    function updateRegions() {
                        var selectedTheme = themeSelect.value;
                        var regions = themeRegions[selectedTheme] || [];

                        regionSelect.innerHTML = '';

                        regions.forEach(function (region) {
                            var option = document.createElement('option');
                            option.value = region.value;
                            option.textContent = region.label;
                            if (region.value === currentRegion) {
                                option.selected = true;
                            }
                            regionSelect.appendChild(option);
                        });
                    }

                    themeSelect.addEventListener('change', function () {
                        currentRegion = regionSelect.value;
                        updateRegions();
                    });

                    updateRegions();
                }

                function enhanceConfigPanel() {
                    enhanceChatbotConfigPanel();

                    // Hide "Hide Unconnected Ports" setting as it's not persisted
                    // for AI Agent nodes - it's a FlowDrop UI display preference only
                    const allFormFields = document.querySelectorAll('.form-field');
                    allFormFields.forEach(function(field) {
                        const label = field.querySelector('label[for="ext-hideUnconnectedHandles"]');
                        if (label && !field.dataset.hiddenByAgent) {
                            field.style.display = 'none';
                            field.dataset.hiddenByAgent = 'true';
                        }
                    });

                    // Hide the entire Display Settings section if all its fields are hidden
                    const displaySettingsSection = document.querySelector('.config-form__extensions');
                    if (displaySettingsSection && !displaySettingsSection.dataset.checkedForHiding) {
                        displaySettingsSection.dataset.checkedForHiding = 'true';
                        const content = displaySettingsSection.querySelector('.config-form__extensions-content');
                        if (content) {
                            const visibleFields = Array.from(content.querySelectorAll('.form-field')).filter(function(field) {
                                return field.style.display !== 'none' && !field.dataset.hiddenByAgent;
                            });
                            // If no visible fields remain, hide the entire section
                            if (visibleFields.length === 0) {
                                displaySettingsSection.style.display = 'none';
                            }
                        }
                    }

                    const allLabels = document.querySelectorAll('label');
                    const propertyFields = new Map();

                    function getFieldContainer(label) {
                        let container = label.closest('div');
                        let maxUp = 5;
                        while (container && maxUp > 0) {
                            if (container.querySelector('input, select, textarea')) {
                                return container;
                            }
                            container = container.parentElement;
                            maxUp--;
                        }
                        return label.parentElement;
                    }

                    function hasTextInput(container) {
                        if (!container) return false;
                        return !!container.querySelector('input[type="text"]');
                    }

                    function addToGroup(propName, fieldType, container) {
                        if (!propertyFields.has(propName)) propertyFields.set(propName, {});
                        propertyFields.get(propName)[fieldType] = container;
                    }

                    function convertToToggleSwitch(checkbox, labelText) {
                        if (checkbox.dataset.switchified) return;
                        checkbox.dataset.switchified = 'true';

                        const wrapper = document.createElement('label');
                        wrapper.className = 'flowdrop-toggle-switch';

                        const slider = document.createElement('span');
                        slider.className = 'flowdrop-toggle-slider';

                        const textSpan = document.createElement('span');
                        textSpan.className = 'flowdrop-toggle-label';
                        textSpan.textContent = labelText;

                        checkbox.parentNode.insertBefore(wrapper, checkbox);
                        wrapper.appendChild(checkbox);
                        wrapper.appendChild(slider);
                        wrapper.appendChild(textSpan);

                        // Hide FlowDrop's original toggle elements (track, thumb, and label)
                        // to avoid redundant UI when we've applied our custom toggle
                        const flowdropToggleParent = wrapper.closest('.form-toggle');
                        if (flowdropToggleParent) {
                            const track = flowdropToggleParent.querySelector('.form-toggle__track');
                            const label = flowdropToggleParent.querySelector('.form-toggle__label');
                            if (track) track.style.display = 'none';
                            if (label) label.style.display = 'none';
                        }
                    }

                    var toolDescToggle = null;
                    var toolDescTextarea = null;
                    var firstPropertyField = null;

                    allLabels.forEach(function (label) {
                        const text = (label.textContent || '').trim();
                        const container = getFieldContainer(label);
                        if (!container) return;

                        const inGroup = container.closest('.flowdrop-property-group, .flowdrop-property-section');
                        if (inGroup) return;

                        var checkbox = container.querySelector('input[type="checkbox"]');
                        if (checkbox && !checkbox.dataset.switchified) {
                            convertToToggleSwitch(checkbox, text);
                            var originalLabel = container.querySelector('label:not(.flowdrop-toggle-switch)');
                            if (originalLabel && originalLabel !== checkbox.closest('label')) {
                                originalLabel.style.display = 'none';
                            }
                        }

                        if (text === 'Override tool description' && checkbox) {
                            toolDescToggle = { container: container, checkbox: checkbox };
                            return;
                        }
                        if (text === 'Override Tool Description' && hasTextInput(container)) {
                            toolDescTextarea = container;
                            return;
                        }

                        let match;

                        match = text.match(/^Override\s+(.+)\s+property description$/i);
                        if (match && hasTextInput(container)) {
                            addToGroup(match[1], 'overrideTextarea', container);
                            if (!firstPropertyField) firstPropertyField = container;
                            return;
                        }

                        match = text.match(/^Override\s+(.+)\s+description$/i);
                        if (match && checkbox) {
                            addToGroup(match[1], 'overrideCheckbox', container);
                            if (!firstPropertyField) firstPropertyField = container;
                            return;
                        }

                        match = text.match(/^Restrictions for\s+(.+)$/i);
                        if (match && container.querySelector('select')) {
                            addToGroup(match[1], 'restrictionDropdown', container);
                            if (!firstPropertyField) firstPropertyField = container;
                            return;
                        }

                        match = text.match(/^Hide\s+(.+)$/i);
                        if (match && checkbox) {
                            addToGroup(match[1], 'hiddenCheckbox', container);
                            if (!firstPropertyField) firstPropertyField = container;
                            return;
                        }

                        match = text.match(/^(.+)\s+values$/i);
                        if (match && hasTextInput(container)) {
                            addToGroup(match[1], 'valuesTextarea', container);
                            if (!firstPropertyField) firstPropertyField = container;
                            return;
                        }
                    });

                    if (toolDescToggle && toolDescTextarea && !toolDescTextarea.dataset.conditionalSetup) {
                        toolDescTextarea.dataset.conditionalSetup = 'true';
                        function updateToolDescVisibility() {
                            toolDescTextarea.classList.toggle('flowdrop-field-hidden', !toolDescToggle.checkbox.checked);
                        }
                        toolDescToggle.checkbox.addEventListener('change', updateToolDescVisibility);
                        updateToolDescVisibility();
                    }

                    // Check if there are any properties with restrictionDropdown before creating section
                    var hasValidProperties = false;
                    propertyFields.forEach(function(fields) {
                        if (fields.restrictionDropdown && !fields.restrictionDropdown.dataset.grouped) {
                            hasValidProperties = true;
                        }
                    });

                    if (hasValidProperties && firstPropertyField && !document.querySelector('.flowdrop-property-section')) {
                        var section = document.createElement('div');
                        section.className = 'flowdrop-property-section';

                        var sectionHeader = document.createElement('div');
                        sectionHeader.className = 'flowdrop-property-section-header';
                        sectionHeader.textContent = 'Property setup';
                        sectionHeader.addEventListener('click', function () {
                            section.classList.toggle('collapsed');
                        });

                        var sectionContent = document.createElement('div');
                        sectionContent.className = 'flowdrop-property-section-content';

                        firstPropertyField.parentNode.insertBefore(section, firstPropertyField);
                        section.appendChild(sectionHeader);
                        section.appendChild(sectionContent);

                        propertyFields.forEach(function (fields, propName) {
                            if (!fields.restrictionDropdown) return;
                            if (fields.restrictionDropdown.dataset.grouped) return;

                            var group = document.createElement('div');
                            group.className = 'flowdrop-property-group';

                            var header = document.createElement('div');
                            header.className = 'flowdrop-property-group-header';
                            header.textContent = propName;
                            header.addEventListener('click', function (e) {
                                e.stopPropagation();
                                group.classList.toggle('collapsed');
                            });

                            var content = document.createElement('div');
                            content.className = 'flowdrop-property-group-content';

                            sectionContent.appendChild(group);
                            group.appendChild(header);
                            group.appendChild(content);

                            if (fields.overrideCheckbox) content.appendChild(fields.overrideCheckbox);
                            if (fields.overrideTextarea) content.appendChild(fields.overrideTextarea);
                            if (fields.restrictionDropdown) {
                                content.appendChild(fields.restrictionDropdown);
                                fields.restrictionDropdown.dataset.grouped = 'true';
                            }
                            if (fields.hiddenCheckbox) content.appendChild(fields.hiddenCheckbox);
                            if (fields.valuesTextarea) content.appendChild(fields.valuesTextarea);

                            if (fields.overrideCheckbox && fields.overrideTextarea) {
                                var cb = fields.overrideCheckbox.querySelector('input[type="checkbox"]');
                                if (cb && !fields.overrideTextarea.dataset.conditionalSetup) {
                                    fields.overrideTextarea.dataset.conditionalSetup = 'true';
                                    function updateOverrideVis() {
                                        fields.overrideTextarea.classList.toggle('flowdrop-field-hidden', !cb.checked);
                                    }
                                    cb.addEventListener('change', updateOverrideVis);
                                    updateOverrideVis();
                                }
                            }

                            if (fields.restrictionDropdown) {
                                var sel = fields.restrictionDropdown.querySelector('select');
                                if (sel && !fields.restrictionDropdown.dataset.restrictionSetup) {
                                    fields.restrictionDropdown.dataset.restrictionSetup = 'true';
                                    function updateRestrictionVis() {
                                        var val = sel.value;
                                        var isForce = val === 'Force value';
                                        var isOnly = val === 'Only allow certain values';
                                        if (fields.hiddenCheckbox) {
                                            fields.hiddenCheckbox.classList.toggle('flowdrop-field-hidden', !isForce);
                                        }
                                        if (fields.valuesTextarea) {
                                            fields.valuesTextarea.classList.toggle('flowdrop-field-hidden', !isForce && !isOnly);
                                        }
                                    }
                                    sel.addEventListener('change', updateRestrictionVis);
                                    updateRestrictionVis();
                                }
                            }
                        });
                    }
                }

                // FlowDrop may auto-layout nodes on hard refresh; store originals to restore
                const originalPositions = new Map();
                if (workflowData.nodes) {
                    workflowData.nodes.forEach(function (node) {
                        if (node.id && node.position) {
                            originalPositions.set(node.id, {
                                x: node.position.x,
                                y: node.position.y
                            });
                        }
                    });
                }

                function restoreOriginalPositions() {
                    if (originalPositions.size === 0) return;

                    const app = editorContainer.flowdropApp;
                    if (!app) return;

                    let currentNodes = [];
                    if (typeof app.getWorkflow === 'function') {
                        const wf = app.getWorkflow();
                        currentNodes = wf ? (wf.nodes || []) : [];
                    }

                    let positionsRestored = 0;
                    originalPositions.forEach(function (originalPos, nodeId) {
                        const node = currentNodes.find(n => n.id === nodeId);
                        if (!node) return;

                        const POSITION_TOLERANCE = 5;
                        const dx = Math.abs(node.position.x - originalPos.x);
                        const dy = Math.abs(node.position.y - originalPos.y);

                        if (dx > POSITION_TOLERANCE || dy > POSITION_TOLERANCE) {
                            console.log('FlowDrop Agents: Restoring position for', nodeId,
                                'from', node.position, 'to', originalPos);

                            if (window.FlowDrop.workflowActions &&
                                typeof window.FlowDrop.workflowActions.updateNodePosition === 'function') {
                                window.FlowDrop.workflowActions.updateNodePosition(nodeId, originalPos);
                                positionsRestored++;
                            }
                        }
                    });

                    if (positionsRestored > 0) {
                        console.log('FlowDrop Agents: Restored positions for', positionsRestored, 'nodes');
                    }
                }

                // Initialize the FlowDrop editor.
                async function initializeEditor() {
                    try {
                        const listUrl = '/admin/config/ai/agents';

                        // Build navbar actions array
                        const navbarActions = [
                            {
                                label: 'Save AI Agent',
                                href: '#',
                                variant: 'primary',
                                icon: 'mdi:floppy-disk',
                                onclick: function () {
                                    saveAgent().catch(function (error) {
                                        console.error('Save failed:', error);
                                    });
                                },
                            },
                        ];

                        // Determine if this is a new/unsaved agent
                        // Use the isNew flag which is updated after save
                        const agentId = config.workflowId || '';
                        const isAddRoute = window.location.pathname.includes('/add/');
                        const isNewAgent = config.isNew || !agentId || agentId.startsWith('new_') || isAddRoute;

                        // Add Configure button
                        // For new agents: link to add form
                        // For existing agents: link to edit form
                        navbarActions.push({
                            label: 'Configure',
                            href: isNewAgent
                                ? '/admin/config/ai/agents/add/form'
                                : '/admin/config/ai/agents/' + agentId + '/edit/form?destination=/admin/config/ai/agents',
                            variant: 'secondary',
                            icon: 'mdi:cog',
                        });

                        // Add Explore button for existing agents (not assistants)
                        if (!isNewAgent && !config.isAssistantMode) {
                            navbarActions.push({
                                label: 'Explore',
                                href: '/admin/config/ai/agents/explore?agent_id=' + agentId,
                                variant: 'secondary',
                                icon: 'mdi:compass',
                            });
                        }

                        navbarActions.push({
                            label: 'Back to List',
                            href: listUrl,
                            variant: 'secondary',
                            icon: 'mdi:arrow-back',
                        });

                        const app = await window.FlowDrop.mountFlowDropApp(editorContainer, {
                            workflow: workflowData,
                            endpointConfig: endpointConfig,
                            height: '100%',
                            width: '100%',
                            fitView: false,
                            initialViewport: { x: 0, y: 0, zoom: 0.4 },
                            showNavbar: true,
                            navbarTitle: workflowData.name || workflowData.label || 'AI Agent',
                            navbarActions: navbarActions,
                        });

                        // Store app reference.
                        window.currentFlowDropApp = app;
                        editorContainer.flowdropApp = app;
                        editorContainer.dataset.flowdropInitialized = 'true';

                        setTimeout(function () {
                            setupSidebarStyling();
                            injectViewControls();
                            setupChangeDetection();
                            setupAutoAttach();
                            setupMagneticDrag();
                            setupConfigPanelEnhancements();
                            setupChatbotNodeStyling();
                        }, 100);

                        setTimeout(function () {
                            restoreOriginalPositions();
                        }, 200);

                        setTimeout(function () {
                            if (app && typeof app.fitView === 'function') {
                                app.fitView({ padding: 0.2, minZoom: 0.3, maxZoom: 1.2 });
                            }
                        }, 500);

                        // Add Ctrl+S shortcut.
                        const keydownHandler = function (event) {
                            if ((event.ctrlKey || event.metaKey) && event.key === 's') {
                                event.preventDefault();
                                saveAgent().catch(function (error) {
                                    console.error('Save failed:', error);
                                });
                            }
                        };
                        document.addEventListener('keydown', keydownHandler);
                        editorContainer.keydownHandler = keydownHandler;

                        console.log('FlowDrop Agents: Editor initialized for', config.workflowId);
                    } catch (error) {
                        console.error('FlowDrop Agents: Failed to initialize editor', error);
                        editorContainer.innerHTML = `
              <div style="padding: 20px; text-align: center; color: #d32f2f;">
                <h3>Failed to Initialize Editor</h3>
                <p>Error: ${error.message}</p>
              </div>
            `;
                    }
                }

                initializeEditor();
            });
        },

        detach: function (context, settings, trigger) {
            const containers = context.querySelectorAll('#flowdrop-agents-editor');
            containers.forEach(function (container) {
                if (container.flowdropApp && typeof container.flowdropApp.destroy === 'function') {
                    container.flowdropApp.destroy();
                    delete container.flowdropApp;
                }
                if (container.keydownHandler) {
                    document.removeEventListener('keydown', container.keydownHandler);
                    delete container.keydownHandler;
                }
                if (container.changeObserver) {
                    container.changeObserver.disconnect();
                    delete container.changeObserver;
                }
                if (container.autoAttachInterval) {
                    clearInterval(container.autoAttachInterval);
                    delete container.autoAttachInterval;
                }
                if (container.dragHighlightStyle) {
                    container.dragHighlightStyle.remove();
                    delete container.dragHighlightStyle;
                }
                if (container.configPanelObserver) {
                    container.configPanelObserver.disconnect();
                    delete container.configPanelObserver;
                }
                if (container.configPanelInterval) {
                    clearInterval(container.configPanelInterval);
                    delete container.configPanelInterval;
                }
                if (container.chatbotStylingInterval) {
                    clearInterval(container.chatbotStylingInterval);
                    delete container.chatbotStylingInterval;
                }
                delete container.dataset.flowdropInitialized;
            });

            // Reset unsaved state.
            hasUnsavedChanges = false;

            if (window.currentFlowDropApp) {
                delete window.currentFlowDropApp;
            }
            if (window.flowdropSave) {
                delete window.flowdropSave;
            }
        },
    };

})(once, Drupal, drupalSettings);
