<?php

declare(strict_types=1);

namespace Drupal\flowdrop_ui_agents\Controller\Api;

use Drupal\ai_agents\Entity\AiAgent;
use Drupal\ai_assistant_api\Entity\AiAssistant;
use Drupal\block\Entity\Block;
use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * API controller for saving AI Assistants via FlowDrop.
 *
 * This mimics the behavior of AiAssistantForm::submitForm() to ensure
 * both the Assistant and its linked Agent are properly updated.
 *
 * Agents are saved in topological order (leaf agents first, then parents)
 * so that parent agents can reference newly created sub-agents.
 */
class AssistantSaveController extends ControllerBase {

  /**
   * Saves an AI Assistant and its linked Agent from FlowDrop workflow data.
   *
   * @param string $assistant_id
   *   The AI Assistant ID.
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object containing workflow JSON.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   JSON response indicating success or failure.
   */
  public function save(string $assistant_id, Request $request): JsonResponse {
    try {
      // Check if this is a new assistant from the request header.
      $isNew = $request->headers->get('X-Modeler-API-isNew') === 'true';

      // Load the assistant.
      $assistant = $this->entityTypeManager()->getStorage('ai_assistant')->load($assistant_id);
      if (!$assistant instanceof AiAssistant) {
        return new JsonResponse([
          'success' => FALSE,
          'error' => 'Assistant not found: ' . $assistant_id,
        ], 404);
      }

      // Get the linked agent.
      // For config entities, get() returns the value directly.
      $agentId = $assistant->get('ai_agent');
      if (!$agentId) {
        return new JsonResponse([
          'success' => FALSE,
          'error' => 'Assistant has no linked agent',
        ], 400);
      }

      $agent = $this->entityTypeManager()->getStorage('ai_agent')->load($agentId);
      if (!$agent instanceof AiAgent) {
        return new JsonResponse([
          'success' => FALSE,
          'error' => 'Linked agent not found: ' . $agentId,
        ], 404);
      }

      // Parse the workflow data from request body.
      $content = $request->getContent();
      $workflowData = json_decode($content, TRUE);

      if (json_last_error() !== JSON_ERROR_NONE) {
        return new JsonResponse([
          'success' => FALSE,
          'error' => 'Invalid JSON: ' . json_last_error_msg(),
        ], 400);
      }

      // Extract data from workflow.
      $metadata = $workflowData['metadata'] ?? [];
      $agentConfig = $metadata['agentConfig'] ?? [];
      $nodes = $workflowData['nodes'] ?? [];
      $edges = $workflowData['edges'] ?? [];

      // Save all sub-agents first in topological order (leaves first).
      // This ensures parent agents can reference newly created sub-agents.
      $savedAgentIds = $this->saveSubAgentsInOrder($nodes, $edges, $agentId);

      // Find the main assistant node and extract its config.
      // The assistant config is stored in the node's data.config.
      // Not in metadata.
      $assistantNodeConfig = $this->findAssistantNodeConfig($nodes, $agentId);

      // Update the main Agent (similar to AiAssistantForm::submitForm).
      $this->updateAgent($agent, $agentConfig, $assistantNodeConfig, $nodes, $edges);

      // Update the Assistant from the node config.
      $this->updateAssistant($assistant, $assistantNodeConfig);

      // Save both entities.
      $agent->save();
      $assistant->save();

      // Save chatbots AFTER assistant (they reference the assistant ID).
      $savedChatbots = $this->saveChatbots($nodes, $assistant_id);

      return new JsonResponse([
        'success' => TRUE,
        'message' => 'Assistant and agent saved successfully',
        'assistant_id' => $assistant->id(),
        'assistant_label' => $assistant->label(),
        'agent_id' => $agent->id(),
        'saved_sub_agents' => $savedAgentIds,
        'saved_chatbots' => $savedChatbots,
        'is_new' => $isNew,
      ]);

    }
    catch (\Exception $e) {
      return new JsonResponse([
        'success' => FALSE,
        'error' => 'Save failed: ' . $e->getMessage(),
      ], 500);
    }
  }

  /**
   * Finds the main assistant node and extracts its config.
   *
   * @param array<string, mixed> $nodes
   *   Array of workflow nodes.
   * @param string $agentId
   *   The agent ID to find.
   *
   * @return array<string, mixed>
   *   The assistant node configuration.
   */
  protected function findAssistantNodeConfig(array $nodes, string $agentId): array {
    $mainNodeId = 'agent_' . $agentId;

    foreach ($nodes as $node) {
      $nodeType = $node['data']['nodeType'] ?? '';
      $nodeId = $node['id'] ?? '';

      // Find the main assistant/agent node.
      if ($nodeId === $mainNodeId || $nodeType === 'assistant') {
        return $node['data']['config'] ?? [];
      }
    }

    return [];
  }

  /**
   * Updates the Agent entity from workflow data.
   *
   * This mirrors the logic in AiAssistantForm::submitForm() lines 537-604.
   *
   * @param \Drupal\ai_agents\Entity\AiAgent $agent
   *   The agent entity to update.
   * @param array<string, mixed> $agentConfig
   *   Agent configuration from metadata.
   * @param array<string, mixed> $assistantNodeConfig
   *   Assistant node configuration.
   * @param array<string, mixed> $nodes
   *   All workflow nodes.
   * @param array<string, mixed> $edges
   *   All workflow edges.
   */
  protected function updateAgent(AiAgent $agent, array $agentConfig, array $assistantNodeConfig, array $nodes, array $edges): void {
    // Update agent fields from agentConfig metadata.
    if (!empty($agentConfig['description'])) {
      $agent->set('description', $agentConfig['description']);
    }
    if (!empty($agentConfig['system_prompt'])) {
      $agent->set('system_prompt', $agentConfig['system_prompt']);
    }

    // Also check assistantNodeConfig for these fields.
    // They may come from node panel.
    if (!empty($assistantNodeConfig['description'])) {
      $agent->set('description', $assistantNodeConfig['description']);
    }
    if (!empty($assistantNodeConfig['systemPrompt'])) {
      $agent->set('system_prompt', $assistantNodeConfig['systemPrompt']);
    }
    if (isset($assistantNodeConfig['maxLoops'])) {
      $agent->set('max_loops', (int) $assistantNodeConfig['maxLoops']);
    }
    elseif (isset($agentConfig['max_loops'])) {
      $agent->set('max_loops', (int) $agentConfig['max_loops']);
    }

    // Build a map of nodes by ID for quick lookup.
    $nodesById = [];
    foreach ($nodes as $node) {
      $nodesById[$node['id']] = $node;
    }

    // Find the main assistant/agent node ID.
    $mainNodeId = 'agent_' . $agent->id();

    // Find direct children of the main node using edges.
    // Only nodes directly connected FROM the main assistant node.
    // Are its tools.
    $directChildNodeIds = [];
    foreach ($edges as $edge) {
      if ($edge['source'] === $mainNodeId) {
        $directChildNodeIds[] = $edge['target'];
      }
    }

    // Extract tools only from direct children of the main assistant node.
    $tools = [];
    $existingTools = $agent->get('tools') ?? [];

    // Load existing settings to preserve data not included in FlowDrop export.
    $existingToolSettings = $agent->get('tool_settings') ?? [];
    $existingToolUsageLimits = $agent->get('tool_usage_limits') ?? [];

    // Separate storage for the two entity fields:
    // - tool_settings: behavioral settings (return_directly, require_usage,
    //   etc.)
    // - tool_usage_limits: property restrictions (force_value, only_allow,
    //   hidden)
    $toolSettings = [];
    $toolUsageLimits = [];

    // SAFETY CHECK: If workflow has only the main node and no edges,
    // preserve existing tools instead of clearing them.
    // This prevents accidental data loss if workflow data is incomplete.
    $hasOnlyMainNode = (count($nodes) === 1 && empty($edges));
    if ($hasOnlyMainNode && !empty($existingTools)) {
      // Preserve existing tools - don't clear them.
      return;
    }

    foreach ($directChildNodeIds as $childNodeId) {
      if (!isset($nodesById[$childNodeId])) {
        continue;
      }

      $node = $nodesById[$childNodeId];
      $nodeType = $node['data']['nodeType'] ?? '';
      $nodeMetadata = $node['data']['metadata'] ?? [];
      $nodeConfig = $node['data']['config'] ?? [];

      if (empty($nodeType) && isset($nodeMetadata['type'])) {
        $nodeType = $nodeMetadata['type'];
      }

      // Handle sub-agent nodes (expanded or collapsed).
      // In expanded mode, sub-agents have nodeType 'agent'.
      // But ownerAgentId differs.
      // In collapsed mode, they have nodeType 'agent-collapsed'.
      if ($nodeType === 'agent' || $nodeType === 'agent-collapsed') {
        // Get the agent ID from various possible locations.
        $subAgentId = $nodeConfig['agent_id']
          ?? $nodeMetadata['ownerAgentId']
          ?? NULL;

        // For expanded agents, extract ID from node ID (agent_xxx).
        if (!$subAgentId && str_starts_with($childNodeId, 'agent_')) {
          $subAgentId = substr($childNodeId, 6);
        }
        // Or from subagent_ prefix.
        if (!$subAgentId && str_starts_with($childNodeId, 'subagent_')) {
          $subAgentId = substr($childNodeId, 9);
        }

        if ($subAgentId && $subAgentId !== $agent->id()) {
          $toolId = 'ai_agents::ai_agent::' . $subAgentId;
          $tools[$toolId] = TRUE;
        }
      }

      // Handle tool nodes.
      if ($nodeType === 'tool') {
        $toolId = $node['data']['toolId'] ?? $nodeConfig['tool_id'] ?? $nodeMetadata['tool_id'] ?? '';
        if ($toolId) {
          $tools[$toolId] = TRUE;

          // Extract behavioral settings for tool_settings.
          $toolSettings[$toolId] = $this->extractBehavioralSettings(
            $nodeConfig,
            $existingToolSettings[$toolId] ?? []
          );

          // Build tool_usage_limits entry from property restrictions.
          $extractedLimits = $this->buildToolUsageLimitsEntry($nodeConfig);

          // Only update if we have property data from the UI.
          // If nodeConfig has no prop_* keys, preserve existing limits.
          $hasPropKeys = FALSE;
          foreach ($nodeConfig as $key => $value) {
            if (str_starts_with($key, 'prop_')) {
              $hasPropKeys = TRUE;
              break;
            }
          }

          if ($hasPropKeys) {
            $toolUsageLimits[$toolId] = $extractedLimits;
          }
          elseif (!empty($existingToolUsageLimits[$toolId])) {
            $toolUsageLimits[$toolId] = $existingToolUsageLimits[$toolId];
          }
        }
      }
    }

    $agent->set('tools', $tools);
    $agent->set('tool_settings', $toolSettings);
    $agent->set('tool_usage_limits', $toolUsageLimits);

    // Assistants always have orchestration_agent = TRUE.
    $agent->set('orchestration_agent', TRUE);
  }

  /**
   * Updates the Assistant entity from node config.
   *
   * Note: The node config uses camelCase keys (from JS), while the entity
   * uses snake_case. This method handles the mapping.
   *
   * @param \Drupal\ai_assistant_api\Entity\AiAssistant $assistant
   *   The assistant entity to update.
   * @param array<string, mixed> $nodeConfig
   *   Node configuration data.
   */
  protected function updateAssistant(AiAssistant $assistant, array $nodeConfig): void {
    // Update assistant-specific fields if provided.
    // Map camelCase (from node config) to snake_case (entity fields).
    if (isset($nodeConfig['label'])) {
      $assistant->set('label', $nodeConfig['label']);
    }
    if (isset($nodeConfig['description'])) {
      $assistant->set('description', $nodeConfig['description']);
    }
    if (isset($nodeConfig['instructions'])) {
      $assistant->set('instructions', $nodeConfig['instructions']);
    }

    // History settings (camelCase from node).
    if (isset($nodeConfig['allowHistory'])) {
      $assistant->set('allow_history', $nodeConfig['allowHistory']);
    }
    if (isset($nodeConfig['historyContextLength'])) {
      $assistant->set('history_context_length', $nodeConfig['historyContextLength']);
    }

    // LLM settings (camelCase from node).
    if (isset($nodeConfig['llmProvider'])) {
      $assistant->set('llm_provider', $nodeConfig['llmProvider']);
    }
    if (isset($nodeConfig['llmModel'])) {
      $assistant->set('llm_model', $nodeConfig['llmModel']);
    }
    if (isset($nodeConfig['llmConfiguration'])) {
      $assistant->set('llm_configuration', $nodeConfig['llmConfiguration']);
    }

    // Other fields.
    if (isset($nodeConfig['errorMessage'])) {
      $assistant->set('error_message', $nodeConfig['errorMessage']);
    }
    if (isset($nodeConfig['roles'])) {
      $assistant->set('roles', $nodeConfig['roles']);
    }

    // Clear old action-based fields (assistants now use agents).
    $assistant->set('pre_action_prompt', '');
    $assistant->set('system_prompt', '');
  }

  /**
   * Extracts tool settings from FlowDrop node config.
   *
   * Converts flat prop_* keys to nested property_restrictions format
   * and extracts other tool settings.
   *
   * @param array<string, mixed> $nodeConfig
   *   The node configuration from FlowDrop.
   * @param string $toolId
   *   The tool ID.
   *
   * @return array<string, mixed>
   *   Tool settings array for saving to tool_settings entity field.
   */
  protected function extractToolSettings(array $nodeConfig, string $toolId): array {
    $settings = [];

    // Extract base tool settings (support both camelCase and snake_case).
    if (isset($nodeConfig['return_directly']) || isset($nodeConfig['returnDirectly'])) {
      $settings['return_directly'] = (bool) ($nodeConfig['return_directly'] ?? $nodeConfig['returnDirectly'] ?? FALSE);
    }
    if (isset($nodeConfig['require_usage']) || isset($nodeConfig['requireUsage'])) {
      $settings['require_usage'] = (bool) ($nodeConfig['require_usage'] ?? $nodeConfig['requireUsage'] ?? FALSE);
    }
    if (isset($nodeConfig['use_artifacts']) || isset($nodeConfig['useArtifacts'])) {
      $settings['use_artifacts'] = (bool) ($nodeConfig['use_artifacts'] ?? $nodeConfig['useArtifacts'] ?? FALSE);
    }
    $descriptionOverrideEnabled = $nodeConfig['description_override_enabled']
      ?? $nodeConfig['descriptionOverrideEnabled']
      ?? NULL;
    if ($descriptionOverrideEnabled !== NULL) {
      $settings['description_override'] = $descriptionOverrideEnabled
        ? ($nodeConfig['description_override'] ?? $nodeConfig['descriptionOverride'] ?? '')
        : '';
    }
    elseif (!empty($nodeConfig['description_override']) || !empty($nodeConfig['descriptionOverride'])) {
      $settings['description_override'] = $nodeConfig['description_override'] ?? $nodeConfig['descriptionOverride'] ?? '';
    }
    if (!empty($nodeConfig['progress_message']) || !empty($nodeConfig['progressMessage'])) {
      $settings['progress_message'] = $nodeConfig['progress_message'] ?? $nodeConfig['progressMessage'] ?? '';
    }

    // Build property_restrictions from flat prop_* keys.
    $restrictions = $this->buildPropertyRestrictions($nodeConfig);
    if (!empty($restrictions)) {
      $settings['property_restrictions'] = $restrictions;
    }

    // Handle RAG-specific settings.
    if ($toolId === 'ai_search:rag_search') {
      if (isset($nodeConfig['index'])) {
        $settings['property_restrictions']['index'] = [
          'action' => 'force_value',
          'force_value' => $nodeConfig['index'],
          'hidden' => TRUE,
        ];
      }
      if (isset($nodeConfig['amount'])) {
        $settings['property_restrictions']['amount'] = [
          'action' => 'force_value',
          'force_value' => $nodeConfig['amount'],
          'hidden' => TRUE,
        ];
      }
      if (isset($nodeConfig['min_score'])) {
        $settings['property_restrictions']['min_score'] = [
          'action' => 'force_value',
          'force_value' => $nodeConfig['min_score'],
          'hidden' => TRUE,
        ];
      }
    }

    return $settings;
  }

  /**
   * Builds property_restrictions from flattened prop_* config keys.
   *
   * Converts FlowDrop's flat schema (prop_X_restriction, prop_X_values, etc.)
   * to Drupal's nested property_restrictions format.
   *
   * @param array<string, mixed> $config
   *   The node config containing flattened prop_* keys.
   *
   * @return array<string, mixed>
   *   Nested property_restrictions array.
   */
  protected function buildPropertyRestrictions(array $config): array {
    // Check for legacy nested format first.
    if (!empty($config['property_restrictions']) && is_array($config['property_restrictions'])) {
      return $config['property_restrictions'];
    }

    $restrictions = [];

    // Find all prop_*_restriction keys.
    foreach ($config as $key => $value) {
      if (!str_starts_with($key, 'prop_') || !str_ends_with($key, '_restriction')) {
        continue;
      }

      // Extract property name: prop_PROPNAME_restriction -> PROPNAME.
      $propName = substr($key, 5, -12);
      if (empty($propName)) {
        continue;
      }

      $restriction = $value ?? '';
      $values = $config['prop_' . $propName . '_values'] ?? '';
      $hidden = !empty($config['prop_' . $propName . '_hidden']);
      $overrideDescEnabled = !empty($config['prop_' . $propName . '_override_desc_enabled']);
      $overrideDesc = $config['prop_' . $propName . '_override_desc'] ?? '';

      // Skip if nothing is set.
      if ($restriction === 'Allow all' && !$hidden && !$overrideDescEnabled) {
        continue;
      }

      $propRestriction = [];

      // Map human-readable restriction values to machine names.
      if ($restriction === 'Force value' || $restriction === 'force_value') {
        $propRestriction['action'] = 'force_value';
        $propRestriction['force_value'] = $values;
      }
      elseif ($restriction === 'Only allow certain values' || $restriction === 'only_allow') {
        $propRestriction['action'] = 'only_allow';
        $propRestriction['allowed_values'] = $values;
      }

      // Hidden checkbox.
      if ($hidden) {
        $propRestriction['hidden'] = TRUE;
      }

      // Description override (only if checkbox enabled).
      if ($overrideDescEnabled && !empty($overrideDesc)) {
        $propRestriction['description_override'] = $overrideDesc;
      }

      if (!empty($propRestriction)) {
        $restrictions[$propName] = $propRestriction;
      }
    }

    return $restrictions;
  }

  /**
   * Extracts tool settings from node config for tool_settings field.
   *
   * This includes both behavioral settings (return_directly, etc.) and
   * property restrictions (including description overrides).
   *
   * @param array<string, mixed> $nodeConfig
   *   The node configuration from FlowDrop.
   * @param array<string, mixed> $existingSettings
   *   Existing tool settings for preserving values omitted by the UI.
   *
   * @return array<string, mixed>
   *   Tool settings array for saving to tool_settings entity field.
   */
  protected function extractBehavioralSettings(array $nodeConfig, array $existingSettings = []): array {
    $settings = [
      'property_restrictions' => [],
    ];

    // Extract base tool settings (support both camelCase and snake_case).
    if (isset($nodeConfig['return_directly']) || isset($nodeConfig['returnDirectly'])) {
      $settings['return_directly'] = (bool) ($nodeConfig['return_directly'] ?? $nodeConfig['returnDirectly'] ?? FALSE);
    }
    if (isset($nodeConfig['require_usage']) || isset($nodeConfig['requireUsage'])) {
      $settings['require_usage'] = (bool) ($nodeConfig['require_usage'] ?? $nodeConfig['requireUsage'] ?? FALSE);
    }
    if (isset($nodeConfig['use_artifacts']) || isset($nodeConfig['useArtifacts'])) {
      $settings['use_artifacts'] = (bool) ($nodeConfig['use_artifacts'] ?? $nodeConfig['useArtifacts'] ?? FALSE);
    }
    $descriptionOverrideEnabled = $nodeConfig['description_override_enabled']
      ?? $nodeConfig['descriptionOverrideEnabled']
      ?? NULL;
    $descriptionOverrideProvided = array_key_exists('description_override', $nodeConfig)
      || array_key_exists('descriptionOverride', $nodeConfig);
    if ($descriptionOverrideEnabled !== NULL) {
      $settings['description_override'] = $descriptionOverrideEnabled
        ? ($nodeConfig['description_override'] ?? $nodeConfig['descriptionOverride'] ?? '')
        : '';
    }
    elseif ($descriptionOverrideProvided) {
      $settings['description_override'] = $nodeConfig['description_override'] ?? $nodeConfig['descriptionOverride'] ?? '';
    }
    elseif (array_key_exists('description_override', $existingSettings)) {
      $settings['description_override'] = $existingSettings['description_override'];
    }
    if (!empty($nodeConfig['progress_message']) || !empty($nodeConfig['progressMessage'])) {
      $settings['progress_message'] = $nodeConfig['progress_message'] ?? $nodeConfig['progressMessage'] ?? '';
    }

    // Extract property description overrides and merge into
    // property_restrictions. This matches the structure used by
    // WorkflowParser and Agent ModelOwner. Stored as:
    // tool_settings[tool_id]['property_restrictions'][property_name]
    // ['description_override'].
    $propertyDescOverrides = $this->extractPropertyDescriptionOverrides($nodeConfig);
    if (!empty($propertyDescOverrides)) {
      // Merge description overrides into property_restrictions.
      foreach ($propertyDescOverrides as $propName => $description) {
        if (!isset($settings['property_restrictions'][$propName])) {
          $settings['property_restrictions'][$propName] = [];
        }
        $settings['property_restrictions'][$propName]['description_override'] = $description;
      }
    }

    // Process extra config keys as property restrictions.
    // This handles tools like ai_search:rag_search that pass raw
    // property values.
    $reservedKeys = [
      'tool_id',
      'label',
      'toolId',
      'nodeType',
      'nodeId',
      'toolDescription',
      'return_directly',
      'require_usage',
      'use_artifacts',
      'description_override',
      'progress_message',
    ];

    foreach ($nodeConfig as $key => $value) {
      // Skip reserved keys, control flags, and prop_* keys.
      if (in_array($key, $reservedKeys, TRUE) ||
          str_starts_with($key, 'prop_') ||
          str_ends_with($key, '_enabled') ||
          str_ends_with($key, '_hidden')) {
        continue;
      }

      // Skip camelCase variants of reserved keys.
      if (in_array($key, ['returnDirectly', 'requireUsage', 'useArtifacts', 'progressMessage', 'descriptionOverride'], TRUE)) {
        continue;
      }

      // Skip if already in settings (avoid overwriting).
      if (isset($settings[$key])) {
        continue;
      }

      // This is a tool-specific property value - add to property_restrictions.
      if (!isset($settings['property_restrictions'][$key])) {
        $settings['property_restrictions'][$key] = [
          'action' => 'force_value',
          'force_value' => $value,
        ];
      }
    }

    return $settings;
  }

  /**
   * Extracts property description overrides from node config.
   *
   * FlowDrop sends: prop_PROPNAME_override_desc_enabled,
   * prop_PROPNAME_override_desc.
   * Drupal stores these inside property_restrictions:
   * tool_settings[tool_id]['property_restrictions'][PROPNAME]['description_override'].
   *
   * This matches the structure used by WorkflowParser and ensures consistency
   * across both save paths (Modeler API and AssistantSaveController).
   *
   * @param array<string, mixed> $nodeConfig
   *   The node configuration from FlowDrop.
   *
   * @return array<string, string>
   *   Property description overrides keyed by property name.
   */
  protected function extractPropertyDescriptionOverrides(array $nodeConfig): array {
    $overrides = [];

    $suffix = '_override_desc_enabled';
    foreach ($nodeConfig as $key => $value) {
      // Look for prop_*_override_desc_enabled keys.
      if (!str_starts_with($key, 'prop_') || !str_ends_with($key, $suffix)) {
        continue;
      }

      // Extract property name: prop_PROPNAME_override_desc_enabled -> PROPNAME.
      $propName = substr($key, 5, -strlen($suffix));
      if (empty($propName)) {
        continue;
      }

      // Only include if the checkbox is enabled.
      if (!empty($value)) {
        $descKey = 'prop_' . $propName . '_override_desc';
        $description = $nodeConfig[$descKey] ?? '';
        if (!empty($description)) {
          $overrides[$propName] = $description;
        }
      }
    }

    return $overrides;
  }

  /**
   * Builds the complete tool_usage_limits entry for a tool.
   *
   * Creates the full structure expected by the ai_agents entity, matching
   * what Drupal's standard AiAgentForm saves. This includes both behavioral
   * settings and property restrictions.
   *
   * @param array<string, mixed> $nodeConfig
   *   The node configuration from FlowDrop.
   *
   * @return array<string, mixed>
   *   Complete tool_usage_limits entry for the tool.
   */
  protected function buildToolUsageLimitsEntry(array $nodeConfig): array {
    $entry = [];

    // Find all properties from prop_* keys and build their limits.
    $propNames = [];
    foreach ($nodeConfig as $key => $value) {
      if (str_starts_with($key, 'prop_') && str_ends_with($key, '_restriction')) {
        $propName = substr($key, 5, -12);
        if (!empty($propName)) {
          $propNames[$propName] = TRUE;
        }
      }
    }

    // Build limit entry for each property.
    foreach (array_keys($propNames) as $propName) {
      $restriction = $nodeConfig['prop_' . $propName . '_restriction'] ?? '';
      $values = $nodeConfig['prop_' . $propName . '_values'] ?? '';
      $hidden = !empty($nodeConfig['prop_' . $propName . '_hidden']);

      $propLimit = [
        'action' => '',
        'hide_property' => $hidden ? 1 : 0,
        'values' => '',
      ];

      // Map human-readable restriction values to machine names.
      // Values must always be an array for Drupal form compatibility.
      if ($restriction === 'Force value' || $restriction === 'force_value') {
        $propLimit['action'] = 'force_value';
        $propLimit['values'] = is_array($values) ? $values : ($values !== '' ? [$values] : []);
      }
      elseif ($restriction === 'Only allow certain values' || $restriction === 'only_allow') {
        $propLimit['action'] = 'only_allow';
        $propLimit['values'] = is_array($values) ? $values : ($values !== '' ? [$values] : []);
      }

      $entry[$propName] = $propLimit;
    }

    return $entry;
  }

  /**
   * Saves all sub-agents in topological order (leaves first, then parents).
   *
   * This ensures that when a parent agent references a sub-agent, the
   * sub-agent already exists and can be properly linked.
   *
   * @param array<int, array<string, mixed>> $nodes
   *   All nodes from the workflow.
   * @param array<int, array<string, mixed>> $edges
   *   All edges from the workflow.
   * @param string $mainAgentId
   *   The main agent ID (to exclude from sub-agent processing).
   *
   * @return array<int, string>
   *   List of agent IDs that were saved.
   */
  protected function saveSubAgentsInOrder(array $nodes, array $edges, string $mainAgentId): array {
    // Build node lookup map.
    $nodesById = [];
    foreach ($nodes as $node) {
      $nodesById[$node['id']] = $node;
    }

    // Build edge map: source -> [targets].
    $childrenMap = [];
    foreach ($edges as $edge) {
      $source = $edge['source'];
      $target = $edge['target'];
      if (!isset($childrenMap[$source])) {
        $childrenMap[$source] = [];
      }
      $childrenMap[$source][] = $target;
    }

    // Find all agent nodes (excluding the main assistant node).
    $agentNodes = [];
    $mainNodeId = 'agent_' . $mainAgentId;

    foreach ($nodes as $node) {
      $nodeType = $node['data']['nodeType'] ?? '';
      $nodeId = $node['id'] ?? '';

      // Skip the main assistant/agent node - it's handled separately.
      if ($nodeId === $mainNodeId || $nodeType === 'assistant') {
        continue;
      }

      // Collect agent nodes (both expanded and collapsed).
      if ($nodeType === 'agent' || $nodeType === 'agent-collapsed') {
        $agentId = $this->extractAgentIdFromNode($node);
        if ($agentId) {
          $agentNodes[$nodeId] = [
            'node' => $node,
            'agentId' => $agentId,
          ];
        }
      }
    }

    // Topological sort: find save order (leaves first).
    $saveOrder = $this->topologicalSortAgents($agentNodes, $childrenMap);

    // Save agents in order.
    $savedAgentIds = [];
    $agentStorage = $this->entityTypeManager()->getStorage('ai_agent');

    foreach ($saveOrder as $nodeId) {
      if (!isset($agentNodes[$nodeId])) {
        continue;
      }

      $agentInfo = $agentNodes[$nodeId];
      $agentId = $agentInfo['agentId'];
      $node = $agentInfo['node'];
      $nodeConfig = $node['data']['config'] ?? [];

      // Load or create the agent.
      $agent = $agentStorage->load($agentId);
      if (!$agent instanceof AiAgent) {
        // Agent doesn't exist or wrong type - skip for now.
        continue;
      }

      // Update agent fields from node config.
      $this->updateSubAgentFromNode($agent, $nodeConfig, $nodesById, $childrenMap, $nodeId);

      // Save the agent.
      $agent->save();
      $savedAgentIds[] = $agentId;
    }

    return $savedAgentIds;
  }

  /**
   * Extracts the agent ID from a node.
   *
   * @param array<string, mixed> $node
   *   The node data.
   *
   * @return string|null
   *   The agent ID or NULL if not found.
   */
  protected function extractAgentIdFromNode(array $node): ?string {
    $nodeId = $node['id'] ?? '';
    $nodeConfig = $node['data']['config'] ?? [];
    $nodeMetadata = $node['data']['metadata'] ?? [];

    // Try various sources for the agent ID.
    $agentId = $nodeConfig['agent_id']
      ?? $nodeMetadata['ownerAgentId']
      ?? NULL;

    // Extract from node ID prefixes.
    if (!$agentId && str_starts_with($nodeId, 'agent_')) {
      $agentId = substr($nodeId, 6);
    }
    if (!$agentId && str_starts_with($nodeId, 'subagent_')) {
      $agentId = substr($nodeId, 9);
    }

    return $agentId;
  }

  /**
   * Performs topological sort on agent nodes (leaves first).
   *
   * @param array<string, array<string, mixed>> $agentNodes
   *   Map of nodeId => agent info.
   * @param array<string, array<int, string>> $childrenMap
   *   Map of nodeId => child node IDs.
   *
   * @return array<int, string>
   *   Node IDs in save order (leaves first).
   */
  protected function topologicalSortAgents(array $agentNodes, array $childrenMap): array {
    $visited = [];
    $result = [];

    // Depth-first search to find leaves first.
    $visit = function (string $nodeId) use (&$visit, &$visited, &$result, $agentNodes, $childrenMap) {
      if (isset($visited[$nodeId])) {
        return;
      }
      $visited[$nodeId] = TRUE;

      // Visit all children first (if they are agent nodes).
      $children = $childrenMap[$nodeId] ?? [];
      foreach ($children as $childId) {
        if (isset($agentNodes[$childId])) {
          $visit($childId);
        }
      }

      // Add this node after its children (post-order).
      if (isset($agentNodes[$nodeId])) {
        $result[] = $nodeId;
      }
    };

    // Start from all agent nodes.
    foreach (array_keys($agentNodes) as $nodeId) {
      $visit($nodeId);
    }

    return $result;
  }

  /**
   * Updates a sub-agent entity from its node config.
   *
   * This updates both display/configuration fields (description, label,
   * systemPrompt, maxLoops) and the agent's tools when edited from the
   * assistant editor.
   *
   * @param \Drupal\ai_agents\Entity\AiAgent $agent
   *   The agent entity to update.
   * @param array<string, mixed> $nodeConfig
   *   The node configuration from FlowDrop.
   * @param array<string, array<string, mixed>> $nodesById
   *   Map of all nodes by ID.
   * @param array<string, array<int, string>> $childrenMap
   *   Map of node ID to child node IDs (from edges).
   * @param string $nodeId
   *   The node ID of this agent.
   */
  protected function updateSubAgentFromNode(AiAgent $agent, array $nodeConfig, array $nodesById, array $childrenMap, string $nodeId): void {
    // Update basic agent fields from node config.
    // Support both camelCase (from JS) and snake_case (from metadata).
    if (!empty($nodeConfig['description'])) {
      $agent->set('description', $nodeConfig['description']);
    }
    if (!empty($nodeConfig['systemPrompt'])) {
      $agent->set('system_prompt', $nodeConfig['systemPrompt']);
    }
    elseif (!empty($nodeConfig['system_prompt'])) {
      $agent->set('system_prompt', $nodeConfig['system_prompt']);
    }
    if (isset($nodeConfig['maxLoops'])) {
      $agent->set('max_loops', (int) $nodeConfig['maxLoops']);
    }
    elseif (isset($nodeConfig['max_loops'])) {
      $agent->set('max_loops', (int) $nodeConfig['max_loops']);
    }
    if (!empty($nodeConfig['label'])) {
      $agent->set('label', $nodeConfig['label']);
    }

    // Update the sub-agent's tools based on its connections in the workflow.
    // Find direct children of this sub-agent node.
    $directChildNodeIds = $childrenMap[$nodeId] ?? [];

    $tools = [];
    // Separate storage for the two entity fields:
    // - tool_settings: behavioral settings (return_directly, require_usage,
    //   etc.)
    // - tool_usage_limits: property restrictions (force_value, only_allow,
    //   hidden)
    $toolSettings = [];
    $toolUsageLimits = [];

    // Load existing settings to preserve data not included in FlowDrop export.
    $existingToolSettings = $agent->get('tool_settings') ?? [];
    $existingToolUsageLimits = $agent->get('tool_usage_limits') ?? [];

    foreach ($directChildNodeIds as $childNodeId) {
      if (!isset($nodesById[$childNodeId])) {
        continue;
      }

      $node = $nodesById[$childNodeId];
      $nodeType = $node['data']['nodeType'] ?? '';
      $nodeMetadata = $node['data']['metadata'] ?? [];
      $childNodeConfig = $node['data']['config'] ?? [];

      if (empty($nodeType) && isset($nodeMetadata['type'])) {
        $nodeType = $nodeMetadata['type'];
      }

      // Handle nested sub-agent nodes (another agent used as a tool).
      if ($nodeType === 'agent' || $nodeType === 'agent-collapsed') {
        $subAgentId = $childNodeConfig['agent_id']
          ?? $nodeMetadata['ownerAgentId']
          ?? NULL;

        if (!$subAgentId && str_starts_with($childNodeId, 'agent_')) {
          $subAgentId = substr($childNodeId, 6);
        }
        if (!$subAgentId && str_starts_with($childNodeId, 'subagent_')) {
          $subAgentId = substr($childNodeId, 9);
        }

        if ($subAgentId && $subAgentId !== $agent->id()) {
          $toolId = 'ai_agents::ai_agent::' . $subAgentId;
          $tools[$toolId] = TRUE;
        }
      }

      // Handle tool nodes.
      if ($nodeType === 'tool') {
        $toolId = $node['data']['toolId'] ?? $childNodeConfig['tool_id'] ?? $nodeMetadata['tool_id'] ?? '';
        if ($toolId) {
          $tools[$toolId] = TRUE;

          // Extract behavioral settings for tool_settings.
          $toolSettings[$toolId] = $this->extractBehavioralSettings(
            $childNodeConfig,
            $existingToolSettings[$toolId] ?? []
          );

          // Build tool_usage_limits entry from property restrictions.
          $extractedLimits = $this->buildToolUsageLimitsEntry($childNodeConfig);

          // Only update if we have property data from the UI.
          $hasPropKeys = FALSE;
          foreach ($childNodeConfig as $key => $value) {
            if (str_starts_with($key, 'prop_')) {
              $hasPropKeys = TRUE;
              break;
            }
          }

          if ($hasPropKeys) {
            $toolUsageLimits[$toolId] = $extractedLimits;
          }
          elseif (!empty($existingToolUsageLimits[$toolId])) {
            $toolUsageLimits[$toolId] = $existingToolUsageLimits[$toolId];
          }
        }
      }
    }

    // Only update tools if the workflow has connections from this sub-agent.
    // If there are no connections, preserve existing tools to avoid data loss.
    // This prevents wiping out a sub-agent's tools when it's referenced in a
    // workflow but its tool configuration isn't being managed in that workflow.
    if (!empty($directChildNodeIds)) {
      $agent->set('tools', $tools);
      $agent->set('tool_settings', $toolSettings);
      $agent->set('tool_usage_limits', $toolUsageLimits);
    }
  }

  /**
   * Saves chatbot blocks linked to an assistant.
   *
   * Chatbots must be saved AFTER the assistant because they reference
   * the assistant ID in their configuration.
   *
   * @param array<int, array<string, mixed>> $nodes
   *   All workflow nodes.
   * @param string $assistantId
   *   The assistant ID to link chatbots to.
   *
   * @return array<int, string>
   *   List of chatbot block IDs that were saved.
   */
  protected function saveChatbots(array $nodes, string $assistantId): array {
    $savedChatbots = [];

    foreach ($nodes as $node) {
      $nodeType = $node['data']['nodeType'] ?? '';
      if ($nodeType !== 'chatbot') {
        continue;
      }

      $nodeConfig = $node['data']['config'] ?? [];
      $blockId = $nodeConfig['blockId'] ?? NULL;

      // Fallback: extract blockId from node ID (format: chatbot_{blockId}).
      if (!$blockId) {
        $nodeId = $node['id'] ?? '';
        if (str_starts_with($nodeId, 'chatbot_')) {
          $blockId = substr($nodeId, 8);
        }
      }

      if (!$blockId) {
        $blockId = $this->generateChatbotBlockId($assistantId, $nodeConfig);
      }

      try {
        $block = $this->loadOrCreateChatbotBlock($blockId, $assistantId, $nodeConfig);
        if ($block) {
          $block->save();
          $savedChatbots[] = $blockId;
        }
      }
      catch (\Exception $e) {
        $this->getLogger('flowdrop_ui_agents')->error(
          'Failed to save chatbot block @id: @error',
          ['@id' => $blockId, '@error' => $e->getMessage()]
        );
      }
    }

    return $savedChatbots;
  }

  /**
   * Generates a unique block ID for a new chatbot.
   *
   * @param string $assistantId
   *   The assistant ID.
   * @param array<string, mixed> $nodeConfig
   *   The node configuration.
   *
   * @return string
   *   A unique block ID.
   */
  protected function generateChatbotBlockId(string $assistantId, array $nodeConfig): string {
    $baseName = $nodeConfig['label'] ?? $nodeConfig['botName'] ?? $assistantId;
    $machineName = preg_replace('/[^a-z0-9_]/', '_', strtolower($baseName));
    $machineName = preg_replace('/_+/', '_', $machineName);
    $machineName = trim($machineName, '_');

    if (strlen($machineName) > 32) {
      $machineName = substr($machineName, 0, 32);
    }

    $blockId = $machineName . '_chatbot';

    $storage = $this->entityTypeManager()->getStorage('block');
    $counter = 1;
    $originalId = $blockId;
    while ($storage->load($blockId)) {
      $blockId = $originalId . '_' . $counter;
      $counter++;
    }

    return $blockId;
  }

  /**
   * Loads an existing chatbot block or creates a new one.
   *
   * @param string $blockId
   *   The block ID.
   * @param string $assistantId
   *   The assistant ID to link to.
   * @param array<string, mixed> $nodeConfig
   *   The node configuration from FlowDrop.
   *
   * @return \Drupal\block\Entity\Block|null
   *   The block entity or NULL on failure.
   */
  protected function loadOrCreateChatbotBlock(string $blockId, string $assistantId, array $nodeConfig): ?Block {
    $storage = $this->entityTypeManager()->getStorage('block');
    $block = $storage->load($blockId);

    $settings = $this->buildChatbotSettings($assistantId, $nodeConfig);
    $theme = $nodeConfig['theme'] ?? 'drupal_cms_olivero';
    $region = $nodeConfig['region'] ?? 'footer_bottom';

    if ($block instanceof Block) {
      $block->set('settings', $settings);
      $block->set('theme', $theme);
      $block->set('region', $region);
      $this->applyChatbotVisibility($block, $nodeConfig);
    }
    else {
      $block = Block::create([
        'id' => $blockId,
        'plugin' => 'ai_deepchat_block',
        'theme' => $theme,
        'region' => $region,
        'weight' => 0,
        'status' => TRUE,
        'settings' => $settings,
      ]);
      $this->applyChatbotVisibility($block, $nodeConfig);
    }

    return $block;
  }

  /**
   * Builds the settings array for a chatbot block.
   *
   * Maps FlowDrop node config (camelCase) to block settings (snake_case).
   *
   * @param string $assistantId
   *   The assistant ID.
   * @param array<string, mixed> $nodeConfig
   *   The node configuration.
   *
   * @return array<string, mixed>
   *   Block settings array.
   */
  protected function buildChatbotSettings(string $assistantId, array $nodeConfig): array {
    return [
      'id' => 'ai_deepchat_block',
      'label' => $nodeConfig['label'] ?? $nodeConfig['botName'] ?? 'Chatbot',
      'label_display' => 'visible',
      'provider' => 'ai_chatbot',
      'ai_assistant' => $assistantId,
      'first_message' => $nodeConfig['firstMessage'] ?? '',
      'bot_name' => $nodeConfig['botName'] ?? 'AI Assistant',
      'bot_image' => $nodeConfig['botImage'] ?? '',
      'default_username' => $nodeConfig['defaultUsername'] ?? '',
      'use_username' => (bool) ($nodeConfig['useUsername'] ?? FALSE),
      'default_avatar' => $nodeConfig['defaultAvatar'] ?? '',
      'use_avatar' => (bool) ($nodeConfig['useAvatar'] ?? FALSE),
      'style_file' => $nodeConfig['styleFile'] ?? 'module:ai_chatbot:bard.yml',
      'width' => $nodeConfig['width'] ?? '500px',
      'height' => $nodeConfig['height'] ?? '500px',
      'placement' => $nodeConfig['placement'] ?? 'bottom-right',
      'collapse_minimal' => (bool) ($nodeConfig['collapseMinimal'] ?? FALSE),
      'show_copy_icon' => (bool) ($nodeConfig['showCopyIcon'] ?? TRUE),
      'toggle_state' => $nodeConfig['toggleState'] ?? 'remember',
      'verbose_mode' => (bool) ($nodeConfig['verboseMode'] ?? FALSE),
    ];
  }

  /**
   * Applies visibility conditions to a chatbot block.
   *
   * @param \Drupal\block\Entity\Block $block
   *   The block entity.
   * @param array<string, mixed> $nodeConfig
   *   The node configuration.
   */
  protected function applyChatbotVisibility(Block $block, array $nodeConfig): void {
    $visibility = [];

    if (!empty($nodeConfig['visibilityPages'])) {
      $visibility['request_path'] = [
        'id' => 'request_path',
        'negate' => (bool) ($nodeConfig['visibilityPagesNegate'] ?? FALSE),
        'pages' => $nodeConfig['visibilityPages'],
      ];
    }

    $responseStatus = $nodeConfig['visibilityResponseStatus'] ?? [];
    if (!empty($responseStatus) && is_array($responseStatus)) {
      $visibility['response_status'] = [
        'id' => 'response_status',
        'status_codes' => array_combine($responseStatus, $responseStatus),
      ];
    }

    $roles = $nodeConfig['visibilityRoles'] ?? [];
    if (!empty($roles) && is_array($roles)) {
      $visibility['user_role'] = [
        'id' => 'user_role',
        'negate' => (bool) ($nodeConfig['visibilityRolesNegate'] ?? FALSE),
        'roles' => array_combine($roles, $roles),
      ];
    }

    $contentTypes = $nodeConfig['visibilityContentTypes'] ?? [];
    if (!empty($contentTypes) && is_array($contentTypes)) {
      $visibility['entity_bundle:node'] = [
        'id' => 'entity_bundle:node',
        'negate' => (bool) ($nodeConfig['visibilityContentTypesNegate'] ?? FALSE),
        'bundles' => array_combine($contentTypes, $contentTypes),
      ];
    }

    $vocabularies = $nodeConfig['visibilityVocabularies'] ?? [];
    if (!empty($vocabularies) && is_array($vocabularies)) {
      $visibility['entity_bundle:taxonomy_term'] = [
        'id' => 'entity_bundle:taxonomy_term',
        'negate' => (bool) ($nodeConfig['visibilityVocabulariesNegate'] ?? FALSE),
        'bundles' => array_combine($vocabularies, $vocabularies),
      ];
    }

    $block->setVisibilityConfig('request_path', $visibility['request_path'] ?? []);
    $block->setVisibilityConfig('response_status', $visibility['response_status'] ?? []);
    $block->setVisibilityConfig('user_role', $visibility['user_role'] ?? []);
    if (isset($visibility['entity_bundle:node'])) {
      $block->setVisibilityConfig('entity_bundle:node', $visibility['entity_bundle:node']);
    }
    if (isset($visibility['entity_bundle:taxonomy_term'])) {
      $block->setVisibilityConfig('entity_bundle:taxonomy_term', $visibility['entity_bundle:taxonomy_term']);
    }
  }

}
