<?php

declare(strict_types=1);

namespace Drupal\flowdrop_ui_agents\Controller\Api;

use Drupal\Core\Controller\ControllerBase;
use Drupal\flowdrop_ui_agents\Service\AgentWorkflowMapper;
use Drupal\modeler_api\Plugin\ModelerApiModelOwner\ModelOwnerInterface;
use Drupal\modeler_api\Plugin\ModelOwnerPluginManager;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;

/**
 * API controller for FlowDrop node types (tools and agents).
 *
 * Provides endpoints for the FlowDrop sidebar to fetch available node types.
 *
 * Category Type System:
 * ---------------------
 * Categories are grouped into "category types" which control sort order.
 * Within each type, categories sort alphabetically. Types are ordered by
 * weight.
 *
 * Example hierarchy:
 *   Type: agents (weight -100) → "Sub-Agent Tools", "Assistants"
 *   Type: chatbots (weight -90) → "DeepChat Bots"
 *   Type: search (weight -80)  → "RAG Indexes", "Search API"
 *   Type: tools (weight 0)     → "Entity Tools", "Information Tools", etc.
 *   Type: other (weight 100)   → "Other" (catch-all)
 */
class NodesController extends ControllerBase {

  /**
   * Category type weights - controls order of category groups in sidebar.
   *
   * Lower weight = appears first. Categories within same type sort
   * alphabetically.
   */
  protected const CATEGORY_TYPE_WEIGHTS = [
  // AI Agents, Sub-Agents, Assistants.
    'agents' => -100,
  // Chatbot integrations (DeepChat, etc.)
    'chatbots' => -90,
  // RAG, Search API, Knowledge retrieval.
    'search' => -80,
  // Model Context Protocol tools.
    'mcp' => -70,
  // Standard tool categories (from FunctionGroup)
    'tools' => 0,
  // Catch-all for uncategorized.
    'other' => 100,
  ];

  /**
   * Maps category names to their category type.
   *
   * Categories not listed here default to 'tools' type.
   * Use lowercase keys for case-insensitive matching.
   */
  protected const CATEGORY_TYPE_MAP = [
    // Agents type - FlowDrop expects 'agents' category for built-in styling.
    // JS renames the sidebar header to "Sub-Agent Tools" for display.
    'agents' => 'agents',
    'sub-agent tools' => 'agents',
    'assistants' => 'agents',
    'orchestrators' => 'agents',
    // Chatbots type.
    'chatbots' => 'chatbots',
    'deepchat' => 'chatbots',
    // Search type.
    'rag indexes' => 'search',
    'search api' => 'search',
    'knowledge retrieval' => 'search',
    // MCP type.
    'mcp tools' => 'mcp',
    'model context protocol' => 'mcp',
    // Other type (explicit catch-all)
    'other' => 'other',
    'uncategorized' => 'other',
  ];

  /**
   * The agent workflow mapper service.
   */
  protected AgentWorkflowMapper $agentWorkflowMapper;

  /**
   * The model owner plugin manager.
   */
  protected ModelOwnerPluginManager $modelOwnerManager;

  /**
   * Constructs the controller.
   */
  public function __construct(
    AgentWorkflowMapper $agentWorkflowMapper,
    ModelOwnerPluginManager $modelOwnerManager,
  ) {
    $this->agentWorkflowMapper = $agentWorkflowMapper;
    $this->modelOwnerManager = $modelOwnerManager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('flowdrop_ui_agents.agent_workflow_mapper'),
      $container->get('plugin.manager.modeler_api.model_owner')
    );
  }

  /**
   * Gets all available nodes (tools and agents).
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   JSON response with all available nodes.
   */
  public function getNodes(): JsonResponse {
    $owner = $this->getModelOwner();

    $tools = $this->agentWorkflowMapper->getAvailableTools($owner);
    $agents = $this->agentWorkflowMapper->getAvailableAgents($owner);
    $chatbots = $this->agentWorkflowMapper->getAvailableChatbots();

    $nodes = array_merge($tools, $agents, $chatbots);

    foreach ($nodes as &$node) {
      $categoryName = $node['category'] ?? 'Other';
      $node['categoryWeight'] = $this->getCategoryWeight($categoryName);
    }
    unset($node);

    return new JsonResponse([
      'success' => TRUE,
      'data' => $nodes,
      'count' => count($nodes),
      'message' => sprintf('Found %d nodes', count($nodes)),
    ]);
  }

  /**
   * Gets nodes grouped by category.
   *
   * Categories are sorted by their type weight first, then alphabetically
   * within each type. The response includes categoryMeta with weights so
   * the frontend can maintain consistent ordering.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   JSON response with nodes grouped by category.
   */
  public function getNodesByCategory(): JsonResponse {
    $owner = $this->getModelOwner();

    $toolsByCategory = $this->agentWorkflowMapper->getToolsByCategory($owner);
    $agents = $this->agentWorkflowMapper->getAvailableAgents($owner);
    $chatbots = $this->agentWorkflowMapper->getAvailableChatbots();

    if (!empty($agents)) {
      $toolsByCategory['Sub-Agent Tools'] = $agents;
    }

    // Always show Chatbots category (even when empty) for future "Add New"
    // button.
    $toolsByCategory['Chatbots'] = $chatbots;

    // Add categoryWeight to each node for frontend sorting.
    // This allows the sidebar to sort categories by weight without needing
    // separate categoryMeta lookup.
    foreach ($toolsByCategory as $categoryName => &$nodes) {
      $weight = $this->getCategoryWeight($categoryName);
      foreach ($nodes as &$node) {
        $node['categoryWeight'] = $weight;
      }
    }
    // Break references.
    unset($nodes, $node);

    // Sort categories by type weight, then alphabetically within type.
    uksort($toolsByCategory, function ($a, $b) {
      $weightA = $this->getCategoryWeight($a);
      $weightB = $this->getCategoryWeight($b);

      if ($weightA !== $weightB) {
        return $weightA <=> $weightB;
      }

      // Same type weight - sort alphabetically.
      return strcasecmp($a, $b);
    });

    // Build category metadata with weights for frontend sorting.
    $categoryMeta = [];
    foreach (array_keys($toolsByCategory) as $categoryName) {
      $categoryMeta[$categoryName] = [
        'weight' => $this->getCategoryWeight($categoryName),
        'type' => $this->getCategoryType($categoryName),
      ];
    }

    $categories = array_keys($toolsByCategory);

    return new JsonResponse([
      'success' => TRUE,
      'data' => $toolsByCategory,
      'categories' => $categories,
      'categoryMeta' => $categoryMeta,
      'message' => sprintf('Found %d categories', count($categories)),
    ]);
  }

  /**
   * Gets nodes for a specific category.
   *
   * @param string $category
   *   The category name.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   JSON response with nodes in the category.
   */
  public function getNodesForCategory(string $category): JsonResponse {
    $owner = $this->getModelOwner();

    // Get all categorized tools.
    $toolsByCategory = $this->agentWorkflowMapper->getToolsByCategory($owner);

    // Handle agents category specially.
    if ($category === 'agents' || $category === 'Sub-Agent Tools') {
      $nodes = $this->agentWorkflowMapper->getAvailableAgents($owner);
    }
    else {
      $nodes = $toolsByCategory[$category] ?? [];
    }

    return new JsonResponse([
      'success' => TRUE,
      'data' => $nodes,
      'count' => count($nodes),
      'category' => $category,
    ]);
  }

  /**
   * Gets metadata for a specific node.
   *
   * @param string $plugin_id
   *   The plugin ID.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   JSON response with node metadata.
   */
  public function getNodeMetadata(string $plugin_id): JsonResponse {
    $owner = $this->getModelOwner();

    // Search in tools first.
    $tools = $this->agentWorkflowMapper->getAvailableTools($owner);
    foreach ($tools as $tool) {
      if ($tool['id'] === $plugin_id || $tool['tool_id'] === $plugin_id) {
        return new JsonResponse([
          'success' => TRUE,
          'data' => $tool,
        ]);
      }
    }

    // Search in agents.
    $agents = $this->agentWorkflowMapper->getAvailableAgents($owner);
    foreach ($agents as $agent) {
      if ($agent['id'] === $plugin_id || ($agent['agent_id'] ?? '') === $plugin_id) {
        return new JsonResponse([
          'success' => TRUE,
          'data' => $agent,
        ]);
      }
    }

    return new JsonResponse([
      'success' => FALSE,
      'error' => sprintf('Node "%s" not found', $plugin_id),
    ], 404);
  }

  /**
   * Gets port configuration for the editor.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   JSON response with port configuration.
   */
  public function getPortConfiguration(): JsonResponse {
    // Define data types that can connect to each other.
    return new JsonResponse([
      'success' => TRUE,
      'data' => [
        'dataTypes' => [
          'trigger' => [
            'name' => 'Trigger',
            'color' => '#ef4444',
            'canConnectTo' => ['trigger'],
          ],
          'string' => [
            'name' => 'String',
            'color' => '#3b82f6',
            'canConnectTo' => ['string', 'mixed', 'any'],
          ],
          'number' => [
            'name' => 'Number',
            'color' => '#22c55e',
            'canConnectTo' => ['number', 'mixed', 'any'],
          ],
          'boolean' => [
            'name' => 'Boolean',
            'color' => '#f59e0b',
            'canConnectTo' => ['boolean', 'mixed', 'any'],
          ],
          'tool' => [
            'name' => 'Tool',
            'color' => '#f97316',
            'canConnectTo' => ['tool'],
          ],
          'agent' => [
            'name' => 'Agent',
            'color' => '#8b5cf6',
            'canConnectTo' => ['agent', 'tool'],
          ],
          'mixed' => [
            'name' => 'Mixed',
            'color' => '#6b7280',
            'canConnectTo' => ['string', 'number', 'boolean', 'mixed', 'any'],
          ],
          'any' => [
            'name' => 'Any',
            'color' => '#6b7280',
            'canConnectTo' => ['string', 'number', 'boolean', 'mixed', 'any', 'tool', 'agent'],
          ],
        ],
      ],
    ]);
  }

  /**
   * Gets the category type for a category name.
   *
   * Category types group related categories together for sorting purposes.
   * Unknown categories default to 'tools' type.
   *
   * @param string $categoryName
   *   The category name (e.g., "Sub-Agent Tools", "Entity Tools").
   *
   * @return string
   *   The category type (e.g., "agents", "tools", "search").
   */
  protected function getCategoryType(string $categoryName): string {
    $normalized = strtolower(trim($categoryName));

    // Check explicit mapping first.
    if (isset(self::CATEGORY_TYPE_MAP[$normalized])) {
      return self::CATEGORY_TYPE_MAP[$normalized];
    }

    // Fallback: check if name contains type keywords.
    if (str_contains($normalized, 'agent') || str_contains($normalized, 'assistant')) {
      return 'agents';
    }
    if (str_contains($normalized, 'chatbot') || str_contains($normalized, 'deepchat')) {
      return 'chatbots';
    }
    if (str_contains($normalized, 'search') || str_contains($normalized, 'rag') || str_contains($normalized, 'index')) {
      return 'search';
    }
    if (str_contains($normalized, 'mcp') || str_contains($normalized, 'context protocol')) {
      return 'mcp';
    }
    if ($normalized === 'other' || $normalized === 'uncategorized') {
      return 'other';
    }

    // Default: standard tools category.
    return 'tools';
  }

  /**
   * Gets the sort weight for a category based on its type.
   *
   * @param string $categoryName
   *   The category name.
   *
   * @return int
   *   The sort weight (lower = appears first).
   */
  protected function getCategoryWeight(string $categoryName): int {
    $type = $this->getCategoryType($categoryName);
    return self::CATEGORY_TYPE_WEIGHTS[$type] ?? 0;
  }

  /**
   * Gets a dummy model owner for the API calls.
   *
   * The AgentWorkflowMapper methods need a ModelOwnerInterface, but for
   * the API we don't have a specific entity context. We create a minimal
   * implementation that satisfies the interface.
   *
   * @return \Drupal\modeler_api\Plugin\ModelerApiModelOwner\ModelOwnerInterface
   *   A model owner instance.
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   */
  protected function getModelOwner(): ModelOwnerInterface {
    /** @var \Drupal\modeler_api\Plugin\ModelerApiModelOwner\ModelOwnerInterface $instance */
    $instance = $this->modelOwnerManager->createInstance('ai_agents_agent');
    return $instance;
  }

}
