<?php

declare(strict_types=1);

namespace Drupal\flowdrop_ui_agents\Service;

use Drupal\ai\Service\FunctionCalling\FunctionGroupPluginManager;
use Drupal\ai\Service\FunctionCalling\FunctionCallPluginManager;
use Drupal\ai_agents\Entity\AiAgent;
use Drupal\Component\Plugin\ConfigurableInterface;
use Drupal\Component\Plugin\PluginInspectionInterface;
use Drupal\Core\Config\Entity\ConfigEntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormState;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Plugin\PluginFormInterface;
use Drupal\modeler_api\Plugin\ModelerApiModelOwner\ModelOwnerInterface;

/**
 * Service to map between AI Agent entities and FlowDrop workflow format.
 *
 * This service provides bidirectional conversion:
 * - LOAD: AI Agent config entity -> FlowDrop workflow (nodes/edges).
 * - SAVE: FlowDrop workflow -> AI Agent config updates.
 */
class AgentWorkflowMapper {

  /**
   * Category to plural form mapping.
   */
  protected const CATEGORY_PLURAL_MAP = [
    'trigger' => 'triggers',
    'input' => 'inputs',
    'output' => 'outputs',
    'model' => 'models',
    'prompt' => 'prompts',
    'processing' => 'processing',
    'logic' => 'logic',
    'data' => 'data',
    'helper' => 'helpers',
    'tool' => 'tools',
    'vectorstore' => 'Vector Stores',
    'embedding' => 'Embeddings',
    'memory' => 'Memories',
    'agent' => 'Agents',
    'agents' => 'Agents',
    'bundle' => 'Bundles',
    'content' => 'Content',
    'search' => 'Search',
    'user' => 'User',
    'drupal_actions' => 'Drupal Actions',
    'General' => 'Tools',
  ];

  /**
   * Category color mapping.
   */
  protected const CATEGORY_COLORS = [
    'trigger' => 'var(--color-ref-red-500)',
    'input' => 'var(--color-ref-blue-500)',
    'output' => 'var(--color-ref-green-500)',
    'model' => 'var(--color-ref-purple-500)',
    'tool' => 'var(--color-ref-orange-500)',
    'agent' => 'var(--color-ref-teal-500)',
    'Sub-Agent Tools' => 'var(--color-ref-teal-500)',
    'chatbot' => 'var(--color-ref-purple-500)',
    'Chatbots' => 'var(--color-ref-purple-500)',
    'General' => 'var(--color-ref-gray-500)',
    'Search' => 'var(--color-ref-blue-500)',
    'Content' => 'var(--color-ref-green-500)',
    'User' => 'var(--color-ref-purple-500)',
    'Custom' => 'var(--color-ref-orange-500)',
  ];

  /**
   * Constructs the AgentWorkflowMapper service.
   */
  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected FunctionCallPluginManager $functionCallPluginManager,
    protected LoggerChannelFactoryInterface $loggerFactory,
    protected FunctionGroupPluginManager $functionGroupPluginManager,
  ) {
  }

  /**
   * Layout constants for node positioning.
   */
  protected const LAYOUT = [
    // Main agent starts at left.
    'agentX' => 100,
    // Vertical center-ish.
    'agentY' => 250,
    // Tools are 350px to the right of their agent.
    'toolOffsetX' => 350,
    // Vertical spacing between tools (must exceed node height ~150px).
    'toolSpacingY' => 180,
    // Sub-agents further right.
    'subAgentOffsetX' => 700,
    // Spacing between sub-agents.
    'subAgentSpacingY' => 500,
    // Approximate node width for calculations.
    'nodeWidth' => 250,
    // Minimum gap between sibling agents.
    'agentGap' => 80,
  ];

  /**
   * Estimates the height of an agent's workflow representation.
   *
   * This is used to calculate vertical offsets for sibling sub-agents
   * to prevent overlap in expanded mode.
   *
   * @param \Drupal\ai_agents\Entity\AiAgent $agent
   *   The agent to estimate.
   * @param int $depth
   *   Current depth.
   *
   * @return int
   *   Estimated height in pixels.
   */
  protected function estimateAgentHeight(AiAgent $agent, int $depth = 0): int {
    if ($depth > 3) {
      // Minimal height.
      return self::LAYOUT['agentY'];
    }

    $tools = $agent->get('tools') ?? [];
    $subAgents = [];
    $regularToolCount = 0;

    foreach ($tools as $toolId => $enabled) {
      if (!$enabled) {
        continue;
      }
      if (str_starts_with($toolId, 'ai_agents::ai_agent::')) {
        $subAgents[] = str_replace('ai_agents::ai_agent::', '', $toolId);
      }
      else {
        $regularToolCount++;
      }
    }

    // Height from tools (dynamic columns).
    $cols = max(2, ceil(sqrt($regularToolCount)));
    $toolRows = ceil($regularToolCount / $cols);
    $toolsHeight = max(200, $toolRows * self::LAYOUT['toolSpacingY']);

    // Height from sub-agents.
    $subAgentsHeight = 0;
    foreach ($subAgents as $subAgentId) {
      $subAgent = $this->entityTypeManager->getStorage('ai_agent')->load($subAgentId);
      if ($subAgent instanceof AiAgent) {
        $subAgentsHeight += $this->estimateAgentHeight($subAgent, $depth + 1);
      }
      else {
        // Fallback.
        $subAgentsHeight += 200;
      }
    }

    // Add spacing between sub-agents.
    if (count($subAgents) > 1) {
      $subAgentsHeight += (count($subAgents) - 1) * self::LAYOUT['agentGap'];
    }

    // Minimum 400px.
    return (int) max($toolsHeight, $subAgentsHeight, 400);
  }

  /**
   * Converts an AI Agent entity to FlowDrop workflow format.
   *
   * @param \Drupal\Core\Config\Entity\ConfigEntityInterface $agent
   *   The AI Agent entity.
   * @param string $expansionMode
   *   How to display sub-agents: 'expanded', 'grouped', or 'collapsed'.
   * @param int $depth
   *   Current recursion depth (internal use).
   * @param int $maxDepth
   *   Maximum recursion depth to prevent infinite loops.
   * @param int $offsetY
   *   Vertical offset for this agent's subtree.
   *
   * @return array<string, mixed>
   *   FlowDrop workflow data structure.
   */
  public function agentToWorkflow(ConfigEntityInterface $agent, string $expansionMode = 'expanded', int $depth = 0, int $maxDepth = 3, int $offsetY = 0): array {
    assert($agent instanceof AiAgent);

    $nodes = [];
    $edges = [];
    $savedPositions = $this->loadPositions($agent);

    // Calculate base X position based on depth.
    // Main agent at left, sub-agents to right.
    $baseX = self::LAYOUT['agentX'] + ($depth * self::LAYOUT['subAgentOffsetX']);
    $baseY = self::LAYOUT['agentY'] + $offsetY;

    // Create the main agent node.
    $agentNodeId = 'agent_' . $agent->id();
    $agentNode = $this->createAgentNode($agent, $agentNodeId);
    $agentNode['data']['metadata']['ownerAgentId'] = $agent->id();

    // Use saved position or calculate default.
    $agentNode['position'] = $savedPositions[$agentNodeId] ?? ['x' => $baseX, 'y' => $baseY];
    $nodes[] = $agentNode;

    // Get tools and create tool nodes.
    $tools = $agent->get('tools') ?? [];

    // tool_settings: behavioral settings (return_directly, require_usage,
    // etc.).
    // tool_usage_limits: property restrictions (force_value, only_allow,
    // hidden).
    $toolSettings = $agent->get('tool_settings') ?? [];
    $toolUsageLimits = $agent->get('tool_usage_limits') ?? [];

    // Combine both into the format expected by createToolNode().
    $combinedSettings = $this->combineToolSettingsAndLimits($toolSettings, $toolUsageLimits);

    // Separate regular tools from sub-agents for positioning.
    $regularTools = [];
    $subAgents = [];

    foreach ($tools as $toolId => $enabled) {
      if (!$enabled) {
        continue;
      }
      if (str_starts_with($toolId, 'ai_agents::ai_agent::')) {
        $subAgents[] = $toolId;
      }
      else {
        $regularTools[] = $toolId;
      }
    }

    // Position regular tools to the right of the agent, in a dynamic grid.
    $toolX = $baseX + self::LAYOUT['toolOffsetX'];
    $regularToolCount = count($regularTools);

    // Dynamic columns: Square root approximation (e.g. 9 -> 3x3, 4 -> 2x2).
    // Min 2 cols.
    $cols = max(2, ceil(sqrt($regularToolCount)));
    $rows = ceil($regularToolCount / $cols);

    // Calculate total grid height to center it vertically relative to
    // agent CENTER. Agent is approx 100px height, so center is +50.
    // Added extra +50 offset to push tools slightly down (prevent
    // top-heavy look).
    $gridHeight = ($rows - 1) * self::LAYOUT['toolSpacingY'];
    $startToolY = ($baseY + 100) - ($gridHeight / 2);

    $toolIndex = 0;
    foreach ($regularTools as $toolId) {
      $col = $toolIndex % $cols;
      $row = floor($toolIndex / $cols);

      // 300px column width
      $currentToolX = $toolX + ($col * 300);
      $currentToolY = $startToolY + ($row * self::LAYOUT['toolSpacingY']);

      $toolNodeId = 'tool_' . $agent->id() . '_' . $toolIndex;
      $toolPosition = $savedPositions[$toolNodeId] ?? ['x' => $currentToolX, 'y' => $currentToolY];

      $toolNode = $this->createToolNode($toolId, $toolNodeId, $combinedSettings[$toolId] ?? [], $toolPosition, $toolIndex);

      if ($toolNode) {
        $toolNode['data']['metadata']['ownerAgentId'] = $agent->id();
        $nodes[] = $toolNode;

        // Create edge from agent to tool.
        $edges[] = [
          'id' => "edge_{$agentNodeId}_to_{$toolNodeId}",
          'source' => $agentNodeId,
          'target' => $toolNodeId,
          'sourceHandle' => "{$agentNodeId}-output-tools",
          'targetHandle' => "{$toolNodeId}-input-tool",
          'type' => 'default',
          'data' => ['dataType' => 'tool'],
        ];
      }

      $toolIndex++;
    }

    // Position sub-agents further to the right.
    // Calculate total height of all sub-agents to center them.
    $totalSubTreeHeight = 0;
    // Cache heights.
    $subAgentHeights = [];

    foreach ($subAgents as $toolId) {
      $subAgentId = str_replace('ai_agents::ai_agent::', '', $toolId);
      $subAgent = $this->entityTypeManager->getStorage('ai_agent')->load($subAgentId);
      if ($subAgent instanceof AiAgent) {
        $h = $this->estimateAgentHeight($subAgent, $depth + 1);
        $subAgentHeights[$subAgentId] = $h;
        $totalSubTreeHeight += $h;
      }
      else {
        $subAgentHeights[$subAgentId] = 200;
        $totalSubTreeHeight += 200;
      }
    }

    // Add spacing.
    if (count($subAgents) > 1) {
      // Spacing constant.
      $totalSubTreeHeight += (count($subAgents) - 1) * 100;
    }

    // Start centering.
    $currentSubAgentY = $baseY - ($totalSubTreeHeight / 2);

    $subAgentIndex = 0;
    foreach ($subAgents as $toolId) {
      $subAgentId = str_replace('ai_agents::ai_agent::', '', $toolId);
      $thisHeight = $subAgentHeights[$subAgentId];

      // Center the child in its allocated slot.
      $childCenterY = $currentSubAgentY + ($thisHeight / 2);

      // The recursive function expects an OFFSET relative to
      // LAYOUT['agentY'], not absolute Y.
      // Child Base Y = LAYOUT['agentY'] + $childOffsetY
      // We want Child Base Y = $childCenterY
      // So $childOffsetY = $childCenterY - self::LAYOUT['agentY'].
      $childOffsetY = $childCenterY - self::LAYOUT['agentY'];

      if ($expansionMode === 'collapsed' || $depth >= $maxDepth) {
        // Collapsed mode: show as single node with badge.
        $subAgentNodeId = 'subagent_' . $subAgentId;

        $subAgentPosition = $savedPositions[$subAgentNodeId] ?? [
          'x' => $baseX + self::LAYOUT['subAgentOffsetX'],
          'y' => $childCenterY,
        ];

        $collapsedNode = $this->createCollapsedAgentNode($subAgentId, $subAgentNodeId, $subAgentPosition, $subAgentIndex);
        if ($collapsedNode) {
          $collapsedNode['data']['metadata']['ownerAgentId'] = $agent->id();
          $nodes[] = $collapsedNode;

          // Create edge from parent agent to collapsed sub-agent.
          $edges[] = [
            'id' => "edge_{$agentNodeId}_to_{$subAgentNodeId}",
            'source' => $agentNodeId,
            'target' => $subAgentNodeId,
            'sourceHandle' => "{$agentNodeId}-output-tools",
            'targetHandle' => "{$subAgentNodeId}-input-tool",
            'type' => 'default',
            'data' => ['dataType' => 'agent'],
          ];
        }

      }
      elseif ($expansionMode === 'grouped') {
        // Grouped mode: pass offset to loadSubAgentGrouped.
        // NOTE: Grouped positioning logic needs similar update if we want
        // it perfect, but focusing on Expanded mode for now as it's the
        // main complexity. For grouped, we can just pass the generic offset.
        $subAgentData = $this->loadSubAgentGrouped($subAgentId, $agentNodeId, $depth + 1, $maxDepth, $savedPositions);
        $nodes = array_merge($nodes, $subAgentData['nodes']);
        $edges = array_merge($edges, $subAgentData['edges']);

      }
      else {
        // Expanded mode: recursively load sub-agent.
        $subAgentData = $this->loadSubAgentExpanded($subAgentId, $agentNodeId, $expansionMode, $depth + 1, $maxDepth, $savedPositions, (int) $childOffsetY);
        $nodes = array_merge($nodes, $subAgentData['nodes']);
        $edges = array_merge($edges, $subAgentData['edges']);
      }

      // Advance Y position for next sibling.
      $currentSubAgentY += $thisHeight + self::LAYOUT['agentGap'];
      $subAgentIndex++;
    }

    return [
      'id' => $agent->id(),
      'label' => $agent->label(),
      'nodes' => $nodes,
      'edges' => $edges,
      'metadata' => [
        'agentConfig' => [
          'system_prompt' => $agent->get('system_prompt') ?? '',
          'description' => $agent->get('description') ?? '',
          'max_loops' => $agent->get('max_loops') ?? 3,
          'orchestration_agent' => $agent->get('orchestration_agent') ?? FALSE,
          'triage_agent' => $agent->get('triage_agent') ?? FALSE,
        ],
        'expansionMode' => $expansionMode,
      ],
    ];
  }

  /**
   * Loads a sub-agent in expanded mode (flat hierarchy).
   *
   * @param string $subAgentId
   *   The sub-agent ID.
   * @param string $parentNodeId
   *   The parent node ID.
   * @param string $expansionMode
   *   Expansion mode.
   * @param int $depth
   *   Current depth.
   * @param int $maxDepth
   *   Maximum depth.
   * @param array<string, mixed> $positions
   *   Node positions.
   * @param int $offsetY
   *   Vertical offset.
   *
   * @return array<string, mixed>
   *   Nodes and edges array.
   */
  protected function loadSubAgentExpanded(string $subAgentId, string $parentNodeId, string $expansionMode, int $depth, int $maxDepth, array $positions, int $offsetY = 0): array {
    $storage = $this->entityTypeManager->getStorage('ai_agent');
    $subAgent = $storage->load($subAgentId);

    if (!$subAgent instanceof ConfigEntityInterface) {
      return ['nodes' => [], 'edges' => []];
    }

    // Recursively get sub-agent workflow.
    $subWorkflow = $this->agentToWorkflow($subAgent, $expansionMode, $depth, $maxDepth, $offsetY);

    // Create edge from parent to sub-agent.
    $subAgentNodeId = 'agent_' . $subAgentId;
    $edges = $subWorkflow['edges'];
    $edges[] = [
      'id' => "edge_{$parentNodeId}_to_{$subAgentNodeId}",
      'source' => $parentNodeId,
      'target' => $subAgentNodeId,
      'sourceHandle' => "{$parentNodeId}-output-tools",
      'targetHandle' => "{$subAgentNodeId}-input-trigger",
      'type' => 'default',
      'data' => ['dataType' => 'agent'],
    ];

    return [
      'nodes' => $subWorkflow['nodes'],
      'edges' => $edges,
    ];
  }

  /**
   * Loads a sub-agent in grouped mode (visual container).
   *
   * @param string $subAgentId
   *   The sub-agent ID.
   * @param string $parentNodeId
   *   The parent node ID.
   * @param int $depth
   *   Current depth.
   * @param int $maxDepth
   *   Maximum depth.
   * @param array<string, mixed> $positions
   *   Node positions.
   *
   * @return array<string, mixed>
   *   Nodes and edges array.
   */
  protected function loadSubAgentGrouped(string $subAgentId, string $parentNodeId, int $depth, int $maxDepth, array $positions): array {
    $storage = $this->entityTypeManager->getStorage('ai_agent');
    $subAgent = $storage->load($subAgentId);

    if (!$subAgent instanceof ConfigEntityInterface) {
      return ['nodes' => [], 'edges' => []];
    }

    // Get sub-agent workflow in expanded mode first.
    $subWorkflow = $this->agentToWorkflow($subAgent, 'grouped', $depth, $maxDepth);

    // Calculate group position based on depth.
    $groupX = self::LAYOUT['agentX'] + ($depth * self::LAYOUT['subAgentOffsetX']);
    // Slightly above to account for padding.
    $groupY = self::LAYOUT['agentY'] - 50;

    // Create group container node.
    $groupNodeId = 'group_' . $subAgentId;
    $groupNode = [
      'id' => $groupNodeId,
      'type' => 'group',
      'position' => $positions[$groupNodeId] ?? ['x' => $groupX, 'y' => $groupY],
      'data' => [
        'label' => $subAgent->label(),
      ],
      'style' => [
        'width' => 600,
        'height' => 350,
      ],
    ];

    // Add parentId to all sub-agent nodes for visual grouping.
    foreach ($subWorkflow['nodes'] as &$node) {
      $node['parentId'] = $groupNodeId;
      $node['extent'] = 'parent';
    }

    // Prepend group node.
    array_unshift($subWorkflow['nodes'], $groupNode);

    // Create edge from parent to group.
    $edges = $subWorkflow['edges'];
    $edges[] = [
      'id' => "edge_{$parentNodeId}_to_{$groupNodeId}",
      'source' => $parentNodeId,
      'target' => $groupNodeId,
      'sourceHandle' => "{$parentNodeId}-output-tools",
      'targetHandle' => "{$groupNodeId}-input-trigger",
      'type' => 'default',
      'data' => ['dataType' => 'agent'],
    ];

    return [
      'nodes' => $subWorkflow['nodes'],
      'edges' => $edges,
    ];
  }

  /**
   * Creates a collapsed agent node (single node with badge).
   *
   * @param string $subAgentId
   *   The sub-agent ID.
   * @param string $nodeId
   *   The node ID.
   * @param array<string, mixed>|null $position
   *   Optional node position.
   * @param int $index
   *   Index for default positioning.
   *
   * @return array<string, mixed>|null
   *   Node data or NULL.
   */
  protected function createCollapsedAgentNode(string $subAgentId, string $nodeId, ?array $position, int $index): ?array {
    $storage = $this->entityTypeManager->getStorage('ai_agent');
    $subAgent = $storage->load($subAgentId);

    if (!$subAgent instanceof AiAgent) {
      return NULL;
    }

    // Count tools in sub-agent.
    $tools = $subAgent->get('tools') ?? [];
    $toolCount = count(array_filter($tools));

    // Calculate default position if not provided.
    if ($position === NULL) {
      $position = [
        'x' => 100,
        'y' => 100 + ($index * 100),
      ];
    }

    return [
      'id' => $nodeId,
      'type' => 'universalNode',
      'position' => $position,
      'data' => [
        'nodeId' => $nodeId,
        'label' => $subAgent->label() . " [{$toolCount} tools]",
        'nodeType' => 'agent-collapsed',
        'config' => [
          'agent_id' => $subAgentId,
          'tool_count' => $toolCount,
        ],
        'metadata' => [
          'id' => 'ai_agents::ai_agent::' . $subAgentId,
          'name' => $subAgent->label(),
          'description' => $subAgent->get('description') ?? '',
          'isCollapsedAgent' => TRUE,
          'type' => 'agent',
          'supportedTypes' => ['agent', 'simple', 'default'],
          'icon' => 'mdi:robot-outline',
          'color' => 'var(--color-ref-purple-300)',
          'inputs' => [
            [
              'id' => 'tool',
              'name' => 'Tool',
              'type' => 'input',
              'dataType' => 'tool',
              'required' => FALSE,
              'description' => 'Tool connection from parent agent',
            ],
          ],
          'outputs' => [
            [
              'id' => 'tool',
              'name' => 'Tool',
              'type' => 'output',
              'dataType' => 'tool',
              'required' => FALSE,
              'description' => 'Tool output',
            ],
          ],
        ],
      ],
    ];
  }

  /**
   * Creates a FlowDrop node for the main agent.
   *
   * Note: Position is set by the caller, not here.
   *
   * @param \Drupal\ai_agents\Entity\AiAgent $agent
   *   The agent entity.
   * @param string $nodeId
   *   The node ID.
   *
   * @return array<string, mixed>
   *   Node data array.
   */
  protected function createAgentNode(AiAgent $agent, string $nodeId): array {
    return [
      'id' => $nodeId,
      'type' => 'universalNode',
      // Will be overridden by caller.
      'position' => ['x' => 0, 'y' => 0],
      'data' => [
        'nodeId' => $nodeId,
        'label' => $agent->label(),
        'nodeType' => 'agent',
        'config' => [
          'label' => $agent->label(),
          'description' => $agent->get('description') ?? '',
          'systemPrompt' => $agent->get('system_prompt') ?? '',
          'maxLoops' => $agent->get('max_loops') ?? 3,
          'orchestrationAgent' => $agent->get('orchestration_agent') ?? FALSE,
          'triageAgent' => $agent->get('triage_agent') ?? FALSE,
        ],
        'metadata' => [
          'id' => 'ai_agent',
          'name' => 'AI Agent',
          'type' => 'agent',
          'supportedTypes' => ['agent', 'simple', 'default'],
          'category' => 'agents',
          'icon' => 'mdi:face-agent',
          'color' => self::CATEGORY_COLORS['agent'],
          'inputs' => [
            [
              'id' => 'trigger',
              'name' => 'Trigger',
              'type' => 'input',
              'dataType' => 'trigger',
              'required' => FALSE,
              'description' => 'Trigger input',
            ],
            [
              'id' => 'message',
              'name' => 'Message',
              'type' => 'input',
              'dataType' => 'string',
              'required' => FALSE,
              'description' => 'Input message',
            ],
          ],
          'outputs' => [
            [
              'id' => 'response',
              'name' => 'Response',
              'type' => 'output',
              'dataType' => 'string',
              'required' => FALSE,
              'description' => 'Agent response output',
            ],
            [
              'id' => 'tools',
              'name' => 'Tools',
              'type' => 'output',
              'dataType' => 'tool',
              'required' => FALSE,
              'description' => 'Tools output for chaining',
            ],
          ],
          'configSchema' => $this->getAgentConfigSchema(),
        ],
      ],
    ];
  }

  /**
   * Creates a FlowDrop node for a tool.
   *
   * @param string $toolId
   *   The tool ID.
   * @param string $nodeId
   *   The node ID.
   * @param array<string, mixed> $settings
   *   Tool settings.
   * @param array<string, mixed>|null $position
   *   Optional node position.
   * @param int $index
   *   Index for default positioning.
   *
   * @return array<string, mixed>|null
   *   Node data or NULL.
   */
  protected function createToolNode(string $toolId, string $nodeId, array $settings, ?array $position, int $index): ?array {
    try {
      $plugin = $this->functionCallPluginManager->createInstance($toolId);
      assert($plugin instanceof PluginInspectionInterface);
      $definition = $plugin->getPluginDefinition();

      // Cast to string to handle TranslatableMarkup objects.
      // @phpstan-ignore-next-line offsetAccess.nonOffsetAccessible
      $label = (string) ($definition['label'] ?? $definition['name'] ?? $toolId);
      // @phpstan-ignore-next-line offsetAccess.nonOffsetAccessible
      $description = (string) ($definition['description'] ?? '');
      $category = $this->getToolCategory($toolId);

      // Calculate default position if not provided.
      if ($position === NULL) {
        $position = [
          'x' => 100,
          'y' => 100 + ($index * 100),
        ];
      }

      // Flatten property_restrictions to prop_* keys for FlowDrop UI.
      $flatConfig = [
        'tool_id' => $toolId,
        'label' => $label,
        'return_directly' => $settings['return_directly'] ?? FALSE,
        'require_usage' => $settings['require_usage'] ?? FALSE,
        'use_artifacts' => $settings['use_artifacts'] ?? FALSE,
        'description_override_enabled' => !empty($settings['description_override']),
        'description_override' => $settings['description_override'] ?? '',
        'toolDescription' => $description,
        'progress_message' => $settings['progress_message'] ?? '',
      ];

      // Convert nested property_restrictions to flat prop_* keys.
      $restrictions = $settings['property_restrictions'] ?? [];
      $actionMap = [
        'force_value' => 'Force value',
        'only_allow' => 'Only allow certain values',
      ];
      foreach ($restrictions as $propName => $propConfig) {
        // Restriction dropdown.
        if (isset($propConfig['action'])) {
          $flatConfig['prop_' . $propName . '_restriction'] = $actionMap[$propConfig['action']] ?? 'Allow all';
        }

        // Hidden checkbox.
        if (!empty($propConfig['hidden'])) {
          $flatConfig['prop_' . $propName . '_hidden'] = TRUE;
        }

        // Values.
        if (isset($propConfig['force_value'])) {
          $flatConfig['prop_' . $propName . '_values'] = $propConfig['force_value'];
        }
        elseif (isset($propConfig['allowed_values'])) {
          $flatConfig['prop_' . $propName . '_values'] = $propConfig['allowed_values'];
        }
      }

      // Property description overrides are stored inside property_restrictions:
      // tool_settings[tool_id]['property_restrictions'][property_name]['description_override'].
      $propRestrictions = $settings['property_restrictions'] ?? [];
      foreach ($propRestrictions as $propName => $restriction) {
        if (!empty($restriction['description_override'])) {
          $flatConfig['prop_' . $propName . '_override_desc_enabled'] = TRUE;
          $flatConfig['prop_' . $propName . '_override_desc'] = $restriction['description_override'];
        }
      }

      return [
        'id' => $nodeId,
        'type' => 'universalNode',
        'position' => $position,
        'data' => [
          'nodeId' => $nodeId,
          'label' => $label,
          'nodeType' => 'tool',
          'toolId' => $toolId,
          'config' => $flatConfig,
          'metadata' => [
            'id' => $toolId,
            'name' => $label,
            'description' => $description,
            'type' => 'tool',
            'supportedTypes' => ['tool', 'simple', 'default'],
            'category' => $this->transformCategoryToPlural($category),
            'icon' => 'mdi:tools',
            'color' => $this->getCategoryColor($category),
            'tool_id' => $toolId,
            'inputs' => [
              [
                'id' => 'tool',
                'name' => 'Tool',
                'type' => 'input',
                'dataType' => 'tool',
                'required' => FALSE,
                'description' => 'Tool connection from agent',
              ],
            ],
            'outputs' => [
              [
                'id' => 'tool',
                'name' => 'Tool',
                'type' => 'output',
                'dataType' => 'tool',
                'required' => FALSE,
                'description' => 'Tool output',
              ],
            ],
            'configSchema' => $this->getToolConfigSchema($toolId),
          ],
        ],
      ];
    }
    catch (\Exception $e) {
      $this->loggerFactory->get('flowdrop_ui_agents')->warning(
        'Could not create tool node for @tool: @error',
        ['@tool' => $toolId, '@error' => $e->getMessage()]
          );
      return NULL;
    }
  }

  /**
   * Gets available tools for the sidebar.
   *
   * Returns tools in FlowDrop node format so they can be dragged directly
   * onto the canvas.
   *
   * @param \Drupal\modeler_api\Plugin\ModelerApiModelOwner\ModelOwnerInterface $owner
   *   The model owner.
   *
   * @return array<int, array<string, mixed>>
   *   Array of available tools.
   */
  public function getAvailableTools(ModelOwnerInterface $owner): array {
    $tools = [];

    // Get all function call plugins (tools).
    $definitions = $this->functionCallPluginManager->getDefinitions();

    foreach ($definitions as $id => $definition) {
      // Skip agent wrappers (ai_agents::ai_agent::*).
      if (str_starts_with($id, 'ai_agents::ai_agent::')) {
        continue;
      }

      // Cast to string to handle TranslatableMarkup objects.
      $label = (string) ($definition['label'] ?? $definition['name'] ?? $id);
      $description = (string) ($definition['description'] ?? '');
      $category = $this->getToolCategory($id);

      // Use the same JSON schema as tool nodes so config panels render
      // correctly.
      $configSchema = $this->getToolConfigSchema($id);

      $tools[] = [
        'id' => $id,
        'name' => $label,
        'type' => 'tool',
        'supportedTypes' => ['tool'],
        'description' => $description,
        'category' => $this->transformCategoryToPlural($category),
        'icon' => 'mdi:tools',
        'color' => $this->getCategoryColor($category),
        'version' => '1.0.0',
        'enabled' => TRUE,
        'tags' => [$category],
        'executor_plugin' => 'tool:' . $id,
        'tool_id' => $id,
        'inputs' => [
          [
            'id' => 'tool',
            'name' => 'Tool',
            'type' => 'input',
            'dataType' => 'tool',
            'required' => FALSE,
            'description' => 'Tool connection from agent',
          ],
        ],
        'outputs' => [
          [
            'id' => 'tool',
            'name' => 'Tool',
            'type' => 'output',
            'dataType' => 'tool',
            'required' => FALSE,
            'description' => 'Tool output',
          ],
        ],
        'config' => [
          'tool_id' => $id,
        ],
        'configSchema' => $configSchema,
      ];
    }

    // Sort by category then name.
    usort($tools, function ($a, $b) {
      $categoryCompare = strcmp($a['category'], $b['category']);
      return $categoryCompare !== 0 ? $categoryCompare : strcmp($a['name'], $b['name']);
    });

    return $tools;
  }

  /**
   * Gets available agents for the sidebar (other than current).
   *
   * @param \Drupal\modeler_api\Plugin\ModelerApiModelOwner\ModelOwnerInterface $owner
   *   The model owner.
   *
   * @return array<int, array<string, mixed>>
   *   Array of available agents.
   */
  public function getAvailableAgents(ModelOwnerInterface $owner): array {
    $agents = [];
    $storage = $this->entityTypeManager->getStorage('ai_agent');
    $allAgents = $storage->loadMultiple();

    foreach ($allAgents as $agent) {
      assert($agent instanceof AiAgent);
      $agentId = $agent->id();
      $label = $agent->label();
      $description = $agent->get('description') ?? '';

      $agents[] = [
        'id' => 'ai_agent_' . $agentId,
        'name' => $label,
        'type' => 'agent',
        'supportedTypes' => ['agent'],
        'description' => $description,
        // Use 'agents' for FlowDrop's internal styling.
        // JS renames header to "Sub-Agent Tools".
        'category' => 'agents',
        'icon' => 'mdi:face-agent',
        'color' => self::CATEGORY_COLORS['agent'],
        'version' => '1.0.0',
        'enabled' => TRUE,
        'tags' => ['agent'],
        'executor_plugin' => 'ai_agents::ai_agent::' . $agentId,
        'agent_id' => $agentId,
        'tool_id' => 'ai_agents::ai_agent::' . $agentId,
        'inputs' => [
          [
            'id' => 'trigger',
            'name' => 'Trigger',
            'type' => 'input',
            'dataType' => 'trigger',
            'required' => FALSE,
            'description' => 'Trigger input',
          ],
          [
            'id' => 'message',
            'name' => 'Message',
            'type' => 'input',
            'dataType' => 'string',
            'required' => FALSE,
            'description' => 'Input message',
          ],
        ],
        'outputs' => [
          [
            'id' => 'response',
            'name' => 'Response',
            'type' => 'output',
            'dataType' => 'string',
            'required' => FALSE,
            'description' => 'Agent response',
          ],
        ],
        'config' => [],
        'configSchema' => $this->getAgentConfigSchema(),
      ];
    }

    // Sort by name.
    usort($agents, fn(array $a, array $b): int => strcmp((string) $a['name'], (string) $b['name']));

    return $agents;
  }

  /**
   * Gets available chatbots for the sidebar (unassigned only).
   *
   * Only returns DeepChat blocks that do NOT have an assistant assigned,
   * as chatbots already linked to an assistant should not appear in the
   * sidebar for selection.
   *
   * @return array<int, array<string, mixed>>
   *   Array of available (unassigned) chatbots.
   */
  public function getAvailableChatbots(): array {
    $chatbots = [];

    try {
      $storage = $this->entityTypeManager->getStorage('block');
      // Load all DeepChat blocks.
      $blocks = $storage->loadByProperties([
        'plugin' => 'ai_deepchat_block',
      ]);
    }
    catch (\Exception $e) {
      $this->loggerFactory->get('flowdrop_ui_agents')->warning(
        'Could not load chatbot blocks: @message',
        ['@message' => $e->getMessage()]
          );
      return [];
    }

    foreach ($blocks as $block) {
      $blockId = $block->id();
      $settings = $block->get('settings') ?? [];

      // Only include chatbots WITHOUT an assistant assigned (unassigned).
      $assistantId = $settings['ai_assistant'] ?? '';
      if (!empty($assistantId)) {
        // This chatbot is already linked to an assistant, skip it.
        continue;
      }

      $label = $settings['label'] ?? $settings['bot_name'] ?? $blockId;
      $botName = $settings['bot_name'] ?? $label;

      $chatbots[] = [
        'id' => 'chatbot_' . $blockId,
        'name' => $label,
        'type' => 'chatbot',
        'supportedTypes' => ['chatbot'],
        'description' => 'DeepChat chatbot: ' . $botName,
        'category' => 'Chatbots',
        'icon' => 'mdi:chat',
        'color' => self::CATEGORY_COLORS['chatbot'],
        'version' => '1.0.0',
        'enabled' => TRUE,
        'tags' => ['chatbot', 'deepchat'],
        'block_id' => $blockId,
        'inputs' => [],
        'outputs' => [
          [
            'id' => 'trigger',
            'name' => 'Trigger',
            'type' => 'output',
            'dataType' => 'trigger',
            'required' => FALSE,
            'description' => 'Triggers the connected assistant',
          ],
        ],
        'config' => [
          'blockId' => $blockId,
          'botName' => $botName,
          'placement' => $settings['placement'] ?? 'bottom-right',
          'theme' => $block->getTheme(),
          'region' => $block->getRegion(),
        ],
        'configSchema' => $this->getChatbotConfigSchema(),
      ];
    }

    // Sort by name.
    usort($chatbots, fn(array $a, array $b): int => strcmp((string) $a['name'], (string) $b['name']));

    return $chatbots;
  }

  /**
   * Gets the config schema for chatbot nodes.
   *
   * @return array<string, mixed>
   *   JSON Schema format config schema.
   */
  protected function getChatbotConfigSchema(): array {
    return [
      'type' => 'object',
      'properties' => [
        'blockId' => [
          'type' => 'string',
          'title' => 'Block ID',
          'description' => 'The block entity ID (internal use)',
          'format' => 'hidden',
        ],
        'botName' => [
          'type' => 'string',
          'title' => 'Bot Name',
          'description' => 'Display name for the chatbot',
        ],
        'placement' => [
          'type' => 'string',
          'title' => 'Placement',
          'description' => 'Where the chatbot appears on screen',
          'enum' => ['bottom-right', 'bottom-left', 'toolbar'],
          'default' => 'bottom-right',
        ],
        'theme' => [
          'type' => 'string',
          'title' => 'Theme',
          'description' => 'Drupal theme for the chatbot block',
        ],
        'region' => [
          'type' => 'string',
          'title' => 'Region',
          'description' => 'Theme region for the chatbot block',
        ],
      ],
    ];
  }

  /**
   * Gets tools grouped by category for sidebar display.
   *
   * @param \Drupal\modeler_api\Plugin\ModelerApiModelOwner\ModelOwnerInterface $owner
   *   The model owner.
   *
   * @return array<string, array<int, array<string, mixed>>>
   *   Tools grouped by category.
   */
  public function getToolsByCategory(ModelOwnerInterface $owner): array {
    $tools = $this->getAvailableTools($owner);
    $grouped = [];

    foreach ($tools as $tool) {
      $category = $tool['category'];
      if (!isset($grouped[$category])) {
        $grouped[$category] = [];
      }
      $grouped[$category][] = $tool;
    }

    // Sort categories alphabetically.
    ksort($grouped);
    // Sort categories alphabetically, but put 'agents' (Sub-Agent Tools)
    // or 'Sub-Agent Tools' first.
    // Sort categories by weight then alphabetically.
    uksort($grouped, function ($a, $b) {
      $aName = strtolower($a);
      $bName = strtolower($b);

      $getWeight = function ($name) {
        $normalized = strtolower(trim($name));
        // Agents go to the top.
        if (str_contains($normalized, 'sub-agent') || str_contains($normalized, 'agent')) {
          return -20;
        }
        // Chatbots slightly below agents but above normal tools.
        if (str_contains($normalized, 'chatbot')) {
          return -5;
        }
        // Default weight.
        return 0;
      };

      $weightA = $getWeight($aName);
      $weightB = $getWeight($bName);

      if ($weightA !== $weightB) {
        return $weightA <=> $weightB;
      }

      return strcmp($aName, $bName);
    });

    return $grouped;
  }

  /**
   * Determines the category for a tool based on its ID.
   */
  protected function getToolCategory(string $toolId): string {
    // Try to get category from plugin definition.
    try {
      $definition = $this->functionCallPluginManager->getDefinition($toolId);

      // Check for 'category' or 'group' in definition.
      $groupId = NULL;
      if (is_array($definition)) {
        $groupId = $definition['category'] ?? $definition['group'] ?? NULL;
      }
      elseif (is_object($definition)) {
        // Handle object definitions (e.g. Attributes).
        $groupId = $definition->category ?? $definition->group ?? NULL;
      }

      if ($groupId) {
        if ($this->functionGroupPluginManager->hasDefinition($groupId)) {
          $groupDef = $this->functionGroupPluginManager->getDefinition($groupId);
          if (is_array($groupDef)) {
            // Debug script confirmed 'group_name' is the label key.
            $label = $groupDef['group_name'] ?? $groupDef['label'] ?? $groupDef['name'] ?? ucfirst($groupId);
            return (string) $label;
          }
        }
        // Fallback: humanize the group ID (replace underscores with spaces).
        return ucwords(str_replace('_', ' ', $groupId));
      }
    }
    catch (\Exception $e) {
      // Fall through to default.
    }

    return 'Other';
  }

  /**
   * Transforms singular category to plural form for FlowDrop API.
   */
  protected function transformCategoryToPlural(string $category): string {
    // If the category contains spaces or uppercase letters, assume it's
    // already a label.
    if (strpos($category, ' ') !== FALSE || strtolower($category) !== $category) {
      return $category;
    }

    return self::CATEGORY_PLURAL_MAP[$category]
      ?? self::CATEGORY_PLURAL_MAP[strtolower($category)]
      ?? strtolower($category) . 's';
  }

  /**
   * Gets CSS color for a category.
   */
  protected function getCategoryColor(string $category): string {
    return self::CATEGORY_COLORS[$category]
      ?? self::CATEGORY_COLORS[strtolower($category)]
      ?? 'var(--color-ref-orange-500)';
  }

  /**
   * Returns JSON Schema for agent node configuration.
   *
   * @return array<string, mixed>
   *   JSON Schema array.
   */
  protected function getAgentConfigSchema(): array {
    return [
      'type' => 'object',
      'properties' => [
        'label' => [
          'type' => 'string',
          'title' => 'Label',
          'description' => 'Human-readable name for the agent',
        ],
        'description' => [
          'type' => 'string',
          'title' => 'Description',
          'description' => 'Description used by triage agents to select this agent',
        ],
        'systemPrompt' => [
          'type' => 'string',
          'format' => 'multiline',
          'title' => 'System Prompt',
          'description' => 'Core instructions for agent behavior',
        ],
        'maxLoops' => [
          'type' => 'integer',
          'title' => 'Max Loops',
          'description' => 'Maximum iterations before stopping (1-100)',
          'default' => 3,
        ],
        'orchestrationAgent' => [
          'type' => 'boolean',
          'title' => 'Orchestration Agent',
          'description' => 'If true, agent only picks other agents for work',
          'default' => FALSE,
        ],
        'triageAgent' => [
          'type' => 'boolean',
          'title' => 'Triage Agent',
          'description' => 'If true, agent can pick other agents AND do its own work',
          'default' => FALSE,
        ],
      ],
      'required' => ['label', 'description', 'systemPrompt'],
    ];
  }

  /**
   * Returns JSON Schema for tool node configuration.
   *
   * @param string $toolId
   *   Optional tool ID for plugin-specific config.
   *
   * @return array<string, mixed>
   *   JSON Schema array.
   */
  protected function getToolConfigSchema(string $toolId = ''): array {
    $properties = [];

    // 1. Tool ID (Hidden)
    $properties['tool_id'] = [
      'type' => 'string',
      'title' => 'Tool ID',
      'description' => 'The tool plugin ID (read-only)',
      'default' => $toolId,
      'format' => 'hidden',
    ];
    $properties['toolDescription'] = [
      'type' => 'string',
      'title' => 'Tool Description',
      'description' => 'Internal tool description for display in FlowDrop.',
      'default' => '',
      'format' => 'hidden',
    ];

    // 2. Return Directly and Require Usage (Most common toggles)
    $properties['return_directly'] = [
      'type' => 'boolean',
      'title' => 'Return Directly',
      'description' => 'Return tool result directly without LLM rewriting. Use for API responses or structured output.',
      'default' => FALSE,
    ];
    $properties['require_usage'] = [
      'type' => 'boolean',
      'title' => 'Require Usage',
      'description' => 'Remind the agent if it tries to output without using this tool first.',
      'default' => FALSE,
    ];

    // 3. Override Description (toggle + textarea)
    $properties['description_override_enabled'] = [
      'type' => 'boolean',
      'title' => 'Override tool description',
      'description' => 'Check this box if you want to override the description of the tool that is sent to the LLM.',
      'default' => FALSE,
    ];
    $properties['description_override'] = [
      'type' => 'string',
      'format' => 'textarea',
      'title' => 'Override Tool Description',
      'description' => 'Custom description sent to LLM instead of default.',
      'default' => '',
    ];

    // 4. Artifact Storage
    $properties['use_artifacts'] = [
      'type' => 'boolean',
      'title' => 'Use Artifact storage',
      'description' => 'Store large responses in artifacts instead of sending to AI. Reference with {{artifact:tool_name:index}}',
      'default' => FALSE,
    ];

    // 5. Progress Message
    $properties['progress_message'] = [
      'type' => 'string',
      'format' => 'textarea',
      'title' => 'Progress Message',
      'description' => 'Message shown in UI while tool is executing.',
      'default' => '',
    ];

    // 6. Plugin Config (if any)
    if (!empty($toolId)) {
      $pluginConfig = $this->getToolPluginConfigSchema($toolId);
      foreach ($pluginConfig as $key => $schema) {
        $properties[$key] = $schema;
      }
    }

    // 7. Property Setup - Flattened for FlowDrop compatibility
    if (!empty($toolId)) {
      try {
        /** @var \Drupal\ai\Service\FunctionCalling\FunctionCallInterface $plugin */
        $plugin = $this->functionCallPluginManager->createInstance($toolId);
        $toolProps = $plugin->normalize()->getProperties();

        if (!empty($toolProps)) {
          foreach ($toolProps as $prop) {
            $propName = $prop->getName();
            $propDescription = $prop->getDescription();
            $humanName = ucwords(str_replace('_', ' ', $propName));

            // Override description checkbox.
            $properties['prop_' . $propName . '_override_desc_enabled'] = [
              'type' => 'boolean',
              'title' => 'Override ' . $humanName . ' description',
              'description' => 'Check to override the description sent to the LLM.',
              'default' => FALSE,
            ];

            // Override description textarea (shown if checkbox checked)
            $properties['prop_' . $propName . '_override_desc'] = [
              'type' => 'string',
              'title' => 'Override ' . $humanName . ' property description',
              'description' => 'Current description: ' . $propDescription,
              'format' => 'textarea',
              'default' => '',
            ];

            // Restrictions dropdown.
            $properties['prop_' . $propName . '_restriction'] = [
              'type' => 'string',
              'title' => 'Restrictions for ' . $humanName,
              'description' => 'Restrict the allowed values or enforce a value.',
              'enum' => ['Allow all', 'Only allow certain values', 'Force value'],
              'default' => 'Allow all',
            ];

            // Hide property checkbox (shown if restriction != Allow all)
            $properties['prop_' . $propName . '_hidden'] = [
              'type' => 'boolean',
              'title' => 'Hide ' . $humanName,
              'description' => 'Hide this property from being sent to the LLM or from being logged.',
              'default' => FALSE,
            ];

            // Values textarea (shown if restriction != Allow all)
            $properties['prop_' . $propName . '_values'] = [
              'type' => 'string',
              'title' => $humanName . ' values',
              'description' => 'The values that are allowed or the value that should be set.',
              'format' => 'textarea',
              'default' => '',
            ];
          }
        }
      }
      catch (\Exception $e) {
        // Ignore errors during schema generation.
      }
    }

    return [
      'type' => 'object',
      'properties' => $properties,
      'required' => ['tool_id'],
    ];
  }

  /**
   * Gets config schema from a tool plugin if available.
   *
   * @param string $toolId
   *   The tool ID.
   *
   * @return array<string, mixed>
   *   Config schema array.
   */
  protected function getToolPluginConfigSchema(string $toolId): array {
    try {
      $plugin = $this->functionCallPluginManager->createInstance($toolId);

      // Check if plugin has configuration form.
      if (!$plugin instanceof ConfigurableInterface && !$plugin instanceof PluginFormInterface) {
        return [];
      }

      // Get default configuration.
      $defaultConfig = [];
      if ($plugin instanceof ConfigurableInterface) {
        $defaultConfig = $plugin->defaultConfiguration();
      }

      // Build configuration form to extract field info.
      if ($plugin instanceof PluginFormInterface) {
        $formState = new FormState();
        $form = $plugin->buildConfigurationForm([], $formState);

        return $this->extractConfigSchemaFromForm($form, $defaultConfig);
      }
    }
    catch (\Exception $e) {
      // Return empty on error.
    }

    return [];
  }

  /**
   * Extracts JSON Schema from a Drupal form array.
   *
   * @param array<string, mixed> $form
   *   Drupal form array.
   * @param array<string, mixed> $defaultConfig
   *   Default configuration values.
   *
   * @return array<string, mixed>
   *   JSON Schema array.
   */
  protected function extractConfigSchemaFromForm(array $form, array $defaultConfig): array {
    $schema = [];

    foreach ($form as $key => $element) {
      // Skip non-element keys.
      if (str_starts_with($key, '#')) {
        continue;
      }

      if (!is_array($element) || !isset($element['#type'])) {
        continue;
      }

      $fieldSchema = [
        'title' => (string) ($element['#title'] ?? $key),
      ];

      if (!empty($element['#description'])) {
        $fieldSchema['description'] = (string) $element['#description'];
      }

      // Map Drupal form type to JSON Schema type.
      $type = $element['#type'];
      switch ($type) {
        case 'checkbox':
          $fieldSchema['type'] = 'boolean';
          break;

        case 'number':
          $fieldSchema['type'] = 'number';
          break;

        case 'select':
          $fieldSchema['type'] = 'string';
          if (!empty($element['#options'])) {
            $fieldSchema['enum'] = array_keys($element['#options']);
          }
          break;

        case 'textarea':
          $fieldSchema['type'] = 'string';
          $fieldSchema['format'] = 'textarea';
          break;

        default:
          $fieldSchema['type'] = 'string';
      }

      // Set default from form or plugin config.
      if (isset($element['#default_value'])) {
        $fieldSchema['default'] = $element['#default_value'];
      }
      elseif (isset($defaultConfig[$key])) {
        $fieldSchema['default'] = $defaultConfig[$key];
      }

      $schema[$key] = $fieldSchema;
    }

    return $schema;
  }

  /**
   * Gets the tool settings schema including per-tool settings.
   *
   * This is used for the sidebar tool definitions.
   *
   * @param string $toolId
   *   The tool ID.
   *
   * @return array<int, array<string, mixed>>
   *   Tool settings schema array.
   */
  protected function getToolSettingsSchema(string $toolId): array {
    // Start with base tool settings that apply to all tools.
    $schema = [
      [
        'config_id' => 'description_override',
        'name' => 'Override Tool Description',
        'description' => 'Custom description sent to LLM instead of default.',
        'value_type' => 'text',
        'default' => '',
      ],
      [
        'config_id' => 'tool_id',
        'name' => 'Tool ID',
        'description' => 'The tool plugin ID (read-only)',
        'value_type' => 'string',
        'required' => TRUE,
        'default' => $toolId,
      ],
      [
        'config_id' => 'return_directly',
        'name' => 'Return Directly',
        'description' => 'Return tool result directly without LLM rewriting.',
        'value_type' => 'boolean',
        'default' => FALSE,
      ],
      [
        'config_id' => 'require_usage',
        'name' => 'Require Usage',
        'description' => 'Remind the agent if it tries to output without using this tool first.',
        'value_type' => 'boolean',
        'default' => FALSE,
      ],
      [
        'config_id' => 'use_artifacts',
        'name' => 'Use Artifact Storage',
        'description' => 'Store large responses in artifacts instead of sending to AI.',
        'value_type' => 'boolean',
        'default' => FALSE,
      ],
      [
        'config_id' => 'progress_message',
        'name' => 'Progress Message',
        'description' => 'Message shown in UI while tool is executing.',
        'value_type' => 'string',
        'default' => '',
      ],
    ];

    return $schema;
  }

  /**
   * Loads saved positions from agent third-party settings.
   *
   * @param \Drupal\ai_agents\Entity\AiAgent $agent
   *   The agent entity.
   *
   * @return array<string, mixed>
   *   Node positions array.
   */
  protected function loadPositions(AiAgent $agent): array {
    return $agent->getThirdPartySetting('flowdrop_ui_agents', 'positions', []);
  }

  /**
   * Saves positions to agent third-party settings.
   *
   * @param \Drupal\ai_agents\Entity\AiAgent $agent
   *   The agent entity.
   * @param array<string, mixed> $positions
   *   Node positions to save.
   */
  public function savePositions(AiAgent $agent, array $positions): void {
    $agent->setThirdPartySetting('flowdrop_ui_agents', 'positions', $positions);
  }

  /**
   * Combines tool_settings and tool_usage_limits for createToolNode().
   *
   * The AI Agent entity stores tool configuration in two separate fields:
   * - tool_settings: behavioral settings (return_directly, require_usage,
   *   etc.)
   * - tool_usage_limits: property restrictions (force_value, only_allow,
   *   hidden)
   *
   * This method combines both into a single structure that createToolNode()
   * expects, converting tool_usage_limits format to property_restrictions
   * format.
   *
   * @param array<string, array<string, mixed>> $toolSettings
   *   Behavioral settings from agent's tool_settings field.
   * @param array<string, array<string, mixed>> $toolUsageLimits
   *   Property restrictions from agent's tool_usage_limits field.
   *
   * @return array<string, array<string, mixed>>
   *   Combined settings keyed by tool ID.
   */
  protected function combineToolSettingsAndLimits(array $toolSettings, array $toolUsageLimits): array {
    $combined = [];

    // Get all tool IDs from both sources.
    $allToolIds = array_unique(array_merge(
      array_keys($toolSettings),
      array_keys($toolUsageLimits)
    ));

    foreach ($allToolIds as $toolId) {
      // Start with behavioral settings (includes property_restrictions).
      $combined[$toolId] = $toolSettings[$toolId] ?? [];

      // Convert and add property restrictions from tool_usage_limits.
      // Property description overrides are stored inside property_restrictions
      // at property_restrictions[property_name]['description_override'].
      if (isset($toolUsageLimits[$toolId])) {
        $restrictions = $this->convertToolUsageLimitsToPropertyRestrictions(
          $toolUsageLimits[$toolId]
        );
        if (!empty($restrictions)) {
          $combined[$toolId]['property_restrictions'] = $restrictions;
        }
      }
    }

    return $combined;
  }

  /**
   * Converts tool_usage_limits format to property_restrictions format.
   *
   * Drupal's tool_usage_limits format (from AiAgentForm):
   *   'property_name' => [
   *     'action' => 'force_value' | 'only_allow' | '',
   *     'hide_property' => 0|1,
   *     'values' => string | array,
   *   ]
   *
   * FlowDrop's property_restrictions format (for createToolNode):
   *   'property_name' => [
   *     'action' => 'force_value' | 'only_allow',
   *     'hidden' => bool,
   *     'force_value' => string,  // for force_value action
   *     'allowed_values' => array,  // for only_allow action
   *   ]
   *
   * @param array<string, mixed> $limits
   *   Property limits for a single tool from tool_usage_limits.
   *
   * @return array<string, array<string, mixed>>
   *   Converted property_restrictions array.
   */
  protected function convertToolUsageLimitsToPropertyRestrictions(array $limits): array {
    $restrictions = [];

    foreach ($limits as $propName => $propConfig) {
      // Skip non-property entries (metadata keys).
      if (!is_array($propConfig)) {
        continue;
      }

      $action = $propConfig['action'] ?? '';

      // Skip if no action or "allow all" (empty action = allow all).
      if (empty($action)) {
        continue;
      }

      $restriction = ['action' => $action];

      // Convert 'hide_property' to 'hidden'.
      if (!empty($propConfig['hide_property'])) {
        $restriction['hidden'] = TRUE;
      }

      // Convert 'values' to the appropriate key based on action.
      $values = $propConfig['values'] ?? NULL;
      if ($values !== NULL && $values !== '') {
        // Normalize values to array.
        $valuesArray = is_array($values) ? $values : [$values];

        if ($action === 'force_value') {
          // Force value uses a single value.
          $restriction['force_value'] = is_array($values) ? reset($values) : $values;
        }
        elseif ($action === 'only_allow') {
          // Only allow uses array of values.
          $restriction['allowed_values'] = $valuesArray;
        }
      }

      $restrictions[$propName] = $restriction;
    }

    return $restrictions;
  }

}
