<?php

declare(strict_types=1);

namespace Drupal\flowdrop_ui_agents\Service;

use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\modeler_api\Api;
use Drupal\modeler_api\Component;
use Drupal\modeler_api\ComponentSuccessor;
use Drupal\modeler_api\Plugin\ModelerApiModelOwner\ModelOwnerInterface;

/**
 * Service to parse FlowDrop workflow JSON.
 *
 * Converts to Modeler API Components.
 *
 * This handles the SAVE direction: FlowDrop workflow -> Modeler API Components.
 */
class WorkflowParser {

  /**
   * The parsed workflow data.
   *
   * @var array<string, mixed>
   */
  protected array $data = [];

  /**
   * The model owner for creating components.
   */
  protected ?ModelOwnerInterface $owner = NULL;

  /**
   * Constructs the WorkflowParser service.
   */
  public function __construct(
    protected LoggerChannelFactoryInterface $loggerFactory,
  ) {
  }

  /**
   * Parses FlowDrop workflow JSON string.
   *
   * @param string $json
   *   The JSON string from FlowDrop.
   *
   * @return array<string, mixed>
   *   Parsed workflow data.
   */
  public function parse(string $json): array {
    if (empty($json)) {
      return $this->getEmptyWorkflow();
    }

    $data = json_decode($json, TRUE);
    if (json_last_error() !== JSON_ERROR_NONE) {
      $this->loggerFactory->get('flowdrop_ui_agents')->error(
        'Failed to parse FlowDrop JSON: @error',
        ['@error' => json_last_error_msg()]
      );
      return $this->getEmptyWorkflow();
    }

    $this->data = $data;
    return $data;
  }

  /**
   * Sets the model owner for component creation.
   */
  public function setOwner(ModelOwnerInterface $owner): void {
    $this->owner = $owner;
  }

  /**
   * Converts parsed workflow data to Modeler API Components.
   *
   * @param array<string, mixed> $data
   *   The parsed workflow data.
   *
   * @return array<int, \Drupal\modeler_api\Component>
   *   Array of Modeler API Component objects.
   */
  public function toComponents(array $data): array {
    if ($this->owner === NULL) {
      $this->loggerFactory->get('flowdrop_ui_agents')->error(
        'Cannot create components without model owner'
      );
      return [];
    }

    $components = [];
    $nodes = $data['nodes'] ?? [];
    $edges = $data['edges'] ?? [];

    // The primary agent ID is the workflow's ID.
    // This is the entity being edited.
    $primaryAgentId = $data['id'] ?? '';

    // Build edge map for finding successors.
    $edgeMap = $this->buildEdgeMap($edges);

    // Track which tools belong to the primary agent.
    $primaryAgentTools = [];

    foreach ($nodes as $node) {
      $nodeType = $node['data']['nodeType'] ?? 'unknown';
      $nodeMetadata = $node['data']['metadata'] ?? [];
      $ownerAgentId = $nodeMetadata['ownerAgentId'] ?? '';

      // Fallback: check metadata.type if nodeType is not set.
      if (($nodeType === 'unknown' || empty($nodeType)) && isset($nodeMetadata['type'])) {
        $nodeType = $nodeMetadata['type'];
      }

      switch ($nodeType) {
        case 'agent':
          // Only create a START component for the PRIMARY agent.
          // Sub-agents (expanded from the primary) should be treated as
          // SUBPROCESS references, not modified.
          if ($ownerAgentId === $primaryAgentId || $ownerAgentId === '') {
            // This is the primary agent being edited.
            $component = $this->createAgentComponent($node, $edgeMap, $data);
            if ($component) {
              $components[] = $component;
            }
          }
          else {
            // This is an expanded sub-agent - create a SUBPROCESS reference.
            // Don't modify it, just reference it as a tool.
            $primaryAgentTools[] = 'ai_agents::ai_agent::' . $ownerAgentId;
          }
          break;

        case 'agent-collapsed':
          // Collapsed agents are sub-agent references.
          $subAgentId = $node['data']['config']['agent_id'] ?? $nodeMetadata['id'] ?? '';
          if ($subAgentId && str_starts_with($subAgentId, 'ai_agents::ai_agent::')) {
            $primaryAgentTools[] = $subAgentId;
          }
          elseif ($subAgentId) {
            $primaryAgentTools[] = 'ai_agents::ai_agent::' . $subAgentId;
          }
          break;

        case 'tool':
          // Only add tools that belong to the primary agent.
          if ($ownerAgentId === $primaryAgentId || $ownerAgentId === '') {
            $component = $this->createToolComponent($node, $edgeMap);
            if ($component) {
              $components[] = $component;
            }
          }
          break;
      }
    }

    // Add link components for edges.
    foreach ($edges as $edge) {
      $component = $this->createLinkComponent($edge);
      if ($component) {
        $components[] = $component;
      }
    }

    // Store sub-agent references for later processing.
    $this->data['_subAgentTools'] = $primaryAgentTools;

    return $components;
  }

  /**
   * Creates an agent component from a FlowDrop node.
   *
   * The configuration keys must match what AI Agents ModelOwner expects:
   * - agent_id: The agent machine name
   * - label: Human-readable name
   * - description: Description for triage agents
   * - system_prompt: Main instructions
   * - max_loops: Max iterations (integer)
   * - orchestration_agent: Boolean.
   * - triage_agent: Boolean.
   *
   * @param array<string, mixed> $node
   *   The workflow node.
   * @param array<string, mixed> $edgeMap
   *   Edge map for finding connections.
   * @param array<string, mixed> $workflowData
   *   Full workflow data.
   *
   * @return \Drupal\modeler_api\Component|null
   *   Component or NULL.
   */
  protected function createAgentComponent(array $node, array $edgeMap, array $workflowData): ?Component {
    $nodeId = $node['id'];
    $config = $node['data']['config'] ?? [];
    $nodeMetadata = $node['data']['metadata'] ?? [];
    $workflowMetadata = $workflowData['metadata']['agentConfig'] ?? [];

    // Get the agent ID from the node's ownerAgentId (set during load).
    // This is critical for multi-agent workflows where sub-agents are expanded.
    // Each agent node must use its OWN ID, not the workflow's top-level ID.
    $agentId = $nodeMetadata['ownerAgentId']
      ?? $config['agent_id']
      ?? $workflowData['id']
      ?? $nodeId;

    // Build agent configuration matching
    // AiAgentForm::defaultConfigMetadata('agent_id').
    $agentConfig = [
      'agent_id' => $agentId,
      'label' => $config['label'] ?? $workflowData['label'] ?? $workflowData['name'] ?? 'Agent',
      'description' => $config['description'] ?? $workflowMetadata['description'] ?? '',
      'system_prompt' => $config['systemPrompt'] ?? $workflowMetadata['system_prompt'] ?? '',
      'max_loops' => (int) ($config['maxLoops'] ?? $workflowMetadata['max_loops'] ?? 3),
      'orchestration_agent' => (bool) ($config['orchestrationAgent'] ?? $workflowMetadata['orchestration_agent'] ?? FALSE),
      'triage_agent' => (bool) ($config['triageAgent'] ?? $workflowMetadata['triage_agent'] ?? FALSE),
      // Required by AI Agents but we don't expose in UI yet.
      'secured_system_prompt' => '[ai_agent:agent_instructions]',
      'default_information_tools' => '',
      'structured_output_enabled' => FALSE,
      'structured_output_schema' => '',
    ];

    // Find connected tools (successors).
    $successors = [];
    $incomingEdges = $edgeMap['incoming'][$nodeId] ?? [];
    foreach ($incomingEdges as $edge) {
      $successors[] = new ComponentSuccessor($edge['source'], '');
    }

    return new Component(
      $this->owner,
      $nodeId,
      Api::COMPONENT_TYPE_START,
      // Plugin ID should be the agent ID.
      $agentId,
      $agentConfig['label'],
      $agentConfig,
      $successors,
    );
  }

  /**
   * Creates a tool component from a FlowDrop node.
   *
   * @param array<string, mixed> $node
   *   The workflow node.
   * @param array<string, mixed> $edgeMap
   *   Edge map for finding connections.
   *
   * @return \Drupal\modeler_api\Component|null
   *   Component or NULL.
   */
  protected function createToolComponent(array $node, array $edgeMap): ?Component {
    $nodeId = $node['id'];
    $config = $node['data']['config'] ?? [];
    $nodeMetadata = $node['data']['metadata'] ?? [];

    // Check multiple locations for tool_id (camelCase and snake_case, in data,
    // config, and metadata).
    $toolId = $node['data']['toolId']
      ?? $config['toolId']
      ?? $config['tool_id']
      ?? $nodeMetadata['tool_id']
      ?? '';

    if (empty($toolId)) {
      return NULL;
    }

    // Build tool configuration.
    // Map snake_case keys (from schema) with camelCase fallbacks (legacy/js).
    $descriptionOverrideEnabled = $config['description_override_enabled']
      ?? $config['descriptionOverrideEnabled']
      ?? NULL;
    $descriptionOverride = $config['description_override'] ?? $config['descriptionOverride'] ?? '';
    if ($descriptionOverrideEnabled !== NULL && !$descriptionOverrideEnabled) {
      $descriptionOverride = '';
    }

    $toolConfig = [
      'return_directly' => $config['return_directly'] ?? $config['returnDirectly'] ?? FALSE,
      'require_usage' => $config['require_usage'] ?? $config['requireUsage'] ?? FALSE,
      'use_artifacts' => $config['use_artifacts'] ?? $config['useArtifacts'] ?? FALSE,
      'description_override' => $descriptionOverride,
      'progress_message' => $config['progress_message'] ?? $config['progressMessage'] ?? '',
    ];

    // Add flat property restriction keys in the format Agent ModelOwner
    // expects.
    // Format: propName___action, propName___values,
    // propName___hide_property.
    $flatRestrictions = $this->buildFlatPropertyRestrictions($config);
    $toolConfig = array_merge($toolConfig, $flatRestrictions);

    // Build nested property_restrictions for tool_settings.
    $nestedRestrictions = $this->buildPropertyRestrictions($config);

    // Merge property description overrides into property_restrictions.
    // Agent ModelOwner only knows about property_restrictions, not
    // property_description_override as a separate key.
    $propertyDescriptionOverrides = $this->buildPropertyDescriptionOverrides($config);
    foreach ($propertyDescriptionOverrides as $propName => $description) {
      if (!isset($nestedRestrictions[$propName])) {
        $nestedRestrictions[$propName] = [];
      }
      $nestedRestrictions[$propName]['description_override'] = $description;
    }

    if (!empty($nestedRestrictions)) {
      $toolConfig['property_restrictions'] = $nestedRestrictions;
    }

    // Merge any other plugin-specific config (everything else in $config).
    // Exclude prop_* keys (processed into flat restrictions above) and
    // common keys.
    $reserved = array_flip(['tool_id', 'label', 'toolId', 'nodeType', 'nodeId', 'toolDescription']);
    $extraConfig = array_diff_key($config, $reserved, $toolConfig);

    // Define keys that should be kept as-is (not converted to restrictions).
    $commonConfigKeys = [
      'return_directly',
      'require_usage',
      'use_artifacts',
      'description_override',
      'progress_message',
    ];

    // Process extra config keys.
    foreach ($extraConfig as $key => $value) {
      // Skip prop_* keys - they're already processed into flat restrictions.
      if (str_starts_with($key, 'prop_')) {
        continue;
      }

      // Skip control flags - they're only used for UI/processing logic.
      if (str_ends_with($key, '_enabled') || str_ends_with($key, '_hidden')) {
        continue;
      }

      // Skip camelCase variants of snake_case keys already in toolConfig.
      if ($key === 'returnDirectly' || $key === 'requireUsage' || $key === 'useArtifacts' || $key === 'progressMessage' || $key === 'descriptionOverride' || $key === 'descriptionOverrideEnabled') {
        continue;
      }

      // Skip if already set.
      if (isset($toolConfig[$key])) {
        continue;
      }

      // Check if this is a common config key that should be kept as-is.
      if (in_array($key, $commonConfigKeys, TRUE)) {
        $toolConfig[$key] = $value;
      }
      else {
        // This is a tool-specific property value - convert to flat
        // restriction format so Agent ModelOwner will save it correctly.
        $safePropName = str_replace(':', '__colon__', $key);
        $actionKey = $safePropName . '___action';
        $valuesKey = $safePropName . '___values';

        if (!isset($toolConfig[$actionKey])) {
          $toolConfig[$actionKey] = 'force_value';
          $toolConfig[$valuesKey] = $value;
        }

        // Also add to nested property_restrictions format.
        if (!isset($toolConfig['property_restrictions'])) {
          $toolConfig['property_restrictions'] = [];
        }
        if (!isset($toolConfig['property_restrictions'][$key])) {
          $toolConfig['property_restrictions'][$key] = [
            'action' => 'force_value',
            'force_value' => $value,
          ];
        }
      }
    }

    // Find successors (what this tool connects to).
    $successors = [];
    $outgoingEdges = $edgeMap['outgoing'][$nodeId] ?? [];
    foreach ($outgoingEdges as $edge) {
      $successors[] = new ComponentSuccessor($edge['target'], '');
    }

    return new Component(
      $this->owner,
      $nodeId,
      Api::COMPONENT_TYPE_ELEMENT,
      $toolId,
      $config['label'] ?? $toolId,
      $toolConfig,
      $successors,
    );
  }

  /**
   * Creates a link component from a FlowDrop edge.
   *
   * @param array<string, mixed> $edge
   *   The workflow edge.
   *
   * @return \Drupal\modeler_api\Component|null
   *   Component or NULL.
   */
  protected function createLinkComponent(array $edge): ?Component {
    return new Component(
      $this->owner,
      $edge['id'],
      Api::COMPONENT_TYPE_LINK,
      '',
      '',
      [
        'source' => $edge['source'],
        'target' => $edge['target'],
      ],
      [],
    );
  }

  /**
   * Builds a map of edges for quick lookup.
   *
   * @param array<string, mixed> $edges
   *   Array of workflow edges.
   *
   * @return array<string, mixed>
   *   Edge map with 'incoming' and 'outgoing' keys.
   */
  protected function buildEdgeMap(array $edges): array {
    $map = [
      'incoming' => [],
      'outgoing' => [],
    ];

    foreach ($edges as $edge) {
      $source = $edge['source'];
      $target = $edge['target'];

      $map['outgoing'][$source][] = $edge;
      $map['incoming'][$target][] = $edge;
    }

    return $map;
  }

  /**
   * Builds property_restrictions from flattened prop_* config keys.
   *
   * Converts FlowDrop's flat schema (prop_X_restriction, prop_X_values, etc.)
   * back to Drupal's nested property_restrictions format.
   *
   * @param array<string, mixed> $config
   *   The node config containing flattened prop_* keys.
   *
   * @return array<string, mixed>
   *   Nested property_restrictions array for Drupal.
   */
  protected function buildPropertyRestrictions(array $config): array {
    // Check for legacy nested format first.
    if (!empty($config['property_restrictions']) && is_array($config['property_restrictions'])) {
      return $config['property_restrictions'];
    }

    $restrictions = [];

    // Find all prop_*_restriction keys.
    foreach ($config as $key => $value) {
      if (!str_starts_with($key, 'prop_') || !str_ends_with($key, '_restriction')) {
        continue;
      }

      // Extract property name: prop_PROPNAME_restriction -> PROPNAME.
      $propName = substr($key, 5, -12);
      if (empty($propName)) {
        continue;
      }

      $restriction = $value ?? '';
      $values = $config['prop_' . $propName . '_values'] ?? '';
      $hidden = !empty($config['prop_' . $propName . '_hidden']);
      $overrideDescEnabled = !empty($config['prop_' . $propName . '_override_desc_enabled']);
      $overrideDesc = $config['prop_' . $propName . '_override_desc'] ?? '';

      // Skip if nothing is set.
      if ($restriction === 'Allow all' && !$hidden && !$overrideDescEnabled) {
        continue;
      }

      $propRestriction = [];

      // Map human-readable restriction values to machine names.
      if ($restriction === 'Force value' || $restriction === 'force_value') {
        $propRestriction['action'] = 'force_value';
        $propRestriction['force_value'] = $values;
      }
      elseif ($restriction === 'Only allow certain values' || $restriction === 'only_allow') {
        $propRestriction['action'] = 'only_allow';
        $propRestriction['allowed_values'] = $values;
      }

      // Hidden checkbox.
      if ($hidden) {
        $propRestriction['hidden'] = TRUE;
      }

      // Description override (only if checkbox enabled).
      if ($overrideDescEnabled && !empty($overrideDesc)) {
        $propRestriction['description_override'] = $overrideDesc;
      }

      if (!empty($propRestriction)) {
        $restrictions[$propName] = $propRestriction;
      }
    }

    return $restrictions;
  }

  /**
   * Builds property description overrides from flattened prop_* config keys.
   *
   * FlowDrop sends: prop_PROPNAME_override_desc_enabled,
   * prop_PROPNAME_override_desc.
   * Drupal stores:
   * tool_settings[tool_id]['property_description_override'][PROPNAME].
   *
   * @param array<string, mixed> $config
   *   The node config containing flattened prop_* keys.
   *
   * @return array<string, string>
   *   Property description overrides keyed by property name.
   */
  protected function buildPropertyDescriptionOverrides(array $config): array {
    $overrides = [];

    $suffix = '_override_desc_enabled';
    foreach ($config as $key => $value) {
      if (!str_starts_with($key, 'prop_') || !str_ends_with($key, $suffix)) {
        continue;
      }

      $propName = substr($key, 5, -strlen($suffix));
      if (empty($propName)) {
        continue;
      }

      if (!empty($value)) {
        $descKey = 'prop_' . $propName . '_override_desc';
        $description = $config[$descKey] ?? '';
        if (!empty($description)) {
          $overrides[$propName] = $description;
        }
      }
    }

    return $overrides;
  }

  /**
   * Builds flat property restriction keys for Agent ModelOwner.
   *
   * Converts FlowDrop's flat prop_* keys to the format expected by
   * AI Agents ModelOwner: propName___action, propName___values, etc.
   *
   * @param array<string, mixed> $config
   *   The node config containing flattened prop_* keys.
   *
   * @return array<string, mixed>
   *   Flat array with propName___field keys for Agent ModelOwner.
   */
  protected function buildFlatPropertyRestrictions(array $config): array {
    $flatConfig = [];

    // Find all prop_*_restriction keys.
    foreach ($config as $key => $value) {
      if (!str_starts_with($key, 'prop_') || !str_ends_with($key, '_restriction')) {
        continue;
      }

      // Extract property name: prop_PROPNAME_restriction -> PROPNAME.
      $propName = substr($key, 5, -12);
      if (empty($propName)) {
        continue;
      }

      $restriction = $value ?? '';
      $values = $config['prop_' . $propName . '_values'] ?? '';
      $hidden = !empty($config['prop_' . $propName . '_hidden']);

      // Skip if nothing meaningful is set.
      if ($restriction === 'Allow all' && !$hidden) {
        continue;
      }

      // Map human-readable restriction to machine action.
      $action = match ($restriction) {
        'Force value', 'force_value' => 'force_value',
        'Only allow certain values', 'only_allow' => 'only_allow',
        default => '',
      };

      // Only add entries if there's meaningful data.
      if (!empty($action)) {
        // Replace colons with __colon__ for Agent ModelOwner parsing.
        $safePropName = str_replace(':', '__colon__', $propName);
        $flatConfig[$safePropName . '___action'] = $action;
        $flatConfig[$safePropName . '___values'] = $values;
        $flatConfig[$safePropName . '___hide_property'] = $hidden ? 1 : 0;
      }
      elseif ($hidden) {
        // Hidden without restriction action.
        $safePropName = str_replace(':', '__colon__', $propName);
        $flatConfig[$safePropName . '___hide_property'] = 1;
      }
    }

    return $flatConfig;
  }

  /**
   * Returns empty workflow structure.
   *
   * @return array<string, mixed>
   *   Empty workflow array.
   */
  protected function getEmptyWorkflow(): array {
    return [
      'id' => '',
      'label' => '',
      'nodes' => [],
      'edges' => [],
      'metadata' => [],
    ];
  }

}
