<?php

declare(strict_types=1);

namespace Drupal\Tests\flowdrop_ui_agents\Kernel;

/**
 * Tests that connecting tools to agents saves correctly via FlowDrop.
 *
 * @group flowdrop_ui_agents
 */
final class AgentToolConnectionTest extends FlowdropAgentsTestBase {

  /**
   * Tests that a single tool connection saves to the agent's tools array.
   */
  public function testSingleToolConnectionSaves(): void {
    // Create an agent with no tools.
    $agent = $this->createTestAgent('test_tool_agent', [
      'tools' => [],
    ]);

    // Create an assistant linked to this agent.
    $assistant = $this->createTestAssistant('test_tool_assistant', 'test_tool_agent');

    // Build workflow with agent connected to a tool.
    $mainNodeId = 'agent_test_tool_agent';
    $toolNodeId = 'tool_entity_bundle_list_1';

    $workflowData = $this->buildWorkflowData(
      nodes: [
        $this->createAssistantNode('test_tool_agent', [
          'label' => 'Test Assistant',
        ]),
        $this->createToolNode('tool:entity_bundle_list', $toolNodeId),
      ],
      edges: [
        $this->createEdge($mainNodeId, $toolNodeId),
      ],
      metadata: []
    );

    // Execute save.
    $request = $this->createJsonRequest($workflowData);
    $controller = $this->getSaveController();
    $response = $controller->save('test_tool_assistant', $request);

    // Verify response.
    $this->assertEquals(200, $response->getStatusCode());

    // Reload and verify tool was added.
    $reloadedAgent = $this->reloadAgent('test_tool_agent');
    $tools = $reloadedAgent->get('tools');
    $this->assertArrayHasKey('tool:entity_bundle_list', $tools);
    $this->assertTrue($tools['tool:entity_bundle_list']);
  }

  /**
   * Tests that multiple tool connections save correctly.
   */
  public function testMultipleToolConnectionsSave(): void {
    // Create an agent with no tools.
    $agent = $this->createTestAgent('test_multi_tool_agent', [
      'tools' => [],
    ]);

    // Create an assistant linked to this agent.
    $assistant = $this->createTestAssistant('test_multi_tool_assistant', 'test_multi_tool_agent');

    // Build workflow with agent connected to multiple tools.
    $mainNodeId = 'agent_test_multi_tool_agent';

    $workflowData = $this->buildWorkflowData(
      nodes: [
        $this->createAssistantNode('test_multi_tool_agent', [
          'label' => 'Test Assistant',
        ]),
        $this->createToolNode('tool:entity_bundle_list', 'tool_1'),
        $this->createToolNode('tool:entity_list', 'tool_2', [], ['x' => 300, 'y' => 200]),
        $this->createToolNode('tool:system_time', 'tool_3', [], ['x' => 300, 'y' => 300]),
      ],
      edges: [
        $this->createEdge($mainNodeId, 'tool_1'),
        $this->createEdge($mainNodeId, 'tool_2'),
        $this->createEdge($mainNodeId, 'tool_3'),
      ],
      metadata: []
    );

    // Execute save.
    $request = $this->createJsonRequest($workflowData);
    $controller = $this->getSaveController();
    $response = $controller->save('test_multi_tool_assistant', $request);

    // Verify response.
    $this->assertEquals(200, $response->getStatusCode());

    // Reload and verify all tools were added.
    $reloadedAgent = $this->reloadAgent('test_multi_tool_agent');
    $tools = $reloadedAgent->get('tools');
    $this->assertCount(3, $tools);
    $this->assertArrayHasKey('tool:entity_bundle_list', $tools);
    $this->assertArrayHasKey('tool:entity_list', $tools);
    $this->assertArrayHasKey('tool:system_time', $tools);
  }

  /**
   * Tests that removing a tool connection updates the agent.
   */
  public function testToolRemovalSaves(): void {
    // Create an agent with existing tools.
    $agent = $this->createTestAgent('test_remove_tool_agent', [
      'tools' => [
        'tool:entity_bundle_list' => TRUE,
        'tool:entity_list' => TRUE,
      ],
    ]);

    // Create an assistant linked to this agent.
    $assistant = $this->createTestAssistant('test_remove_tool_assistant', 'test_remove_tool_agent');

    // Build workflow with only ONE tool (removing entity_list).
    $mainNodeId = 'agent_test_remove_tool_agent';

    $workflowData = $this->buildWorkflowData(
      nodes: [
        $this->createAssistantNode('test_remove_tool_agent', [
          'label' => 'Test Assistant',
        ]),
        $this->createToolNode('tool:entity_bundle_list', 'tool_1'),
      ],
      edges: [
        $this->createEdge($mainNodeId, 'tool_1'),
      ],
      metadata: []
    );

    // Execute save.
    $request = $this->createJsonRequest($workflowData);
    $controller = $this->getSaveController();
    $response = $controller->save('test_remove_tool_assistant', $request);

    // Verify response.
    $this->assertEquals(200, $response->getStatusCode());

    // Reload and verify only one tool remains.
    $reloadedAgent = $this->reloadAgent('test_remove_tool_agent');
    $tools = $reloadedAgent->get('tools');
    $this->assertCount(1, $tools);
    $this->assertArrayHasKey('tool:entity_bundle_list', $tools);
    $this->assertArrayNotHasKey('tool:entity_list', $tools);
  }

  /**
   * Tests that tools without edges to agent are not added.
   */
  public function testUnconnectedToolsNotAdded(): void {
    // Create an agent with no tools.
    $agent = $this->createTestAgent('test_unconnected_agent', [
      'tools' => [],
    ]);

    // Create an assistant linked to this agent.
    $assistant = $this->createTestAssistant('test_unconnected_assistant', 'test_unconnected_agent');

    // Build workflow with a tool that is NOT connected to the main agent.
    $mainNodeId = 'agent_test_unconnected_agent';

    $workflowData = $this->buildWorkflowData(
      nodes: [
        $this->createAssistantNode('test_unconnected_agent', [
          'label' => 'Test Assistant',
        ]),
        // This tool has no edge connecting it to the main agent.
        $this->createToolNode('tool:entity_bundle_list', 'orphan_tool'),
      ],
      edges: [
        // No edge from main agent to the tool.
      ],
      metadata: []
    );

    // Execute save.
    $request = $this->createJsonRequest($workflowData);
    $controller = $this->getSaveController();
    $response = $controller->save('test_unconnected_assistant', $request);

    // Verify response.
    $this->assertEquals(200, $response->getStatusCode());

    // Reload and verify no tools were added (since none are connected).
    $reloadedAgent = $this->reloadAgent('test_unconnected_agent');
    $tools = $reloadedAgent->get('tools');
    $this->assertEmpty($tools);
  }

  /**
   * Tests that RAG search tool config saves to tool_settings.
   */
  public function testRagSearchToolConfigSaves(): void {
    // Create an agent with no tools.
    $agent = $this->createTestAgent('test_rag_agent', [
      'tools' => [],
      'tool_settings' => [],
    ]);

    // Create an assistant linked to this agent.
    $assistant = $this->createTestAssistant('test_rag_assistant', 'test_rag_agent');

    // Build workflow with RAG search tool and its config.
    $mainNodeId = 'agent_test_rag_agent';

    $workflowData = $this->buildWorkflowData(
      nodes: [
        $this->createAssistantNode('test_rag_agent', [
          'label' => 'Test Assistant',
        ]),
        $this->createToolNode('ai_search:rag_search', 'rag_tool', [
          'index' => 'my_search_index',
          'amount' => 10,
          'min_score' => 0.7,
        ]),
      ],
      edges: [
        $this->createEdge($mainNodeId, 'rag_tool'),
      ],
      metadata: []
    );

    // Execute save.
    $request = $this->createJsonRequest($workflowData);
    $controller = $this->getSaveController();
    $response = $controller->save('test_rag_assistant', $request);

    // Verify response.
    $this->assertEquals(200, $response->getStatusCode());

    // Reload and verify RAG tool config was saved to tool_settings.
    $reloadedAgent = $this->reloadAgent('test_rag_agent');
    $tools = $reloadedAgent->get('tools');
    $this->assertArrayHasKey('ai_search:rag_search', $tools);

    $toolSettings = $reloadedAgent->get('tool_settings');
    $this->assertArrayHasKey('ai_search:rag_search', $toolSettings);

    // RAG settings are now stored in property_restrictions format.
    $restrictions = $toolSettings['ai_search:rag_search']['property_restrictions'] ?? [];
    $this->assertEquals('my_search_index', $restrictions['index']['force_value']);
    $this->assertEquals(10, $restrictions['amount']['force_value']);
    $this->assertEquals(0.7, $restrictions['min_score']['force_value']);
  }

  /**
   * Tests that adding a sub-agent as a tool works correctly.
   */
  public function testSubAgentAsToolSaves(): void {
    // Create the main agent and a sub-agent.
    $mainAgent = $this->createTestAgent('test_parent_agent', [
      'tools' => [],
    ]);
    $subAgent = $this->createTestAgent('test_child_agent', [
      'system_prompt' => 'I am a sub-agent',
    ]);

    // Create an assistant linked to the main agent.
    $assistant = $this->createTestAssistant('test_parent_assistant', 'test_parent_agent');

    // Build workflow with sub-agent connected to main agent.
    $mainNodeId = 'agent_test_parent_agent';

    $workflowData = $this->buildWorkflowData(
      nodes: [
        $this->createAssistantNode('test_parent_agent', [
          'label' => 'Test Assistant',
        ]),
        $this->createSubAgentNode('test_child_agent', 'subagent_test_child_agent'),
      ],
      edges: [
        $this->createEdge($mainNodeId, 'subagent_test_child_agent'),
      ],
      metadata: []
    );

    // Execute save.
    $request = $this->createJsonRequest($workflowData);
    $controller = $this->getSaveController();
    $response = $controller->save('test_parent_assistant', $request);

    // Verify response.
    $this->assertEquals(200, $response->getStatusCode());

    // Reload and verify sub-agent was added as a tool.
    $reloadedAgent = $this->reloadAgent('test_parent_agent');
    $tools = $reloadedAgent->get('tools');
    $this->assertArrayHasKey('ai_agents::ai_agent::test_child_agent', $tools);
    $this->assertTrue($tools['ai_agents::ai_agent::test_child_agent']);
  }

  /**
   * Tests that collapsed sub-agent nodes also save correctly.
   */
  public function testCollapsedSubAgentSaves(): void {
    // Create the main agent and a sub-agent.
    $mainAgent = $this->createTestAgent('test_collapsed_parent', [
      'tools' => [],
    ]);
    $subAgent = $this->createTestAgent('test_collapsed_child', [
      'system_prompt' => 'I am a collapsed sub-agent',
    ]);

    // Create an assistant linked to the main agent.
    $assistant = $this->createTestAssistant('test_collapsed_assistant', 'test_collapsed_parent');

    // Build workflow with collapsed sub-agent node.
    $mainNodeId = 'agent_test_collapsed_parent';

    $workflowData = $this->buildWorkflowData(
      nodes: [
        $this->createAssistantNode('test_collapsed_parent', [
          'label' => 'Test Assistant',
        ]),
        // Create a collapsed sub-agent node.
        $this->createSubAgentNode('test_collapsed_child', 'collapsed_child', TRUE),
      ],
      edges: [
        $this->createEdge($mainNodeId, 'collapsed_child'),
      ],
      metadata: []
    );

    // Execute save.
    $request = $this->createJsonRequest($workflowData);
    $controller = $this->getSaveController();
    $response = $controller->save('test_collapsed_assistant', $request);

    // Verify response.
    $this->assertEquals(200, $response->getStatusCode());

    // Reload and verify sub-agent was added as a tool.
    $reloadedAgent = $this->reloadAgent('test_collapsed_parent');
    $tools = $reloadedAgent->get('tools');
    $this->assertArrayHasKey('ai_agents::ai_agent::test_collapsed_child', $tools);
  }

}
