<?php

declare(strict_types=1);

namespace Drupal\Tests\flowdrop_ui_agents\Kernel;

use Symfony\Component\HttpFoundation\Request;

/**
 * Tests that AI Assistants save correctly via FlowDrop.
 *
 * This tests the Assistant-specific fields (LLM settings, history, etc.)
 * as well as the coordination between Assistant and Agent saving.
 *
 * @group flowdrop_ui_agents
 */
final class AssistantSaveTest extends FlowdropAgentsTestBase {

  /**
   * Tests that assistant label saves correctly.
   */
  public function testAssistantLabelSaves(): void {
    // Create agent and assistant.
    $agent = $this->createTestAgent('label_test_agent');
    $assistant = $this->createTestAssistant('label_test_assistant', 'label_test_agent', [
      'label' => 'Original Label',
    ]);

    // Build workflow with updated label.
    $workflowData = $this->buildWorkflowData(
      nodes: [
        $this->createAssistantNode('label_test_agent', [
          'label' => 'Updated Assistant Label',
        ]),
      ],
      edges: [],
      metadata: []
    );

    // Execute save.
    $request = $this->createJsonRequest($workflowData);
    $controller = $this->getSaveController();
    $response = $controller->save('label_test_assistant', $request);

    // Verify.
    $this->assertEquals(200, $response->getStatusCode());

    $reloadedAssistant = $this->reloadAssistant('label_test_assistant');
    $this->assertEquals('Updated Assistant Label', $reloadedAssistant->get('label'));
  }

  /**
   * Tests that assistant instructions save correctly.
   */
  public function testAssistantInstructionsSave(): void {
    // Create agent and assistant.
    $agent = $this->createTestAgent('instr_test_agent');
    $assistant = $this->createTestAssistant('instr_test_assistant', 'instr_test_agent', [
      'instructions' => 'Original instructions',
    ]);

    // Build workflow with updated instructions.
    $workflowData = $this->buildWorkflowData(
      nodes: [
        $this->createAssistantNode('instr_test_agent', [
          'label' => 'Test',
          'instructions' => 'Updated instructions for the assistant',
        ]),
      ],
      edges: [],
      metadata: []
    );

    // Execute save.
    $request = $this->createJsonRequest($workflowData);
    $controller = $this->getSaveController();
    $response = $controller->save('instr_test_assistant', $request);

    // Verify.
    $this->assertEquals(200, $response->getStatusCode());

    $reloadedAssistant = $this->reloadAssistant('instr_test_assistant');
    $this->assertEquals('Updated instructions for the assistant', $reloadedAssistant->get('instructions'));
  }

  /**
   * Tests that assistant history settings save correctly.
   */
  public function testAssistantHistorySettingsSave(): void {
    // Create agent and assistant.
    $agent = $this->createTestAgent('history_test_agent');
    $assistant = $this->createTestAssistant('history_test_assistant', 'history_test_agent', [
      'allow_history' => FALSE,
      'history_context_length' => 2,
    ]);

    // Build workflow with updated history settings.
    $workflowData = $this->buildWorkflowData(
      nodes: [
        $this->createAssistantNode('history_test_agent', [
          'label' => 'Test',
          'allowHistory' => TRUE,
          'historyContextLength' => 10,
        ]),
      ],
      edges: [],
      metadata: []
    );

    // Execute save.
    $request = $this->createJsonRequest($workflowData);
    $controller = $this->getSaveController();
    $response = $controller->save('history_test_assistant', $request);

    // Verify.
    $this->assertEquals(200, $response->getStatusCode());

    $reloadedAssistant = $this->reloadAssistant('history_test_assistant');
    $this->assertTrue((bool) $reloadedAssistant->get('allow_history'));
    $this->assertEquals(10, $reloadedAssistant->get('history_context_length'));
  }

  /**
   * Tests that assistant LLM settings save correctly.
   */
  public function testAssistantLlmSettingsSave(): void {
    // Create agent and assistant.
    $agent = $this->createTestAgent('llm_test_agent');
    $assistant = $this->createTestAssistant('llm_test_assistant', 'llm_test_agent', [
      'llm_provider' => '__default__',
      'llm_model' => 'gpt-4',
      'llm_configuration' => [],
    ]);

    // Build workflow with updated LLM settings.
    $workflowData = $this->buildWorkflowData(
      nodes: [
        $this->createAssistantNode('llm_test_agent', [
          'label' => 'Test',
          'llmProvider' => 'openai',
          'llmModel' => 'gpt-4-turbo',
          'llmConfiguration' => [
            'temperature' => 0.7,
            'max_tokens' => 2000,
          ],
        ]),
      ],
      edges: [],
      metadata: []
    );

    // Execute save.
    $request = $this->createJsonRequest($workflowData);
    $controller = $this->getSaveController();
    $response = $controller->save('llm_test_assistant', $request);

    // Verify.
    $this->assertEquals(200, $response->getStatusCode());

    $reloadedAssistant = $this->reloadAssistant('llm_test_assistant');
    $this->assertEquals('openai', $reloadedAssistant->get('llm_provider'));
    $this->assertEquals('gpt-4-turbo', $reloadedAssistant->get('llm_model'));

    $llmConfig = $reloadedAssistant->get('llm_configuration');
    $this->assertEquals(0.7, $llmConfig['temperature']);
    $this->assertEquals(2000, $llmConfig['max_tokens']);
  }

  /**
   * Tests that assistant error message saves correctly.
   */
  public function testAssistantErrorMessageSaves(): void {
    // Create agent and assistant.
    $agent = $this->createTestAgent('error_test_agent');
    $assistant = $this->createTestAssistant('error_test_assistant', 'error_test_agent', [
      'error_message' => 'Original error message',
    ]);

    // Build workflow with updated error message.
    $workflowData = $this->buildWorkflowData(
      nodes: [
        $this->createAssistantNode('error_test_agent', [
          'label' => 'Test',
          'errorMessage' => 'Custom error: Something went wrong!',
        ]),
      ],
      edges: [],
      metadata: []
    );

    // Execute save.
    $request = $this->createJsonRequest($workflowData);
    $controller = $this->getSaveController();
    $response = $controller->save('error_test_assistant', $request);

    // Verify.
    $this->assertEquals(200, $response->getStatusCode());

    $reloadedAssistant = $this->reloadAssistant('error_test_assistant');
    $this->assertEquals('Custom error: Something went wrong!', $reloadedAssistant->get('error_message'));
  }

  /**
   * Tests that assistant roles save correctly.
   */
  public function testAssistantRolesSave(): void {
    // Create agent and assistant.
    $agent = $this->createTestAgent('roles_test_agent');
    $assistant = $this->createTestAssistant('roles_test_assistant', 'roles_test_agent', [
      'roles' => [],
    ]);

    // Build workflow with updated roles.
    $workflowData = $this->buildWorkflowData(
      nodes: [
        $this->createAssistantNode('roles_test_agent', [
          'label' => 'Test',
          'roles' => ['administrator', 'editor'],
        ]),
      ],
      edges: [],
      metadata: []
    );

    // Execute save.
    $request = $this->createJsonRequest($workflowData);
    $controller = $this->getSaveController();
    $response = $controller->save('roles_test_assistant', $request);

    // Verify.
    $this->assertEquals(200, $response->getStatusCode());

    $reloadedAssistant = $this->reloadAssistant('roles_test_assistant');
    $roles = $reloadedAssistant->get('roles');
    $this->assertContains('administrator', $roles);
    $this->assertContains('editor', $roles);
  }

  /**
   * Tests that BOTH assistant AND agent are saved together.
   *
   * This is the primary test for the dual-save behavior - both entities
   * should be updated in a single save operation.
   */
  public function testBothAssistantAndAgentSave(): void {
    // Create agent and assistant with initial values.
    $agent = $this->createTestAgent('dual_save_agent', [
      'system_prompt' => 'Original agent prompt',
      'max_loops' => 3,
    ]);
    $assistant = $this->createTestAssistant('dual_save_assistant', 'dual_save_agent', [
      'instructions' => 'Original instructions',
      'llm_model' => 'gpt-3.5-turbo',
    ]);

    // Build workflow with updates to both.
    $workflowData = $this->buildWorkflowData(
      nodes: [
        $this->createAssistantNode('dual_save_agent', [
          'label' => 'Dual Save Test',
          // Assistant fields.
          'instructions' => 'Updated instructions',
          'llmModel' => 'gpt-4',
          // Agent fields (via assistantNodeConfig).
          'systemPrompt' => 'Updated agent prompt',
          'maxLoops' => 5,
        ]),
      ],
      edges: [],
      metadata: []
    );

    // Execute save.
    $request = $this->createJsonRequest($workflowData);
    $controller = $this->getSaveController();
    $response = $controller->save('dual_save_assistant', $request);

    // Verify response indicates both were saved.
    $this->assertEquals(200, $response->getStatusCode());
    $content = $response->getContent();
    $this->assertNotFalse($content);
    $responseData = json_decode($content, TRUE);
    $this->assertTrue($responseData['success']);
    $this->assertEquals('dual_save_assistant', $responseData['assistant_id']);
    $this->assertEquals('dual_save_agent', $responseData['agent_id']);

    // Verify agent was updated.
    $reloadedAgent = $this->reloadAgent('dual_save_agent');
    $this->assertEquals('Updated agent prompt', $reloadedAgent->get('system_prompt'));
    $this->assertEquals(5, $reloadedAgent->get('max_loops'));

    // Verify assistant was updated.
    $reloadedAssistant = $this->reloadAssistant('dual_save_assistant');
    $this->assertEquals('Updated instructions', $reloadedAssistant->get('instructions'));
    $this->assertEquals('gpt-4', $reloadedAssistant->get('llm_model'));
  }

  /**
   * Tests that old action-based fields are cleared on save.
   *
   * Assistants now use agents, so pre_action_prompt and system_prompt
   * on the assistant entity should be cleared.
   */
  public function testOldFieldsCleared(): void {
    // Create agent and assistant with old-style fields populated.
    $agent = $this->createTestAgent('clear_test_agent');
    $assistant = $this->createTestAssistant('clear_test_assistant', 'clear_test_agent', [
      'pre_action_prompt' => 'Old pre-action prompt',
      'system_prompt' => 'Old system prompt on assistant',
    ]);

    // Build workflow.
    $workflowData = $this->buildWorkflowData(
      nodes: [
        $this->createAssistantNode('clear_test_agent', [
          'label' => 'Clear Test',
        ]),
      ],
      edges: [],
      metadata: []
    );

    // Execute save.
    $request = $this->createJsonRequest($workflowData);
    $controller = $this->getSaveController();
    $response = $controller->save('clear_test_assistant', $request);

    // Verify.
    $this->assertEquals(200, $response->getStatusCode());

    // These fields should be cleared (set to empty string).
    $reloadedAssistant = $this->reloadAssistant('clear_test_assistant');
    $this->assertEquals('', $reloadedAssistant->get('pre_action_prompt'));
    $this->assertEquals('', $reloadedAssistant->get('system_prompt'));
  }

  /**
   * Tests error handling for missing assistant.
   */
  public function testMissingAssistantReturns404(): void {
    $workflowData = $this->buildWorkflowData(
      nodes: [],
      edges: [],
      metadata: []
    );

    $request = $this->createJsonRequest($workflowData);
    $controller = $this->getSaveController();
    $response = $controller->save('nonexistent_assistant', $request);

    $this->assertEquals(404, $response->getStatusCode());
    $content = $response->getContent();
    $this->assertNotFalse($content);
    $responseData = json_decode($content, TRUE);
    $this->assertFalse($responseData['success']);
    $this->assertStringContainsString('not found', $responseData['error']);
  }

  /**
   * Tests error handling for invalid JSON.
   */
  public function testInvalidJsonReturns400(): void {
    // Create agent and assistant.
    $agent = $this->createTestAgent('json_test_agent');
    $assistant = $this->createTestAssistant('json_test_assistant', 'json_test_agent');

    // Create request with invalid JSON.
    $request = Request::create(
      '/api/flowdrop-agents/assistant/json_test_assistant/save',
      'POST',
      [],
      [],
      [],
      ['CONTENT_TYPE' => 'application/json'],
      'this is not valid json {'
    );

    $controller = $this->getSaveController();
    $response = $controller->save('json_test_assistant', $request);

    $this->assertEquals(400, $response->getStatusCode());
    $content = $response->getContent();
    $this->assertNotFalse($content);
    $responseData = json_decode($content, TRUE);
    $this->assertFalse($responseData['success']);
    $this->assertStringContainsString('Invalid JSON', $responseData['error']);
  }

  /**
   * Tests error handling for assistant with no linked agent.
   */
  public function testMissingAgentLinkReturns400(): void {
    // Create assistant without a linked agent (directly via storage).
    // Must include all required fields to satisfy config schema.
    $assistant = $this->entityTypeManager
      ->getStorage('ai_assistant')
      ->create([
        'id' => 'no_agent_assistant',
        'label' => 'No Agent Assistant',
        'description' => 'Test assistant',
        'ai_agent' => NULL,
        'allow_history' => '1',
        'history_context_length' => '5',
        'instructions' => 'Test',
        'llm_provider' => '__default__',
        'llm_model' => 'gpt-4',
        'llm_configuration' => [],
        'error_message' => 'Error',
        'specific_error_messages' => [],
        'roles' => [],
      ]);
    $assistant->save();

    $workflowData = $this->buildWorkflowData(
      nodes: [],
      edges: [],
      metadata: []
    );

    $request = $this->createJsonRequest($workflowData);
    $controller = $this->getSaveController();
    $response = $controller->save('no_agent_assistant', $request);

    $this->assertEquals(400, $response->getStatusCode());
    $content = $response->getContent();
    $this->assertNotFalse($content);
    $responseData = json_decode($content, TRUE);
    $this->assertFalse($responseData['success']);
    $this->assertStringContainsString('no linked agent', $responseData['error']);
  }

  /**
   * Tests complete save workflow with all fields.
   *
   * This is an integration test that saves all supported fields at once
   * to ensure they all work together.
   */
  public function testCompleteAssistantSaveWorkflow(): void {
    // Create agent and assistant.
    $agent = $this->createTestAgent('complete_test_agent');
    $assistant = $this->createTestAssistant('complete_test_assistant', 'complete_test_agent');

    // Build workflow with ALL supported fields.
    $workflowData = $this->buildWorkflowData(
      nodes: [
        $this->createAssistantNode('complete_test_agent', [
          // Assistant fields.
          'label' => 'Complete Test Assistant',
          'description' => 'A comprehensive test assistant',
          'instructions' => 'Follow these detailed instructions',
          'allowHistory' => TRUE,
          'historyContextLength' => 8,
          'llmProvider' => 'anthropic',
          'llmModel' => 'claude-3-opus',
          'llmConfiguration' => ['temperature' => 0.5],
          'errorMessage' => 'Oops! Try again.',
          'roles' => ['administrator'],
          // Agent fields.
          'systemPrompt' => 'You are a helpful assistant',
          'maxLoops' => 6,
        ]),
        // Add a tool.
        $this->createToolNode('tool:entity_bundle_list', 'complete_tool'),
      ],
      edges: [
        $this->createEdge('agent_complete_test_agent', 'complete_tool'),
      ],
      metadata: []
    );

    // Execute save.
    $request = $this->createJsonRequest($workflowData);
    $controller = $this->getSaveController();
    $response = $controller->save('complete_test_assistant', $request);

    // Verify success.
    $this->assertEquals(200, $response->getStatusCode());

    // Verify ALL assistant fields.
    $reloadedAssistant = $this->reloadAssistant('complete_test_assistant');
    $this->assertEquals('Complete Test Assistant', $reloadedAssistant->get('label'));
    $this->assertEquals('A comprehensive test assistant', $reloadedAssistant->get('description'));
    $this->assertEquals('Follow these detailed instructions', $reloadedAssistant->get('instructions'));
    $this->assertTrue((bool) $reloadedAssistant->get('allow_history'));
    $this->assertEquals(8, $reloadedAssistant->get('history_context_length'));
    $this->assertEquals('anthropic', $reloadedAssistant->get('llm_provider'));
    $this->assertEquals('claude-3-opus', $reloadedAssistant->get('llm_model'));
    $this->assertEquals('Oops! Try again.', $reloadedAssistant->get('error_message'));
    $this->assertContains('administrator', $reloadedAssistant->get('roles'));

    // Verify ALL agent fields.
    $reloadedAgent = $this->reloadAgent('complete_test_agent');
    $this->assertEquals('You are a helpful assistant', $reloadedAgent->get('system_prompt'));
    $this->assertEquals(6, $reloadedAgent->get('max_loops'));
    $this->assertTrue($reloadedAgent->get('orchestration_agent'));

    // Verify tool was added.
    $tools = $reloadedAgent->get('tools');
    $this->assertArrayHasKey('tool:entity_bundle_list', $tools);
  }

  /**
   * Tests that sub-agent tool property restrictions persist via Assistant save.
   *
   * This is the primary regression test for Issue #3566777.
   * Property restrictions on sub-agent tools must save correctly when
   * saved via the Assistant route.
   *
   * @see https://www.drupal.org/project/flowdrop_ui_agents/issues/3566777
   */
  public function testSubAgentToolPropertyRestrictionsPersist(): void {
    // Create main agent and sub-agent.
    $mainAgent = $this->createTestAgent('main_agent', [
      'tools' => [],
    ]);
    $subAgent = $this->createTestAgent('sub_agent_with_tools', [
      'tools' => ['tool:entity_bundle_list' => TRUE],
      'tool_usage_limits' => [],
    ]);

    // Create assistant linked to main agent.
    $assistant = $this->createTestAssistant('prop_test_assistant', 'main_agent');

    // Build workflow with main agent connected to sub-agent,
    // and sub-agent connected to a tool with property restrictions.
    $workflowData = $this->buildWorkflowData(
      nodes: [
        $this->createAssistantNode('main_agent', [
          'label' => 'Main Agent',
        ]),
        // Sub-agent node (expanded).
        [
          'id' => 'agent_sub_agent_with_tools',
          'type' => 'agent',
          'position' => ['x' => 300, 'y' => 100],
          'data' => [
            'nodeType' => 'agent',
            'config' => [
              'agent_id' => 'sub_agent_with_tools',
              'label' => 'Sub Agent',
            ],
            'metadata' => [
              'ownerAgentId' => 'sub_agent_with_tools',
            ],
          ],
        ],
        // Tool with property restrictions.
        $this->createToolNode('tool:entity_bundle_list', 'tool_bundle_list', [
          'tool_id' => 'tool:entity_bundle_list',
          'label' => 'List Bundles',
          'return_directly' => FALSE,
          'prop_entity_type_restriction' => 'Force value',
          'prop_entity_type_values' => 'node',
          'prop_entity_type_hidden' => FALSE,
        ]),
      ],
      edges: [
        $this->createEdge('agent_main_agent', 'agent_sub_agent_with_tools'),
        $this->createEdge('agent_sub_agent_with_tools', 'tool_bundle_list'),
      ],
      metadata: []
    );

    // Execute save via Assistant route.
    $request = $this->createJsonRequest($workflowData);
    $controller = $this->getSaveController();
    $response = $controller->save('prop_test_assistant', $request);

    // Verify response.
    $this->assertEquals(200, $response->getStatusCode());

    // Reload sub-agent and verify property restrictions persisted.
    $reloadedSubAgent = $this->reloadAgent('sub_agent_with_tools');
    $toolUsageLimits = $reloadedSubAgent->get('tool_usage_limits');

    $this->assertArrayHasKey('tool:entity_bundle_list', $toolUsageLimits);
    $toolLimits = $toolUsageLimits['tool:entity_bundle_list'];

    $this->assertArrayHasKey('entity_type', $toolLimits);
    $this->assertEquals('force_value', $toolLimits['entity_type']['action']);
    // Values must be an array for Drupal form compatibility.
    $this->assertIsArray($toolLimits['entity_type']['values']);
    $this->assertContains('node', $toolLimits['entity_type']['values']);
  }

  /**
   * Tests that "Force value" restriction saves values as array.
   *
   * The Drupal form expects values to always be an array, regardless of
   * restriction type. This ensures both force_value and only_allow work.
   */
  public function testForceValueSavesAsArray(): void {
    // Create agent and assistant.
    $agent = $this->createTestAgent('force_array_agent', [
      'tools' => ['tool:entity_bundle_list' => TRUE],
      'tool_usage_limits' => [],
    ]);
    $assistant = $this->createTestAssistant('force_array_assistant', 'force_array_agent');

    // Build workflow with force_value restriction.
    $workflowData = $this->buildWorkflowData(
      nodes: [
        $this->createAssistantNode('force_array_agent', [
          'label' => 'Force Array Test',
        ]),
        $this->createToolNode('tool:entity_bundle_list', 'tool_1', [
          'tool_id' => 'tool:entity_bundle_list',
          'prop_entity_type_restriction' => 'Force value',
          'prop_entity_type_values' => 'taxonomy_term',
        ]),
      ],
      edges: [
        $this->createEdge('agent_force_array_agent', 'tool_1'),
      ],
      metadata: []
    );

    // Execute save.
    $request = $this->createJsonRequest($workflowData);
    $controller = $this->getSaveController();
    $response = $controller->save('force_array_assistant', $request);

    $this->assertEquals(200, $response->getStatusCode());

    // Reload and verify values is an array.
    $reloadedAgent = $this->reloadAgent('force_array_agent');
    $toolUsageLimits = $reloadedAgent->get('tool_usage_limits');

    $this->assertArrayHasKey('tool:entity_bundle_list', $toolUsageLimits);
    $entityTypeLimits = $toolUsageLimits['tool:entity_bundle_list']['entity_type'];

    $this->assertEquals('force_value', $entityTypeLimits['action']);
    // Critical: values must be array, not string.
    $this->assertIsArray($entityTypeLimits['values']);
    $this->assertEquals(['taxonomy_term'], $entityTypeLimits['values']);
  }

  /**
   * Tests that "Only allow certain values" restriction saves correctly.
   */
  public function testOnlyAllowSavesCorrectly(): void {
    // Create agent and assistant.
    $agent = $this->createTestAgent('only_allow_agent', [
      'tools' => ['tool:entity_bundle_list' => TRUE],
      'tool_usage_limits' => [],
    ]);
    $assistant = $this->createTestAssistant('only_allow_assistant', 'only_allow_agent');

    // Build workflow with only_allow restriction.
    $workflowData = $this->buildWorkflowData(
      nodes: [
        $this->createAssistantNode('only_allow_agent', [
          'label' => 'Only Allow Test',
        ]),
        $this->createToolNode('tool:entity_bundle_list', 'tool_1', [
          'tool_id' => 'tool:entity_bundle_list',
          'prop_entity_type_restriction' => 'Only allow certain values',
          'prop_entity_type_values' => 'node',
        ]),
      ],
      edges: [
        $this->createEdge('agent_only_allow_agent', 'tool_1'),
      ],
      metadata: []
    );

    // Execute save.
    $request = $this->createJsonRequest($workflowData);
    $controller = $this->getSaveController();
    $response = $controller->save('only_allow_assistant', $request);

    $this->assertEquals(200, $response->getStatusCode());

    // Reload and verify.
    $reloadedAgent = $this->reloadAgent('only_allow_agent');
    $toolUsageLimits = $reloadedAgent->get('tool_usage_limits');

    $this->assertArrayHasKey('tool:entity_bundle_list', $toolUsageLimits);
    $entityTypeLimits = $toolUsageLimits['tool:entity_bundle_list']['entity_type'];

    $this->assertEquals('only_allow', $entityTypeLimits['action']);
    $this->assertIsArray($entityTypeLimits['values']);
    $this->assertEquals(['node'], $entityTypeLimits['values']);
  }

  /**
   * Tests that multiple property restrictions save correctly.
   */
  public function testMultiplePropertyRestrictionsSave(): void {
    // Create agent and assistant.
    $agent = $this->createTestAgent('multi_prop_agent', [
      'tools' => ['tool:entity_bundle_list' => TRUE],
      'tool_usage_limits' => [],
    ]);
    $assistant = $this->createTestAssistant('multi_prop_assistant', 'multi_prop_agent');

    // Build workflow with multiple property restrictions on one tool.
    $workflowData = $this->buildWorkflowData(
      nodes: [
        $this->createAssistantNode('multi_prop_agent', [
          'label' => 'Multi Prop Test',
        ]),
        $this->createToolNode('tool:entity_list', 'tool_1', [
          'tool_id' => 'tool:entity_list',
          'prop_entity_type_restriction' => 'Force value',
          'prop_entity_type_values' => 'node',
          'prop_bundle_restriction' => 'Only allow certain values',
          'prop_bundle_values' => 'article',
        ]),
      ],
      edges: [
        $this->createEdge('agent_multi_prop_agent', 'tool_1'),
      ],
      metadata: []
    );

    // Execute save.
    $request = $this->createJsonRequest($workflowData);
    $controller = $this->getSaveController();
    $response = $controller->save('multi_prop_assistant', $request);

    $this->assertEquals(200, $response->getStatusCode());

    // Reload and verify both restrictions saved.
    $reloadedAgent = $this->reloadAgent('multi_prop_agent');
    $toolUsageLimits = $reloadedAgent->get('tool_usage_limits');

    $this->assertArrayHasKey('tool:entity_list', $toolUsageLimits);
    $toolLimits = $toolUsageLimits['tool:entity_list'];

    // Verify entity_type restriction.
    $this->assertArrayHasKey('entity_type', $toolLimits);
    $this->assertEquals('force_value', $toolLimits['entity_type']['action']);
    $this->assertEquals(['node'], $toolLimits['entity_type']['values']);

    // Verify bundle restriction.
    $this->assertArrayHasKey('bundle', $toolLimits);
    $this->assertEquals('only_allow', $toolLimits['bundle']['action']);
    $this->assertEquals(['article'], $toolLimits['bundle']['values']);
  }

  /**
   * Tests that property description overrides persist via Assistant save.
   *
   * Property description overrides should save to:
   * tool_settings[tool_id]['property_restrictions'][property_name]
   * ['description_override']
   *
   * This is stored inside property_restrictions to avoid Agent
   * ModelOwner issues.
   */
  public function testPropertyDescriptionOverridePersists(): void {
    // Create agent and assistant.
    $agent = $this->createTestAgent('desc_override_agent', [
      'tools' => ['tool:entity_bundle_list' => TRUE],
      'tool_settings' => [],
    ]);
    $assistant = $this->createTestAssistant('desc_override_assistant', 'desc_override_agent');

    // Build workflow with property description override.
    $workflowData = $this->buildWorkflowData(
      nodes: [
        $this->createAssistantNode('desc_override_agent', [
          'label' => 'Desc Override Test',
        ]),
        $this->createToolNode('tool:entity_bundle_list', 'tool_1', [
          'tool_id' => 'tool:entity_bundle_list',
          'prop_entity_type_override_desc_enabled' => TRUE,
          'prop_entity_type_override_desc' => 'Custom description for entity type',
        ]),
      ],
      edges: [
        $this->createEdge('agent_desc_override_agent', 'tool_1'),
      ],
      metadata: []
    );

    // Execute save.
    $request = $this->createJsonRequest($workflowData);
    $controller = $this->getSaveController();
    $response = $controller->save('desc_override_assistant', $request);

    $this->assertEquals(200, $response->getStatusCode());

    // Reload and verify property_description_override is saved correctly.
    $reloadedAgent = $this->reloadAgent('desc_override_agent');
    $toolSettings = $reloadedAgent->get('tool_settings');

    $this->assertArrayHasKey('tool:entity_bundle_list', $toolSettings);
    $this->assertArrayHasKey('property_restrictions', $toolSettings['tool:entity_bundle_list']);
    $this->assertArrayHasKey('entity_type', $toolSettings['tool:entity_bundle_list']['property_restrictions']);
    $this->assertEquals(
      'Custom description for entity type',
      $toolSettings['tool:entity_bundle_list']['property_restrictions']['entity_type']['description_override']
    );
  }

  /**
   * Tests that property description override works with restrictions.
   *
   * A property can have both a restriction (Force value) AND a description
   * override. Both should persist correctly.
   */
  public function testPropertyDescriptionOverrideWithRestriction(): void {
    // Create agent and assistant.
    $agent = $this->createTestAgent('desc_with_restrict_agent', [
      'tools' => ['tool:entity_bundle_list' => TRUE],
      'tool_settings' => [],
      'tool_usage_limits' => [],
    ]);
    $assistant = $this->createTestAssistant('desc_with_restrict_assistant', 'desc_with_restrict_agent');

    // Build workflow with both restriction AND description override.
    $workflowData = $this->buildWorkflowData(
      nodes: [
        $this->createAssistantNode('desc_with_restrict_agent', [
          'label' => 'Desc With Restrict Test',
        ]),
        $this->createToolNode('tool:entity_bundle_list', 'tool_1', [
          'tool_id' => 'tool:entity_bundle_list',
          // Restriction.
          'prop_entity_type_restriction' => 'Force value',
          'prop_entity_type_values' => 'node',
          // Description override.
          'prop_entity_type_override_desc_enabled' => TRUE,
          'prop_entity_type_override_desc' => 'Only allows node entity type',
        ]),
      ],
      edges: [
        $this->createEdge('agent_desc_with_restrict_agent', 'tool_1'),
      ],
      metadata: []
    );

    // Execute save.
    $request = $this->createJsonRequest($workflowData);
    $controller = $this->getSaveController();
    $response = $controller->save('desc_with_restrict_assistant', $request);

    $this->assertEquals(200, $response->getStatusCode());

    // Reload and verify both restriction AND description override.
    $reloadedAgent = $this->reloadAgent('desc_with_restrict_agent');

    // Check tool_usage_limits for restriction.
    $toolUsageLimits = $reloadedAgent->get('tool_usage_limits');
    $this->assertArrayHasKey('tool:entity_bundle_list', $toolUsageLimits);
    $this->assertEquals('force_value', $toolUsageLimits['tool:entity_bundle_list']['entity_type']['action']);
    $this->assertEquals(['node'], $toolUsageLimits['tool:entity_bundle_list']['entity_type']['values']);

    // Check tool_settings for description override.
    $toolSettings = $reloadedAgent->get('tool_settings');
    $this->assertArrayHasKey('tool:entity_bundle_list', $toolSettings);
    $this->assertArrayHasKey('property_restrictions', $toolSettings['tool:entity_bundle_list']);
    $this->assertEquals(
      'Only allows node entity type',
      $toolSettings['tool:entity_bundle_list']['property_restrictions']['entity_type']['description_override']
    );
  }

  /**
   * Tests that tool description overrides persist when omitted from workflow.
   */
  public function testToolDescriptionOverridePersistsWhenOmitted(): void {
    $agent = $this->createTestAgent('desc_override_missing_agent', [
      'tools' => ['tool:entity_bundle_list' => TRUE],
      'tool_settings' => [
        'tool:entity_bundle_list' => [
          'description_override' => 'Custom tool description',
        ],
      ],
      'tool_usage_limits' => [],
    ]);
    $assistant = $this->createTestAssistant('desc_override_missing_assistant', 'desc_override_missing_agent');

    $workflowData = $this->buildWorkflowData(
      nodes: [
        $this->createAssistantNode('desc_override_missing_agent', [
          'label' => 'Description Override Missing',
        ]),
        $this->createToolNode('tool:entity_bundle_list', 'tool_1', [
          'tool_id' => 'tool:entity_bundle_list',
        ]),
      ],
      edges: [
        $this->createEdge('agent_desc_override_missing_agent', 'tool_1'),
      ],
      metadata: []
    );

    $request = $this->createJsonRequest($workflowData);
    $controller = $this->getSaveController();
    $response = $controller->save('desc_override_missing_assistant', $request);

    $this->assertEquals(200, $response->getStatusCode());

    $reloadedAgent = $this->reloadAgent('desc_override_missing_agent');
    $toolSettings = $reloadedAgent->get('tool_settings');
    $this->assertArrayHasKey('tool:entity_bundle_list', $toolSettings);
    $this->assertEquals(
      'Custom tool description',
      $toolSettings['tool:entity_bundle_list']['description_override']
    );
  }

}
