<?php

declare(strict_types=1);

namespace Drupal\Tests\flowdrop_ui_agents\Kernel;

/**
 * Tests that a single agent saves correctly via FlowDrop.
 *
 * @group flowdrop_ui_agents
 */
final class SingleAgentSaveTest extends FlowdropAgentsTestBase {

  /**
   * Tests that an agent's system prompt is updated correctly.
   */
  public function testAgentSystemPromptSaves(): void {
    // Create an agent with initial values.
    $agent = $this->createTestAgent('test_single_agent', [
      'system_prompt' => 'Original system prompt',
    ]);

    // Create an assistant linked to this agent (required for save controller).
    $assistant = $this->createTestAssistant('test_assistant', 'test_single_agent');

    // Build workflow with updated system prompt.
    $workflowData = $this->buildWorkflowData(
      nodes: [
        $this->createAssistantNode('test_single_agent', [
          'systemPrompt' => 'Updated system prompt via FlowDrop',
          'label' => 'Test Assistant',
        ]),
      ],
      edges: [],
      metadata: []
    );

    // Execute save.
    $request = $this->createJsonRequest($workflowData);
    $controller = $this->getSaveController();
    $response = $controller->save('test_assistant', $request);

    // Verify response.
    $this->assertEquals(200, $response->getStatusCode());
    $content = $response->getContent();
    $this->assertNotFalse($content);
    $responseData = json_decode($content, TRUE);
    $this->assertTrue($responseData['success']);

    // Reload and verify agent was updated.
    $reloadedAgent = $this->reloadAgent('test_single_agent');
    $this->assertEquals('Updated system prompt via FlowDrop', $reloadedAgent->get('system_prompt'));
  }

  /**
   * Tests that an agent's max_loops is updated correctly.
   */
  public function testAgentMaxLoopsSaves(): void {
    // Create an agent with initial values.
    $agent = $this->createTestAgent('test_loops_agent', [
      'max_loops' => 3,
    ]);

    // Create an assistant linked to this agent.
    $assistant = $this->createTestAssistant('test_loops_assistant', 'test_loops_agent');

    // Build workflow with updated max_loops.
    $workflowData = $this->buildWorkflowData(
      nodes: [
        $this->createAssistantNode('test_loops_agent', [
          'maxLoops' => 10,
          'label' => 'Test Assistant',
        ]),
      ],
      edges: [],
      metadata: []
    );

    // Execute save.
    $request = $this->createJsonRequest($workflowData);
    $controller = $this->getSaveController();
    $response = $controller->save('test_loops_assistant', $request);

    // Verify response.
    $this->assertEquals(200, $response->getStatusCode());

    // Reload and verify agent was updated.
    $reloadedAgent = $this->reloadAgent('test_loops_agent');
    $this->assertEquals(10, $reloadedAgent->get('max_loops'));
  }

  /**
   * Tests that an agent's description is updated correctly.
   */
  public function testAgentDescriptionSaves(): void {
    // Create an agent with initial values.
    $agent = $this->createTestAgent('test_desc_agent', [
      'description' => 'Original description',
    ]);

    // Create an assistant linked to this agent.
    $assistant = $this->createTestAssistant('test_desc_assistant', 'test_desc_agent');

    // Build workflow with updated description.
    $workflowData = $this->buildWorkflowData(
      nodes: [
        $this->createAssistantNode('test_desc_agent', [
          'description' => 'Updated description via FlowDrop',
          'label' => 'Test Assistant',
        ]),
      ],
      edges: [],
      metadata: []
    );

    // Execute save.
    $request = $this->createJsonRequest($workflowData);
    $controller = $this->getSaveController();
    $response = $controller->save('test_desc_assistant', $request);

    // Verify response.
    $this->assertEquals(200, $response->getStatusCode());

    // Reload and verify agent was updated.
    $reloadedAgent = $this->reloadAgent('test_desc_agent');
    $this->assertEquals('Updated description via FlowDrop', $reloadedAgent->get('description'));
  }

  /**
   * Tests that orchestration_agent is set to TRUE for assistants.
   */
  public function testOrchestrationAgentSetForAssistant(): void {
    // Create an agent that is NOT an orchestration agent initially.
    $agent = $this->createTestAgent('test_orch_agent', [
      'orchestration_agent' => FALSE,
    ]);

    // Create an assistant linked to this agent.
    $assistant = $this->createTestAssistant('test_orch_assistant', 'test_orch_agent');

    // Build workflow.
    $workflowData = $this->buildWorkflowData(
      nodes: [
        $this->createAssistantNode('test_orch_agent', [
          'label' => 'Test Assistant',
        ]),
      ],
      edges: [],
      metadata: []
    );

    // Execute save.
    $request = $this->createJsonRequest($workflowData);
    $controller = $this->getSaveController();
    $response = $controller->save('test_orch_assistant', $request);

    // Verify response.
    $this->assertEquals(200, $response->getStatusCode());

    // Reload and verify orchestration_agent is now TRUE.
    $reloadedAgent = $this->reloadAgent('test_orch_agent');
    $this->assertTrue($reloadedAgent->get('orchestration_agent'));
  }

  /**
   * Tests that multiple agent fields can be updated in a single save.
   */
  public function testMultipleAgentFieldsSave(): void {
    // Create an agent with initial values.
    $agent = $this->createTestAgent('test_multi_agent', [
      'system_prompt' => 'Original prompt',
      'description' => 'Original description',
      'max_loops' => 3,
    ]);

    // Create an assistant linked to this agent.
    $assistant = $this->createTestAssistant('test_multi_assistant', 'test_multi_agent');

    // Build workflow with all fields updated.
    $workflowData = $this->buildWorkflowData(
      nodes: [
        $this->createAssistantNode('test_multi_agent', [
          'systemPrompt' => 'New prompt',
          'description' => 'New description',
          'maxLoops' => 7,
          'label' => 'Test Assistant',
        ]),
      ],
      edges: [],
      metadata: []
    );

    // Execute save.
    $request = $this->createJsonRequest($workflowData);
    $controller = $this->getSaveController();
    $response = $controller->save('test_multi_assistant', $request);

    // Verify response.
    $this->assertEquals(200, $response->getStatusCode());

    // Reload and verify all fields were updated.
    $reloadedAgent = $this->reloadAgent('test_multi_agent');
    $this->assertEquals('New prompt', $reloadedAgent->get('system_prompt'));
    $this->assertEquals('New description', $reloadedAgent->get('description'));
    $this->assertEquals(7, $reloadedAgent->get('max_loops'));
  }

  /**
   * Tests save with agentConfig in metadata (alternative config source).
   */
  public function testAgentConfigFromMetadata(): void {
    // Create an agent with initial values.
    $agent = $this->createTestAgent('test_meta_agent', [
      'system_prompt' => 'Original prompt',
    ]);

    // Create an assistant linked to this agent.
    $assistant = $this->createTestAssistant('test_meta_assistant', 'test_meta_agent');

    // Build workflow with config in metadata.agentConfig.
    $workflowData = $this->buildWorkflowData(
      nodes: [
        $this->createAssistantNode('test_meta_agent', [
          'label' => 'Test Assistant',
        ]),
      ],
      edges: [],
      metadata: [
        'agentConfig' => [
          'system_prompt' => 'Prompt from metadata',
          'description' => 'Description from metadata',
        ],
      ]
    );

    // Execute save.
    $request = $this->createJsonRequest($workflowData);
    $controller = $this->getSaveController();
    $response = $controller->save('test_meta_assistant', $request);

    // Verify response.
    $this->assertEquals(200, $response->getStatusCode());

    // Reload and verify agent was updated from metadata.
    $reloadedAgent = $this->reloadAgent('test_meta_agent');
    $this->assertEquals('Prompt from metadata', $reloadedAgent->get('system_prompt'));
    $this->assertEquals('Description from metadata', $reloadedAgent->get('description'));
  }

}
