/**
 * @file
 * Displays stored messages from sessionStorage.
 *
 * This is used to display success messages after redirects from AJAX saves.
 * Matches the exact HTML structure used by Gin/Claro themes.
 */

(function (Drupal) {
  'use strict';

  // Run immediately on page load.
  (function () {
    // Check for stored message in sessionStorage.
    const storedMessage = sessionStorage.getItem('flowdrop_save_message');

    if (storedMessage) {
      try {
        const messageData = JSON.parse(storedMessage);

        // Clear the stored message immediately.
        sessionStorage.removeItem('flowdrop_save_message');

        // Display the message as a Drupal status message.
        if (messageData.message) {
          const displayMessage = function () {
            // Find or create the messages container.
            let messagesContainer = document.querySelector('[data-drupal-messages]');

            if (!messagesContainer) {
              // Create container in proper location (before main content).
              const mainContent = document.querySelector('main') ||
                                 document.querySelector('.region-content') ||
                                 document.querySelector('#content');

              if (mainContent) {
                messagesContainer = document.createElement('div');
                messagesContainer.setAttribute('data-drupal-messages', '');
                messagesContainer.className = 'messages-list';
                mainContent.parentNode.insertBefore(messagesContainer, mainContent);
              } else {
                return;
              }
            }

            // Build message HTML matching Gin theme structure.
            const messageHTML = `
              <div class="messages__wrapper">
                <div role="contentinfo" aria-labelledby="status-messages-title" class="messages-list__item messages messages--status">
                  <div class="messages__header">
                    <h2 id="status-messages-title" class="messages__title">
                      Status message
                    </h2>
                  </div>
                  <div class="messages__content">
                    ${messageData.message}
                  </div>
                  <button type="button" class="button button--dismiss js-message-button-hide" title="${Drupal.t('Hide')}">
                    <span class="icon-close"></span>
                    ${Drupal.t('Hide')}
                  </button>
                </div>
              </div>
            `;

            // Insert the message.
            messagesContainer.innerHTML = messageHTML;

            // Set up dismiss button functionality.
            const dismissButton = messagesContainer.querySelector('.js-message-button-hide');
            if (dismissButton) {
              dismissButton.addEventListener('click', function() {
                messagesContainer.innerHTML = '';
              });
            }

            // Announce for screen readers.
            if (typeof Drupal.announce === 'function') {
              Drupal.announce(messageData.message);
            }
          };

          // Display when DOM is ready.
          if (document.readyState === 'loading') {
            document.addEventListener('DOMContentLoaded', displayMessage);
          } else {
            displayMessage();
          }
        }
      } catch (e) {
        sessionStorage.removeItem('flowdrop_save_message');
      }
    }
  })();

})(Drupal);
