<?php

declare(strict_types=1);

namespace Drupal\flowdrop_ui_agents\Controller;

use Drupal\ai\AiProviderPluginManager;
use Drupal\ai_agents\Entity\AiAgent;
use Drupal\ai_assistant_api\Entity\AiAssistant;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Extension\ThemeHandlerInterface;
use Drupal\flowdrop_ui_agents\NodeType;
use Drupal\flowdrop_ui_agents\Service\AgentWorkflowMapper;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use Symfony\Component\Yaml\Yaml;

/**
 * Controller for editing AI Assistants with FlowDrop.
 *
 * This controller loads the AI Assistant's linked AI Agent and renders
 * the FlowDrop editor for it, while also providing access to Assistant-specific
 * settings.
 *
 * Assistants can only have:
 * - Other AI Agents as tools (sub-agents)
 * - RAG search (if ai_search module is enabled)
 */
final class AssistantEditorController extends ControllerBase {

  /**
   * The agent workflow mapper service.
   */
  protected AgentWorkflowMapper $agentWorkflowMapper;

  /**
   * The AI provider plugin manager (optional).
   */
  protected ?AiProviderPluginManager $aiProviderManager;

  /**
   * The theme handler service.
   */
  protected ThemeHandlerInterface $themeHandler;

  /**
   * Constructs the controller.
   */
  public function __construct(
    AgentWorkflowMapper $agentWorkflowMapper,
    ThemeHandlerInterface $themeHandler,
    ?AiProviderPluginManager $aiProviderManager = NULL,
  ) {
    $this->agentWorkflowMapper = $agentWorkflowMapper;
    $this->themeHandler = $themeHandler;
    $this->aiProviderManager = $aiProviderManager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    // Try to get ai.provider service if available.
    $aiProviderManager = NULL;
    // @phpstan-ignore if.alwaysTrue
    if ($container->has('ai.provider')) {
      $aiProviderManager = $container->get('ai.provider');
    }

    return new static(
      $container->get('flowdrop_ui_agents.agent_workflow_mapper'),
      $container->get('theme_handler'),
      $aiProviderManager
    );
  }

  /**
   * Renders the FlowDrop editor for an AI Assistant.
   *
   * @param \Drupal\ai_assistant_api\Entity\AiAssistant $ai_assistant
   *   The AI Assistant entity.
   *
   * @return array<string, mixed>
   *   The render array for the editor.
   */
  public function edit(AiAssistant $ai_assistant): array {
    // Get the linked AI Agent ID.
    // For config entities, get() returns the value directly.
    $agentId = $ai_assistant->get('ai_agent');
    if (!$agentId) {
      throw new NotFoundHttpException('This AI Assistant does not have a linked AI Agent.');
    }

    // Load the AI Agent.
    $agent = $this->entityTypeManager()->getStorage('ai_agent')->load($agentId);
    if (!$agent instanceof AiAgent) {
      throw new NotFoundHttpException('The linked AI Agent could not be found.');
    }

    // Convert the agent to workflow format.
    $workflowData = $this->agentWorkflowMapper->agentToWorkflow($agent);

    // Add assistant-specific metadata to the workflow.
    $assistantConfig = $this->getAssistantConfig($ai_assistant);
    $workflowData['metadata']['assistantConfig'] = $assistantConfig;

    // Convert the main agent node to an assistant node with different visuals.
    $workflowData = $this->convertMainNodeToAssistant($workflowData, $ai_assistant, $assistantConfig);

    // Add chatbots linked to this assistant (upstream nodes).
    $workflowData = $this->addLinkedChatbots($workflowData, $ai_assistant);

    // Get available agents (excluding self) for the sidebar.
    $availableAgents = $this->getAvailableAgents($ai_assistant->id());

    // Get RAG tools if ai_search is enabled.
    $availableRagTools = $this->getAvailableRagTools($agent);

    // Build the editor render array.
    return [
      '#type' => 'container',
      '#attributes' => [
        'id' => 'flowdrop-agents-editor',
        'class' => ['flowdrop-agents-editor-container'],
        'style' => 'height: calc(100vh - 240px); min-height: 600px; width: 100%; border: 1px solid #efefef;',
        'data-workflow-id' => $agent->id(),
        'data-assistant-id' => $ai_assistant->id(),
        'data-is-new' => 'false',
        'data-read-only' => 'false',
      ],
      '#attached' => [
        'library' => [
          'flowdrop_ui_agents/editor',
        ],
        'drupalSettings' => [
          'flowdrop_agents' => [
            'workflowId' => $agent->id(),
            'assistantId' => $ai_assistant->id(),
            'isNew' => FALSE,
            'readOnly' => FALSE,
            'workflow' => $workflowData,
            'chatbotConfigSchema' => $this->getChatbotConfigSchema(),
            // For assistants, only agents and RAG are available as tools.
            'availableTools' => $availableRagTools,
            'availableAgents' => $availableAgents,
            'toolsByCategory' => $this->groupToolsByCategory($availableAgents, $availableRagTools),
            'modelOwner' => 'ai_agents_agent',
            'modeler' => 'flowdrop_agents',
            'isAssistantMode' => TRUE,
          ],
          'modeler_api' => [
            // Use custom save endpoint for assistants.
            'save_url' => '/api/flowdrop-agents/assistant/' . $ai_assistant->id() . '/save',
            'token_url' => '/session/token',
          ],
        ],
      ],
    ];
  }

  /**
   * Gets assistant configuration for the workflow metadata.
   *
   * @param \Drupal\ai_assistant_api\Entity\AiAssistant $ai_assistant
   *   The AI Assistant entity.
   *
   * @return array<string, mixed>
   *   Assistant configuration array.
   */
  protected function getAssistantConfig(AiAssistant $ai_assistant): array {
    return [
      'id' => $ai_assistant->id(),
      'label' => $ai_assistant->label(),
      'description' => $ai_assistant->get('description') ?? '',
      'instructions' => $ai_assistant->get('instructions') ?? '',
      'system_prompt' => $ai_assistant->get('system_prompt') ?? '',
      'pre_action_prompt' => $ai_assistant->get('pre_action_prompt') ?? '',
      'allow_history' => $ai_assistant->get('allow_history') ?? 'session',
      'history_context_length' => $ai_assistant->get('history_context_length') ?? '2',
      'llm_provider' => $ai_assistant->get('llm_provider') ?? '',
      'llm_model' => $ai_assistant->get('llm_model') ?? '',
      'llm_configuration' => $ai_assistant->get('llm_configuration') ?? [],
      'error_message' => $ai_assistant->get('error_message') ?? '',
      'roles' => $ai_assistant->get('roles') ?? [],
      'use_function_calling' => $ai_assistant->get('use_function_calling') ?? FALSE,
    ];
  }

  /**
   * Gets available agents for the sidebar (excluding self).
   *
   * @param int|string|null $excludeId
   *   Optional agent ID to exclude from results.
   *
   * @return array<int, array<string, mixed>>
   *   Array of available agents.
   */
  protected function getAvailableAgents(int|string|null $excludeId): array {
    $agents = [];
    $storage = $this->entityTypeManager()->getStorage('ai_agent');
    $allAgents = $storage->loadMultiple();

    foreach ($allAgents as $agent) {
      assert($agent instanceof AiAgent);
      $agentId = $agent->id();

      // Exclude the assistant's own linked agent.
      if ($agentId === $excludeId) {
        continue;
      }

      // For config entities, get() returns the value directly.
      $description = $agent->get('description') ?? '';

      $agents[] = [
        'id' => 'ai_agent_' . $agentId,
        'name' => $agent->label(),
        'type' => 'agent',
        'supportedTypes' => ['agent'],
        'description' => $description,
        'category' => 'agents',
        'icon' => 'mdi:robot',
        'color' => 'var(--color-ref-purple-500)',
        'version' => '1.0.0',
        'enabled' => TRUE,
        'tags' => ['agent'],
        'executor_plugin' => 'ai_agents::ai_agent::' . $agentId,
        'agent_id' => $agentId,
        'tool_id' => 'ai_agents::ai_agent::' . $agentId,
        'inputs' => [
          [
            'id' => 'trigger',
            'name' => 'Trigger',
            'type' => 'input',
            'dataType' => 'trigger',
            'required' => FALSE,
            'description' => 'Trigger input',
          ],
        ],
        'outputs' => [
          [
            'id' => 'response',
            'name' => 'Response',
            'type' => 'output',
            'dataType' => 'string',
            'required' => FALSE,
            'description' => 'Agent response',
          ],
        ],
        'config' => [],
        'configSchema' => [],
      ];
    }

    usort($agents, fn(array $a, array $b): int => strcmp((string) $a['name'], (string) $b['name']));

    return $agents;
  }

  /**
   * Gets available RAG tools if ai_search module is enabled.
   *
   * @param \Drupal\ai_agents\Entity\AiAgent $agent
   *   The agent entity.
   *
   * @return array<int, array<string, mixed>>
   *   Array of available RAG tools.
   */
  protected function getAvailableRagTools(AiAgent $agent): array {
    $tools = [];

    if (!$this->moduleHandler()->moduleExists('ai_search')) {
      return $tools;
    }

    // Get current RAG settings from agent if configured.
    // For config entities, get() returns the value directly.
    $toolUsageLimits = $agent->get('tool_usage_limits') ?? [];
    $ragSettings = $toolUsageLimits['ai_search:rag_search'] ?? [];

    // Get available search indexes for RAG.
    /** @var \Drupal\Core\Entity\EntityStorageInterface $indexStorage */
    $indexStorage = $this->entityTypeManager()->getStorage('search_api_index');
    $indexes = $indexStorage->loadMultiple();
    $indexOptions = [];
    foreach ($indexes as $index) {
      $indexOptions[$index->id()] = $index->label();
    }

    $tools[] = [
      'id' => 'ai_search:rag_search',
      'name' => 'RAG Search',
      'type' => 'tool',
      'supportedTypes' => ['tool'],
      'description' => 'Search knowledge base using RAG (Retrieval-Augmented Generation)',
      'category' => 'rag',
      'icon' => 'mdi:database-search',
      'color' => 'var(--color-ref-blue-500)',
      'version' => '1.0.0',
      'enabled' => TRUE,
      'tags' => ['rag', 'search'],
      'tool_id' => 'ai_search:rag_search',
      'inputs' => [
        [
          'id' => 'tool',
          'name' => 'Tool',
          'type' => 'input',
          'dataType' => 'tool',
          'required' => FALSE,
          'description' => 'Tool connection from agent',
        ],
      ],
      'outputs' => [
        [
          'id' => 'tool',
          'name' => 'Tool',
          'type' => 'output',
          'dataType' => 'tool',
          'required' => FALSE,
          'description' => 'Tool output',
        ],
      ],
      'config' => [
        'index' => $ragSettings['index']['values'][0] ?? '',
        'amount' => $ragSettings['amount']['values'][0] ?? 5,
        'min_score' => $ragSettings['min_score']['values'][0] ?? 0.5,
      ],
      'configSchema' => [
        'type' => 'object',
        'properties' => [
          'index' => [
            'type' => 'string',
            'title' => 'RAG Database',
            'description' => 'Select which search index to use',
            'enum' => array_keys($indexOptions),
            'enumNames' => array_values($indexOptions),
          ],
          'amount' => [
            'type' => 'integer',
            'title' => 'Max Results',
            'description' => 'Maximum number of results to return',
            'default' => 5,
          ],
          'min_score' => [
            'type' => 'number',
            'title' => 'Threshold',
            'description' => 'Minimum similarity score (0-1)',
            'default' => 0.5,
            'minimum' => 0,
            'maximum' => 1,
          ],
        ],
      ],
      'indexOptions' => $indexOptions,
    ];

    return $tools;
  }

  /**
   * Converts the main agent node to an assistant node.
   *
   * This transforms the primary agent node in the workflow to have:
   * - Different visual (icon, color).
   * - Assistant-specific settings in the config panel.
   *
   * @param array<string, mixed> $workflowData
   *   The workflow data array.
   * @param \Drupal\ai_assistant_api\Entity\AiAssistant $ai_assistant
   *   The AI Assistant entity.
   * @param array<string, mixed> $assistantConfig
   *   Assistant configuration array.
   *
   * @return array<string, mixed>
   *   Modified workflow data.
   */
  protected function convertMainNodeToAssistant(array $workflowData, AiAssistant $ai_assistant, array $assistantConfig): array {
    // For config entities, get() returns the value directly.
    $agentId = $ai_assistant->get('ai_agent');
    $mainNodeId = 'agent_' . $agentId;

    foreach ($workflowData['nodes'] as &$node) {
      if ($node['id'] === $mainNodeId) {
        // Change node type to assistant.
        $node['data']['nodeType'] = 'assistant';
        $node['data']['label'] = $ai_assistant->label();

        // Update metadata for assistant visual.
        $node['data']['metadata']['id'] = 'ai_assistant';
        $node['data']['metadata']['name'] = 'AI Assistant';
        $node['data']['metadata']['type'] = 'assistant';
        $node['data']['metadata']['icon'] = 'mdi:account-voice';
        $node['data']['metadata']['color'] = 'var(--color-ref-teal-500)';
        $node['data']['metadata']['supportedTypes'] = ['assistant', 'agent', 'simple', 'default'];

        // Add assistant-specific config fields.
        $node['data']['config'] = array_merge($node['data']['config'], [
          'assistantId' => $ai_assistant->id(),
          // LLM settings.
          'llmProvider' => $assistantConfig['llm_provider'],
          'llmModel' => $assistantConfig['llm_model'],
          'llmConfiguration' => $assistantConfig['llm_configuration'],
          // History settings.
          'allowHistory' => $assistantConfig['allow_history'],
          'historyContextLength' => $assistantConfig['history_context_length'],
          // Other assistant settings.
          'instructions' => $assistantConfig['instructions'],
          'errorMessage' => $assistantConfig['error_message'],
          'roles' => $assistantConfig['roles'],
        ]);

        // Update config schema for assistant-specific fields.
        $node['data']['metadata']['configSchema'] = $this->getAssistantConfigSchema();

        break;
      }
    }

    return $workflowData;
  }

  /**
   * Adds chatbots linked to this assistant as upstream nodes.
   *
   * Finds all DeepChat blocks that reference this assistant and adds them
   * as chatbot nodes positioned to the left of the assistant node.
   *
   * @param array<string, mixed> $workflowData
   *   The current workflow data.
   * @param \Drupal\ai_assistant_api\Entity\AiAssistant $ai_assistant
   *   The AI Assistant entity.
   *
   * @return array<string, mixed>
   *   Modified workflow data with chatbot nodes added.
   */
  protected function addLinkedChatbots(array $workflowData, AiAssistant $ai_assistant): array {
    try {
      $storage = $this->entityTypeManager()->getStorage('block');
      $blocks = $storage->loadByProperties([
        'plugin' => 'ai_deepchat_block',
      ]);
    }
    catch (\Exception $e) {
      // If blocks can't be loaded, just return unchanged.
      return $workflowData;
    }

    $assistantId = $ai_assistant->id();
    $linkedChatbots = [];

    // Find chatbots linked to this assistant.
    foreach ($blocks as $block) {
      $settings = $block->get('settings') ?? [];
      if (($settings['ai_assistant'] ?? '') === $assistantId) {
        $linkedChatbots[] = $block;
      }
    }

    if (empty($linkedChatbots)) {
      return $workflowData;
    }

    // Find the assistant node to get its position.
    $assistantNodeId = 'agent_' . $ai_assistant->get('ai_agent');
    $assistantNode = NULL;
    foreach ($workflowData['nodes'] as $node) {
      if ($node['id'] === $assistantNodeId) {
        $assistantNode = $node;
        break;
      }
    }

    // Position chatbots to the left of the assistant.
    $baseX = ($assistantNode['position']['x'] ?? 100) - 350;
    $baseY = $assistantNode['position']['y'] ?? 250;
    $spacing = 200;

    foreach ($linkedChatbots as $index => $block) {
      $blockId = $block->id();
      $settings = $block->get('settings') ?? [];
      $nodeId = 'chatbot_' . $blockId;

      $label = $settings['label'] ?? $settings['bot_name'] ?? $blockId;
      $botName = $settings['bot_name'] ?? $label;

      $visibility = $block->getVisibility();
      $visibilityConfig = $this->extractVisibilityConfig($visibility);

      $chatbotNode = [
        'id' => $nodeId,
        'type' => NodeType::UNIVERSAL_NODE,
        'position' => [
          'x' => $baseX,
          'y' => $baseY + ($index * $spacing),
        ],
        'data' => [
          'nodeId' => $nodeId,
          'nodeType' => NodeType::CHATBOT,
          'label' => $label,
          'config' => [
            'label' => $label,
            'blockId' => $blockId,
            'theme' => $block->getTheme(),
            'region' => $block->getRegion(),
            'firstMessage' => $settings['first_message'] ?? '',
            'botName' => $botName,
            'botImage' => $settings['bot_image'] ?? '',
            'defaultUsername' => $settings['default_username'] ?? '',
            'useUsername' => (bool) ($settings['use_username'] ?? FALSE),
            'defaultAvatar' => $settings['default_avatar'] ?? '',
            'useAvatar' => (bool) ($settings['use_avatar'] ?? FALSE),
            'styleFile' => $settings['style_file'] ?? 'module:ai_chatbot:bard.yml',
            'width' => $settings['width'] ?? '500px',
            'height' => $settings['height'] ?? '500px',
            'placement' => $settings['placement'] ?? 'bottom-right',
            'collapseMinimal' => (bool) ($settings['collapse_minimal'] ?? FALSE),
            'showCopyIcon' => (bool) ($settings['show_copy_icon'] ?? TRUE),
            'toggleState' => $settings['toggle_state'] ?? 'remember',
            'verboseMode' => (bool) ($settings['verbose_mode'] ?? FALSE),
            'visibilityPages' => $visibilityConfig['pages'],
            'visibilityPagesNegate' => $visibilityConfig['pages_negate'],
            'visibilityResponseStatus' => $visibilityConfig['response_status'],
            'visibilityRoles' => $visibilityConfig['roles'],
            'visibilityRolesNegate' => $visibilityConfig['roles_negate'],
            'visibilityContentTypes' => $visibilityConfig['content_types'],
            'visibilityContentTypesNegate' => $visibilityConfig['content_types_negate'],
            'visibilityVocabularies' => $visibilityConfig['vocabularies'],
            'visibilityVocabulariesNegate' => $visibilityConfig['vocabularies_negate'],
          ],
          'metadata' => [
            'id' => 'chatbot:' . $blockId,
            'name' => $label,
            'description' => 'Chat input',
            'type' => 'chat',
            'category' => 'chat',
            'icon' => 'mdi:chat',
            'color' => 'var(--color-ref-purple-500)',
            'supportedTypes' => ['chatbot', 'chat'],
            'inputs' => [],
            'outputs' => [
              [
                'id' => 'trigger',
                'name' => 'Trigger',
                'type' => 'output',
                'dataType' => 'trigger',
                'required' => FALSE,
                'description' => 'Triggers the connected assistant',
              ],
            ],
            'configSchema' => $this->getChatbotConfigSchema(),
          ],
        ],
      ];

      $workflowData['nodes'][] = $chatbotNode;

      // Create edge from chatbot to assistant.
      $edge = [
        'id' => 'edge_' . $nodeId . '_to_' . $assistantNodeId,
        'source' => $nodeId,
        'sourceHandle' => $nodeId . '-output-trigger',
        'target' => $assistantNodeId,
        'targetHandle' => $assistantNodeId . '-input-trigger',
        'type' => 'default',
      ];

      $workflowData['edges'][] = $edge;
    }

    return $workflowData;
  }

  /**
   * Extracts visibility configuration from block visibility settings.
   *
   * @param array<string, mixed> $visibility
   *   The block visibility array.
   *
   * @return array<string, mixed>
   *   Extracted visibility config.
   */
  protected function extractVisibilityConfig(array $visibility): array {
    $config = [
      'pages' => '',
      'pages_negate' => FALSE,
      'response_status' => [],
      'roles' => [],
      'roles_negate' => FALSE,
      'content_types' => [],
      'content_types_negate' => FALSE,
      'vocabularies' => [],
      'vocabularies_negate' => FALSE,
    ];

    if (isset($visibility['request_path'])) {
      $config['pages'] = $visibility['request_path']['pages'] ?? '';
      $config['pages_negate'] = (bool) ($visibility['request_path']['negate'] ?? FALSE);
    }

    if (isset($visibility['response_status'])) {
      $config['response_status'] = array_values($visibility['response_status']['status_codes'] ?? []);
    }

    if (isset($visibility['user_role'])) {
      $config['roles'] = array_values($visibility['user_role']['roles'] ?? []);
      $config['roles_negate'] = (bool) ($visibility['user_role']['negate'] ?? FALSE);
    }

    if (isset($visibility['entity_bundle:node'])) {
      $config['content_types'] = array_values($visibility['entity_bundle:node']['bundles'] ?? []);
      $config['content_types_negate'] = (bool) ($visibility['entity_bundle:node']['negate'] ?? FALSE);
    }

    if (isset($visibility['entity_bundle:taxonomy_term'])) {
      $config['vocabularies'] = array_values($visibility['entity_bundle:taxonomy_term']['bundles'] ?? []);
      $config['vocabularies_negate'] = (bool) ($visibility['entity_bundle:taxonomy_term']['negate'] ?? FALSE);
    }

    return $config;
  }

  /**
   * Gets the config schema for chatbot nodes.
   *
   * @return array<string, mixed>
   *   JSON Schema format config schema.
   */
  protected function getChatbotConfigSchema(): array {
    $defaultTheme = $this->config('system.theme')->get('default');

    $themes = [];
    $themeNames = [];
    $themeRegions = [];

    foreach ($this->themeHandler->listInfo() as $name => $theme) {
      // Only include installed themes (those that have been installed).
      if (isset($theme->info['name'])) {
        $themes[] = $name;
        $themeNames[] = $theme->info['name'];

        $regionList = [];
        foreach (system_region_list($name) as $key => $label) {
          $regionList[] = ['value' => $key, 'label' => (string) $label];
        }
        $themeRegions[$name] = $regionList;
      }
    }

    $defaultRegions = $themeRegions[$defaultTheme] ?? [];
    $regions = array_column($defaultRegions, 'value');
    $regionNames = array_column($defaultRegions, 'label');

    $styles = $this->getAvailableChatbotStyles();

    $roles = [];
    $roleNames = [];
    $roleEntities = $this->entityTypeManager()->getStorage('user_role')->loadMultiple();
    foreach ($roleEntities as $rid => $role) {
      if ($rid === 'anonymous') {
        continue;
      }
      $roles[] = $rid;
      $roleNames[] = $role->label();
    }

    $contentTypes = [];
    $contentTypeNames = [];
    $nodeTypes = $this->entityTypeManager()->getStorage('node_type')->loadMultiple();
    foreach ($nodeTypes as $type) {
      $contentTypes[] = $type->id();
      $contentTypeNames[] = $type->label();
    }

    $vocabularies = [];
    $vocabularyNames = [];
    $vocabEntities = $this->entityTypeManager()->getStorage('taxonomy_vocabulary')->loadMultiple();
    foreach ($vocabEntities as $vocab) {
      $vocabularies[] = $vocab->id();
      $vocabularyNames[] = $vocab->label();
    }

    return [
      'type' => 'object',
      'properties' => [
        // Basic settings.
        'label' => [
          'type' => 'string',
          'title' => 'Title',
          'description' => 'Administrative title for this chatbot block',
        ],

        // Block placement.
        'theme' => [
          'type' => 'string',
          'title' => 'Theme',
          'description' => 'Drupal theme for the chatbot block',
          'enum' => $themes,
          'enumNames' => $themeNames,
          'default' => $defaultTheme,
          'group' => 'placement',
        ],
        'region' => [
          'type' => 'string',
          'title' => 'Region',
          'description' => 'Theme region for the chatbot block',
          'enum' => $regions,
          'enumNames' => $regionNames,
          'default' => 'footer_bottom',
          'group' => 'placement',
        ],

        // Message settings group.
        'firstMessage' => [
          'type' => 'string',
          'format' => 'textarea',
          'title' => 'First Message',
          'description' => 'The first message to start things off. Can use markdown.',
          'group' => 'messages',
        ],
        'botName' => [
          'type' => 'string',
          'title' => 'Bot Name',
          'description' => 'The name of the bot',
          'group' => 'messages',
        ],
        'botImage' => [
          'type' => 'string',
          'title' => 'Bot Image',
          'description' => 'The image/avatar of the bot',
          'group' => 'messages',
        ],
        'defaultUsername' => [
          'type' => 'string',
          'title' => 'Default User Name',
          'description' => 'The name of the user if not logged in',
          'group' => 'messages',
        ],
        'useUsername' => [
          'type' => 'boolean',
          'title' => 'Use Username',
          'description' => 'Use the username in chat messages if logged in',
          'default' => FALSE,
          'group' => 'messages',
        ],
        'defaultAvatar' => [
          'type' => 'string',
          'title' => 'Default Avatar',
          'description' => 'The avatar of the user if not logged in',
          'group' => 'messages',
        ],
        'useAvatar' => [
          'type' => 'boolean',
          'title' => 'Use Avatar',
          'description' => 'Use the avatar in chat messages if logged in',
          'default' => FALSE,
          'group' => 'messages',
        ],

        // Styling settings group.
        'styleFile' => [
          'type' => 'string',
          'title' => 'Style',
          'description' => 'The style theme of the chat window',
          'enum' => array_keys($styles),
          'enumNames' => array_values($styles),
          'default' => 'module:ai_chatbot:bard.yml',
          'group' => 'styling',
        ],
        'width' => [
          'type' => 'string',
          'title' => 'Width',
          'description' => 'The width of the chat window (e.g., 500px)',
          'default' => '500px',
          'group' => 'styling',
        ],
        'height' => [
          'type' => 'string',
          'title' => 'Height',
          'description' => 'The height of the chat window (e.g., 500px)',
          'default' => '500px',
          'group' => 'styling',
        ],
        'placement' => [
          'type' => 'string',
          'title' => 'Placement',
          'description' => 'Where the chatbot appears on screen',
          'enum' => ['toolbar', 'bottom-right', 'bottom-left'],
          'enumNames' => ['Toolbar', 'Bottom Right', 'Bottom Left'],
          'default' => 'bottom-right',
          'group' => 'styling',
        ],
        'collapseMinimal' => [
          'type' => 'boolean',
          'title' => 'Collapsed Minimal',
          'description' => 'Show a minimal toggle button when minimized',
          'default' => FALSE,
          'group' => 'styling',
        ],
        'showCopyIcon' => [
          'type' => 'boolean',
          'title' => 'Add Copy Icon',
          'description' => 'Adds a copy icon below each text for easy copying',
          'default' => TRUE,
          'group' => 'styling',
        ],

        // Advanced settings group.
        'toggleState' => [
          'type' => 'string',
          'title' => 'Toggle State',
          'description' => 'The state of the toggle button',
          'enum' => ['remember', 'open', 'close'],
          'enumNames' => ['Remember', 'Opened', 'Closed'],
          'default' => 'remember',
          'group' => 'advanced',
        ],
        'verboseMode' => [
          'type' => 'boolean',
          'title' => 'Verbose Mode',
          'description' => 'Show detailed processing information',
          'default' => FALSE,
          'group' => 'advanced',
        ],

        'visibilityPages' => [
          'type' => 'string',
          'format' => 'textarea',
          'title' => 'Pages',
          'description' => 'Enter one path per line. Use * as wildcard. Example: /node/*',
          'default' => '',
          'group' => 'visibility',
          'visibilityTab' => 'pages',
        ],
        'visibilityPagesNegate' => [
          'type' => 'boolean',
          'title' => 'Show for the listed pages',
          'default' => FALSE,
          'group' => 'visibility',
          'visibilityTab' => 'pages',
          'format' => 'radio',
          'radioOptions' => [
            ['value' => FALSE, 'label' => 'Show for the listed pages'],
            ['value' => TRUE, 'label' => 'Hide for the listed pages'],
          ],
        ],
        'visibilityResponseStatus' => [
          'type' => 'array',
          'title' => 'Response Status',
          'description' => 'Show on pages with selected response status (empty = all)',
          'items' => [
            'type' => 'string',
            'enum' => ['200', '403', '404'],
            'enumNames' => ['Success (200)', 'Access denied (403)', 'Page not found (404)'],
          ],
          'default' => [],
          'group' => 'visibility',
          'visibilityTab' => 'response_status',
          'multiple' => TRUE,
          'format' => 'checkboxes',
        ],
        'visibilityRoles' => [
          'type' => 'array',
          'title' => 'Roles',
          'description' => 'Show only to selected roles (empty = all roles)',
          'items' => [
            'type' => 'string',
            'enum' => $roles,
            'enumNames' => $roleNames,
          ],
          'default' => [],
          'group' => 'visibility',
          'visibilityTab' => 'roles',
          'multiple' => TRUE,
          'format' => 'checkboxes',
        ],
        'visibilityRolesNegate' => [
          'type' => 'boolean',
          'title' => 'Negate',
          'description' => 'Show to all roles EXCEPT those selected',
          'default' => FALSE,
          'group' => 'visibility',
          'visibilityTab' => 'roles',
        ],
        'visibilityContentTypes' => [
          'type' => 'array',
          'title' => 'Content Types',
          'description' => 'Show only on selected content types (empty = all)',
          'items' => [
            'type' => 'string',
            'enum' => $contentTypes,
            'enumNames' => $contentTypeNames,
          ],
          'default' => [],
          'group' => 'visibility',
          'visibilityTab' => 'content_types',
          'multiple' => TRUE,
          'format' => 'checkboxes',
        ],
        'visibilityContentTypesNegate' => [
          'type' => 'boolean',
          'title' => 'Negate',
          'description' => 'Show on all content types EXCEPT those selected',
          'default' => FALSE,
          'group' => 'visibility',
          'visibilityTab' => 'content_types',
        ],
        'visibilityVocabularies' => [
          'type' => 'array',
          'title' => 'Vocabulary',
          'description' => 'Show only on taxonomy term pages of selected vocabularies (empty = all)',
          'items' => [
            'type' => 'string',
            'enum' => $vocabularies,
            'enumNames' => $vocabularyNames,
          ],
          'default' => [],
          'group' => 'visibility',
          'visibilityTab' => 'vocabulary',
          'multiple' => TRUE,
          'format' => 'checkboxes',
        ],
        'visibilityVocabulariesNegate' => [
          'type' => 'boolean',
          'title' => 'Negate',
          'description' => 'Show on all vocabularies EXCEPT those selected',
          'default' => FALSE,
          'group' => 'visibility',
          'visibilityTab' => 'vocabulary',
        ],
      ],
      'groups' => [
        'placement' => [
          'title' => 'Block Placement',
          'collapsed' => FALSE,
        ],
        'messages' => [
          'title' => 'Message Settings',
          'collapsed' => TRUE,
        ],
        'styling' => [
          'title' => 'Styling Settings',
          'collapsed' => TRUE,
        ],
        'advanced' => [
          'title' => 'Advanced Settings',
          'collapsed' => TRUE,
        ],
        'visibility' => [
          'title' => 'Visibility',
          'collapsed' => TRUE,
          'description' => 'Configure where this chatbot appears',
        ],
      ],
      'themeRegions' => $themeRegions,
    ];
  }

  /**
   * Gets available chatbot styles from modules and themes.
   *
   * @return array<string, string>
   *   Array of style keys to human-readable names.
   */
  protected function getAvailableChatbotStyles(): array {
    $styles = [];
    $moduleHandler = $this->moduleHandler();

    $moduleList = ['ai_chatbot'];
    $moduleHandler->alter('ai_chatbot_style_modules', $moduleList);

    foreach ($moduleList as $moduleName) {
      if (!$moduleHandler->moduleExists($moduleName)) {
        continue;
      }
      $modulePath = $moduleHandler->getModule($moduleName)->getPath();
      $styles += $this->getStylesFromPath($modulePath . '/deepchat_styles', 'module:' . $moduleName);
    }

    foreach ($this->themeHandler->listInfo() as $theme) {
      $styles += $this->getStylesFromPath($theme->getPath() . '/deepchat_styles', 'theme:' . $theme->getName());
    }

    if (empty($styles)) {
      $styles = [
        'module:ai_chatbot:bard.yml' => 'Bard',
        'module:ai_chatbot:chatgpt.yml' => 'ChatGPT',
        'module:ai_chatbot:toolbar.yml' => 'Toolbar',
      ];
    }

    return $styles;
  }

  /**
   * Loads styles from a directory path.
   *
   * @param string $path
   *   The directory path to scan.
   * @param string $prefix
   *   The prefix to use for style keys.
   *
   * @return array<string, string>
   *   Array of style keys to names.
   */
  protected function getStylesFromPath(string $path, string $prefix = ''): array {
    $styles = [];

    if (!is_dir($path)) {
      return $styles;
    }

    $files = scandir($path);
    if ($files === FALSE) {
      return $styles;
    }

    foreach ($files as $file) {
      if (!preg_match('/\.ya?ml$/', $file)) {
        continue;
      }
      $contents = file_get_contents($path . '/' . $file);
      if ($contents === FALSE) {
        continue;
      }
      $style = Yaml::parse($contents);
      if (isset($style['name']) && isset($style['parameters'])) {
        $key = $prefix ? $prefix . ':' . $file : $file;
        $styles[$key] = (string) $style['name'];
      }
    }

    return $styles;
  }

  /**
   * Returns JSON Schema for assistant node configuration.
   *
   * @return array<string, mixed>
   *   JSON Schema array.
   */
  protected function getAssistantConfigSchema(): array {
    // Get available LLM providers.
    $providers = $this->getAvailableLlmProviders();
    $providerOptions = array_keys($providers);
    $providerNames = array_values($providers);

    // Get available models for chat operation type.
    $available_models = $this->getAvailableModelsForOperationType('chat');
    $model_ids = array_keys($available_models);
    $model_options = [];

    foreach ($available_models as $model_id => $model_info) {
      $model_options[] = [
        'value' => $model_id,
        'label' => $model_info['name'] . ' (' . $model_info['provider'] . ')',
      ];
    }

    // Get default model from AI settings.
    $default_model = $this->getDefaultModelForOperationType('chat');
    if (empty($default_model) && !empty($model_ids)) {
      $default_model = $model_ids[0];
    }

    return [
      'type' => 'object',
      'properties' => [
        'label' => [
          'type' => 'string',
          'title' => 'Label',
          'description' => 'Human-readable name for the assistant',
        ],
        'description' => [
          'type' => 'string',
          'title' => 'Description',
          'description' => 'Description of what this assistant does',
        ],
        'instructions' => [
          'type' => 'string',
          'format' => 'multiline',
          'title' => 'Instructions',
          'description' => 'Additional instructions for the assistant',
        ],
        'systemPrompt' => [
          'type' => 'string',
          'format' => 'multiline',
          'title' => 'System Prompt',
          'description' => 'Core instructions for assistant behavior',
        ],
        'llmProvider' => [
          'type' => 'string',
          'title' => 'LLM Provider',
          'description' => 'AI provider to use (e.g., OpenAI, Anthropic)',
          'enum' => $providerOptions,
          'enumNames' => $providerNames,
        ],
        'llmModel' => [
          'type' => 'string',
          'title' => 'LLM Model',
          'description' => 'AI model to use for chat',
          'default' => $default_model,
          'enum' => $model_ids,
          'options' => $model_options,
        ],
        'allowHistory' => [
          'type' => 'string',
          'title' => 'Conversation History',
          'description' => 'How to handle conversation history',
          'enum' => ['none', 'session', 'persistent'],
          'enumNames' => ['None', 'Session Only', 'Persistent'],
          'default' => 'session',
        ],
        'historyContextLength' => [
          'type' => 'string',
          'title' => 'History Length',
          'description' => 'Number of previous messages to include',
          'enum' => ['0', '2', '5', '10', '20', '50'],
          'enumNames' => ['0', '2', '5', '10', '20', '50'],
          'default' => '2',
        ],
        'errorMessage' => [
          'type' => 'string',
          'title' => 'Error Message',
          'description' => 'Message shown when an error occurs',
        ],
        'maxLoops' => [
          'type' => 'integer',
          'title' => 'Max Loops',
          'description' => 'Maximum iterations before stopping (1-100)',
          'default' => 3,
        ],
      ],
      'required' => ['label', 'description'],
    ];
  }

  /**
   * Gets available LLM providers.
   *
   * @return array<string, string>
   *   Array of provider IDs to labels.
   */
  protected function getAvailableLlmProviders(): array {
    $providers = [];

    // Try to get providers from ai_provider plugin manager.
    if ($this->aiProviderManager !== NULL) {
      try {
        $definitions = $this->aiProviderManager->getDefinitions();
        foreach ($definitions as $id => $definition) {
          $providers[$id] = (string) ($definition['label'] ?? $id);
        }
      }
      catch (\Exception $e) {
        // Fall back to common providers if service call fails.
      }
    }

    // Always include common providers as fallback.
    if (empty($providers)) {
      $providers = [
        'openai' => 'OpenAI',
        'anthropic' => 'Anthropic',
        'ollama' => 'Ollama',
      ];
    }

    return $providers;
  }

  /**
   * Get available models for a specific operation type.
   *
   * @param string $operation_type
   *   The operation type (e.g., 'chat', 'text_to_speech', 'embeddings').
   * @param array<string, mixed> $capabilities
   *   Optional capabilities to filter by.
   *
   * @return array<string, array<string, mixed>>
   *   Array of available models for the operation type.
   */
  protected function getAvailableModelsForOperationType(string $operation_type, array $capabilities = []): array {
    $models = [];

    if ($this->aiProviderManager === NULL) {
      return $models;
    }

    $provider_definitions = $this->aiProviderManager->getDefinitions();

    foreach ($provider_definitions as $provider_id => $provider_definition) {
      try {
        // Ensure provider_id is a string.
        $provider_id = (string) $provider_id;

        // Create provider instance.
        $provider = $this->aiProviderManager->createInstance($provider_id);

        // Check if provider supports this operation type.
        // @phpstan-ignore method.notFound
        if (!$provider->isUsable($operation_type, $capabilities)) {
          continue;
        }

        // Get configured models for this operation type.
        // @phpstan-ignore method.notFound
        $provider_models = $provider->getConfiguredModels($operation_type, $capabilities);

        if (!is_array($provider_models)) {
          continue;
        }

        foreach ($provider_models as $model_id => $model_name) {
          // Ensure model_id is a string.
          $model_id = (string) $model_id;
          $model_name = (string) $model_name;

          // Build model configuration.
          $models[$model_id] = [
            'id' => $model_id,
            'name' => $model_name,
            'provider' => $provider_id,
            'operation_type' => $operation_type,
          ];
        }
      }
      catch (\Exception) {
        // Continue with other providers if one fails.
        continue;
      }
    }

    return $models;
  }

  /**
   * Get the default model ID for a given operation type.
   *
   * @param string $operation_type
   *   The operation type (e.g., 'chat', 'completion').
   *
   * @return string|null
   *   The default model ID or null if not configured.
   */
  protected function getDefaultModelForOperationType(string $operation_type): ?string {
    if ($this->aiProviderManager === NULL) {
      return NULL;
    }

    try {
      $default_config = $this->aiProviderManager->getDefaultProviderForOperationType($operation_type);
      if (is_array($default_config) && isset($default_config['model_id'])) {
        return (string) $default_config['model_id'];
      }
      return NULL;
    }
    catch (\Exception) {
      return NULL;
    }
  }

  /**
   * Groups tools by category for sidebar display.
   *
   * @param array<int, array<string, mixed>> $agents
   *   Array of agent tools.
   * @param array<int, array<string, mixed>> $ragTools
   *   Array of RAG tools.
   *
   * @return array<string, array<int, array<string, mixed>>>
   *   Tools grouped by category.
   */
  protected function groupToolsByCategory(array $agents, array $ragTools): array {
    $grouped = [];

    if (!empty($agents)) {
      $grouped['agents'] = $agents;
    }

    if (!empty($ragTools)) {
      $grouped['rag'] = $ragTools;
    }

    return $grouped;
  }

  /**
   * Title callback for the edit page.
   */
  public function editTitle(AiAssistant $ai_assistant): string {
    return (string) $this->t('Edit @label with FlowDrop', ['@label' => $ai_assistant->label()]);
  }

}
