<?php

declare(strict_types=1);

namespace Drupal\flowdrop_ui_agents\Hook;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;
use Psr\Log\LoggerInterface;

/**
 * Entity operation hooks for FlowDrop UI Agents.
 */
class EntityOperations {

  use StringTranslationTrait;

  /**
   * The logger channel.
   */
  protected LoggerInterface $logger;

  /**
   * Constructs the EntityOperations hook handler.
   *
   * @param \Drupal\Core\Session\AccountInterface $currentUser
   *   The current user.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerFactory
   *   The logger factory.
   */
  public function __construct(
    protected AccountInterface $currentUser,
    protected EntityTypeManagerInterface $entityTypeManager,
    LoggerChannelFactoryInterface $loggerFactory,
  ) {
    $this->logger = $loggerFactory->get('flowdrop_ui_agents');
  }

  /**
   * Implements hook_entity_operation().
   *
   * Adds "Edit with FlowDrop" operation for ai_assistant entities.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity to add operations for.
   *
   * @return array<string, mixed>
   *   Array of entity operations.
   */
  #[Hook('entity_operation')]
  public function entityOperation(EntityInterface $entity): array {
    $operations = [];
    $entityType = $entity->getEntityTypeId();

    // Check permission for AI Agent editing.
    $permission = 'modeler api edit ai_agent with flowdrop_agents';
    if (!$this->currentUser->hasPermission($permission)) {
      return $operations;
    }

    if ($entityType === 'ai_assistant') {
      $agentId = NULL;
      try {
        // @phpstan-ignore method.notFound
        $agentId = $entity->get('ai_agent');
      }
      catch (\Exception $e) {
        $this->logger->error(
          'Failed to retrieve ai_agent property from entity {type}:{id} — @error',
          [
            '{type}' => $entityType,
            '{id}' => $entity->id(),
            '@error' => $e->getMessage(),
          ]
        );
      }
      if ($agentId) {
        // Verify the linked agent exists.
        $agent = $this->entityTypeManager->getStorage('ai_agent')->load($agentId);
        if ($agent) {
          // Add option to edit the assistant (which includes its agent).
          $operations['edit_flowdrop_agents'] = [
            'title' => $this->t('Edit with FlowDrop for AI Agents'),
            'url' => Url::fromRoute('flowdrop_ui_agents.assistant.edit', [
              'ai_assistant' => $entity->id(),
            ]),
            'weight' => 10,
          ];
        }
      }
    }

    return $operations;
  }

  /**
   * Implements hook_entity_operation_alter().
   *
   * Modifies entity operations for ai_agent entities:
   * - Changes "Edit" operation to point to FlowDrop UI.
   * - Adds "Configure" operation for form-based editing.
   * - Removes "Edit with modeler" operation (FlowDrop UI is now the default).
   *
   * @param array<string, mixed> $operations
   *   Array of entity operations.
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity for which operations are being altered.
   */
  #[Hook('entity_operation_alter')]
  public function entityOperationAlter(array &$operations, EntityInterface $entity): void {
    $entityType = $entity->getEntityTypeId();

    // Check permission for AI Agent editing.
    $permission = 'modeler api edit ai_agent with flowdrop_agents';
    if (!$this->currentUser->hasPermission($permission)) {
      return;
    }

    if ($entityType === 'ai_agent') {
      // Change the existing "Edit" operation to point to FlowDrop UI.
      if (isset($operations['edit'])) {
        $operations['edit']['url'] = Url::fromRoute('entity.ai_agent.edit_with.flowdrop_agents', [
          'ai_agent' => $entity->id(),
        ]);
      }

      // Remove "Edit with modeler" since FlowDrop UI is now the default.
      if (isset($operations['edit_with_modeler'])) {
        unset($operations['edit_with_modeler']);
      }

      // Add "Configure" operation for form-based editing.
      $operations['configure'] = [
        'title' => $this->t('Configure'),
        'url' => Url::fromRoute('entity.ai_agent.edit_form', [
          'ai_agent' => $entity->id(),
        ], [
          'query' => ['destination' => '/admin/config/ai/agents'],
        ]),
        'weight' => 10,
      ];
    }
  }

  /**
   * Implements hook_modeler_api_model_owner_info_alter().
   *
   * Modifies the AI Agent model owner plugin to prioritize FlowDrop UI:
   * - Changes "New AI Agent with modeler" to just "New AI Agent"
   *
   * @param array<string, mixed> $definitions
   *   Array of plugin definitions.
   */
  #[Hook('modeler_api_model_owner_info_alter')]
  public function modelerApiModelOwnerInfoAlter(array &$definitions): void {
    // Modify the ai_agents_agent plugin to use FlowDrop UI as default.
    if (isset($definitions['ai_agents_agent'])) {
      // Change the modeler label to be the primary "New AI Agent" link.
      $definitions['ai_agents_agent']['uiLabelNewModelWithModeler'] = $this->t('New AI Agent');
    }
  }

  /**
   * Implements hook_menu_local_actions_alter().
   *
   * Removes the form-based "New AI Agent" action link.
   *
   * @param array<string, mixed> $local_actions
   *   Array of local actions.
   */
  #[Hook('menu_local_actions_alter')]
  public function menuLocalActionsAlter(array &$local_actions): void {
    // Remove all actions that route to the form-based add.
    foreach ($local_actions as $key => $action) {
      // Remove the form-based "New AI Agent" action by route name.
      if (isset($action['route_name']) && $action['route_name'] === 'entity.ai_agent.add_form') {
        unset($local_actions[$key]);
      }
    }
  }

}
