<?php

declare(strict_types=1);

namespace Drupal\flowdrop_ui_agents\Plugin\ModelerApiModeler;

use Drupal\Component\Utility\Random;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Config\Entity\ConfigEntityInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\ai_agents\Entity\AiAgent;
use Drupal\flowdrop_ui_agents\NodeType;
use Drupal\flowdrop_ui_agents\Service\AgentWorkflowMapper;
use Drupal\flowdrop_ui_agents\Service\WorkflowParser;
use Drupal\modeler_api\Api;
use Drupal\modeler_api\Attribute\Modeler;
use Drupal\modeler_api\Component;
use Drupal\modeler_api\Plugin\ModelerApiModelOwner\ModelOwnerInterface;
use Drupal\modeler_api\Plugin\ModelerApiModeler\ModelerBase;
use Drupal\modeler_api\Plugin\ModelerApiModeler\ModelerInterface;

/**
 * FlowDrop modeler plugin specifically for AI Agents.
 *
 * This modeler provides a FlowDrop-based visual editor for AI Agents,
 * converting between AI Agent config entities and FlowDrop workflow format.
 */
#[Modeler(
  id: "flowdrop_agents",
  label: new TranslatableMarkup("FlowDrop for AI Agents"),
  description: new TranslatableMarkup("Visual editor for AI Agents using FlowDrop UI.")
)]
class FlowDropAgents extends ModelerBase {

  /**
   * The agent workflow mapper service.
   */
  protected AgentWorkflowMapper $agentWorkflowMapper;

  /**
   * The workflow parser service.
   */
  protected WorkflowParser $workflowParser;

  /**
   * Parsed workflow data from raw JSON.
   *
   * @var array<string, mixed>
   */
  protected array $parsedData = [];

  /**
   * Normalizes tool node descriptions in workflow data.
   *
   * Ensures config.toolDescription and metadata.description reflect the
   * canonical tool description (not property overrides).
   *
   * @param array<string, mixed> $data
   *   Workflow data.
   * @param array<string, mixed> $toolSettings
   *   Optional tool settings array containing description overrides.
   *
   * @return array<string, mixed>
   *   Normalized workflow data.
   */
  protected function normalizeToolDescriptions(array $data, array $toolSettings = []): array {
    $nodes = $data['nodes'] ?? [];
    if (empty($nodes)) {
      return $data;
    }

    $toolManager = $this->getContainer()->get('plugin.manager.ai.function_calls');

    foreach ($nodes as $index => $node) {
      $nodeType = $node['data']['nodeType'] ?? '';
      if ($nodeType !== 'tool') {
        continue;
      }

      $config = $node['data']['config'] ?? [];
      $metadata = $node['data']['metadata'] ?? [];
      $toolId = $node['data']['toolId']
        ?? $config['tool_id']
        ?? $metadata['tool_id']
        ?? '';

      if (empty($toolId)) {
        continue;
      }

      try {
        $definition = $toolManager->getDefinition($toolId);
      }
      catch (\Exception $e) {
        continue;
      }

      $description = (string) ($definition['description'] ?? '');
      if ($description === '') {
        continue;
      }

      if (is_array($config)) {
        $config['toolDescription'] = $description;

        if (!empty($toolSettings[$toolId]['description_override'])) {
          $config['description_override_enabled'] = TRUE;
          $config['description_override'] = $toolSettings[$toolId]['description_override'];
        }

        $propDescOverrides = $toolSettings[$toolId]['property_description_override'] ?? [];
        if (!empty($propDescOverrides) && is_array($propDescOverrides)) {
          foreach ($propDescOverrides as $propName => $override) {
            if ($override === '' || $override === NULL) {
              continue;
            }
            $config['prop_' . $propName . '_override_desc_enabled'] = TRUE;
            $config['prop_' . $propName . '_override_desc'] = $override;
          }
        }

        $node['data']['config'] = $config;
      }
      if (is_array($metadata)) {
        $metadata['description'] = $description;
        $node['data']['metadata'] = $metadata;
      }

      $nodes[$index] = $node;
    }

    $data['nodes'] = $nodes;
    return $data;
  }

  /**
   * Get the agent workflow mapper service.
   */
  protected function agentWorkflowMapper(): AgentWorkflowMapper {
    if (!isset($this->agentWorkflowMapper)) {
      $this->agentWorkflowMapper = $this->getContainer()->get('flowdrop_ui_agents.agent_workflow_mapper');
    }
    return $this->agentWorkflowMapper;
  }

  /**
   * Get the workflow parser service.
   */
  protected function workflowParser(): WorkflowParser {
    if (!isset($this->workflowParser)) {
      $this->workflowParser = $this->getContainer()->get('flowdrop_ui_agents.workflow_parser');
    }
    return $this->workflowParser;
  }

  /**
   * {@inheritdoc}
   */
  public function getRawFileExtension(): ?string {
    return 'json';
  }

  /**
   * {@inheritdoc}
   */
  public function isEditable(): bool {
    return TRUE;
  }

  /**
   * {@inheritdoc}
   *
   * @return array<string, mixed>
   *   Render array for the editor.
   */
  public function edit(ModelOwnerInterface $owner, string $id, string $data, bool $isNew = FALSE, bool $readOnly = FALSE): array {
    // Parse existing data if available.
    $workflow = [];
    if (!empty($data)) {
      $workflow = json_decode($data, TRUE) ?? [];
    }
    $toolSettings = [];
    try {
      $storage = $this->getContainer()->get('entity_type.manager')->getStorage($owner->configEntityTypeId());
      $model = $storage->load($id);
      if ($model instanceof ConfigEntityInterface) {
        $toolSettings = $model->get('tool_settings') ?? [];
      }
    }
    catch (\Exception $e) {
      $toolSettings = [];
    }

    $workflow = $this->normalizeToolDescriptions($workflow, $toolSettings);

    // Get available tools and agents for the sidebar.
    $mapper = $this->agentWorkflowMapper();
    $availableTools = $mapper->getAvailableTools($owner);
    $availableAgents = $mapper->getAvailableAgents($owner);
    $toolsByCategory = $mapper->getToolsByCategory($owner);

    $build = [
      '#type' => 'container',
      '#attributes' => [
        'id' => 'flowdrop-agents-editor',
        'class' => ['flowdrop-agents-editor-container'],
        'style' => 'height: calc(100vh - 240px); min-height: 600px; width: 100%; border: 1px solid #efefef;',
        'data-workflow-id' => $id,
        'data-is-new' => $isNew ? 'true' : 'false',
        'data-read-only' => $readOnly ? 'true' : 'false',
      ],
      '#attached' => [
        'library' => [
          'flowdrop_ui_agents/editor',
        ],
        'drupalSettings' => [
          'flowdrop_agents' => [
            'workflowId' => $id,
            'isNew' => $isNew,
            'readOnly' => $readOnly,
            'workflow' => $workflow,
            'availableTools' => $availableTools,
            'availableAgents' => $availableAgents,
            'toolsByCategory' => $toolsByCategory,
            'modelOwner' => $owner->getPluginId(),
            'modeler' => 'flowdrop_agents',
          ],
        ],
      ],
    ];

    return $build;
  }

  /**
   * {@inheritdoc}
   *
   * @return array<string, mixed>
   *   Render array for the editor.
   */
  public function convert(ModelOwnerInterface $owner, ConfigEntityInterface $model, bool $readOnly = FALSE): array {
    // Convert AI Agent entity to FlowDrop workflow format.
    $mapper = $this->agentWorkflowMapper();
    $workflowData = $mapper->agentToWorkflow($model);
    $toolSettings = $model->get('tool_settings') ?? [];
    $workflowData = $this->normalizeToolDescriptions($workflowData, $toolSettings);

    // Get available tools and agents for sidebar.
    $availableTools = $mapper->getAvailableTools($owner);
    $availableAgents = $mapper->getAvailableAgents($owner);
    $toolsByCategory = $mapper->getToolsByCategory($owner);

    $build = [
      '#type' => 'container',
      '#attributes' => [
        'id' => 'flowdrop-agents-editor',
        'class' => ['flowdrop-agents-editor-container'],
        'style' => 'height: calc(100vh - 240px); min-height: 600px; width: 100%; border: 1px solid #efefef;',
        'data-workflow-id' => $model->id(),
        'data-is-new' => 'false',
        'data-read-only' => $readOnly ? 'true' : 'false',
      ],
      '#attached' => [
        'library' => [
          'flowdrop_ui_agents/editor',
        ],
        'drupalSettings' => [
          'flowdrop_agents' => [
            'workflowId' => $model->id(),
            'isNew' => FALSE,
            'readOnly' => $readOnly,
            'workflow' => $workflowData,
            'availableTools' => $availableTools,
            'availableAgents' => $availableAgents,
            'toolsByCategory' => $toolsByCategory,
            'modelOwner' => $owner->getPluginId(),
            'modeler' => 'flowdrop_agents',
          ],
        ],
      ],
    ];

    return $build;
  }

  /**
   * {@inheritdoc}
   */
  public function generateId(): string {
    $random = new Random();
    return 'agent_' . strtolower($random->name(8));
  }

  /**
   * {@inheritdoc}
   */
  public function enable(ModelOwnerInterface $owner): ModelerInterface {
    // Nothing special needed for enable.
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function disable(ModelOwnerInterface $owner): ModelerInterface {
    // Nothing special needed for disable.
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function clone(ModelOwnerInterface $owner, string $id, string $label): ModelerInterface {
    // Update the parsed data with new ID and label.
    if (!empty($this->parsedData)) {
      $this->parsedData['id'] = $id;
      $this->parsedData['label'] = $label;
    }
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function prepareEmptyModelData(string &$id): string {
    $id = $this->generateId();

    // Return empty workflow structure.
    $emptyWorkflow = [
      'id' => $id,
      'label' => 'New Agent',
      'nodes' => [],
      'edges' => [],
      'metadata' => [
        'agentConfig' => [
          'system_prompt' => '',
          'description' => '',
          'max_loops' => 3,
        ],
      ],
    ];

    return json_encode($emptyWorkflow) ?: '{}';
  }

  /**
   * The model owner for component creation.
   */
  protected ?ModelOwnerInterface $modelOwner = NULL;

  /**
   * {@inheritdoc}
   */
  public function parseData(ModelOwnerInterface $owner, string $data): void {
    $this->modelOwner = $owner;
    $this->parsedData = $this->workflowParser()->parse($data);
    // Set the owner on the parser for component creation.
    $this->workflowParser()->setOwner($owner);
    $this->parsedData = $this->normalizeToolDescriptions($this->parsedData);
  }

  /**
   * {@inheritdoc}
   */
  public function getId(): string {
    return $this->parsedData['id'] ?? '';
  }

  /**
   * {@inheritdoc}
   */
  public function getLabel(): string {
    return $this->parsedData['label'] ?? '';
  }

  /**
   * {@inheritdoc}
   *
   * @return array<int, string>
   *   Array of tag strings.
   */
  public function getTags(): array {
    return $this->parsedData['metadata']['tags'] ?? [];
  }

  /**
   * {@inheritdoc}
   */
  public function getChangelog(): string {
    return $this->parsedData['metadata']['changelog'] ?? '';
  }

  /**
   * {@inheritdoc}
   */
  public function getStorage(): string {
    return $this->parsedData['metadata']['storage'] ?? '';
  }

  /**
   * {@inheritdoc}
   */
  public function getDocumentation(): string {
    return $this->parsedData['metadata']['documentation'] ?? '';
  }

  /**
   * {@inheritdoc}
   */
  public function getStatus(): bool {
    return $this->parsedData['metadata']['status'] ?? TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function getVersion(): string {
    return $this->parsedData['metadata']['version'] ?? '1.0.0';
  }

  /**
   * {@inheritdoc}
   */
  public function getRawData(): string {
    return json_encode($this->parsedData) ?: '{}';
  }

  /**
   * {@inheritdoc}
   */
  public function readComponents(): array {
    $this->ensureSubAgentsExist();
    $components = $this->workflowParser()->toComponents($this->parsedData);

    // Add SUBPROCESS components for sub-agents referenced in the workflow.
    // The parser stores these in _subAgentTools during toComponents().
    $subAgentTools = $this->parsedData['_subAgentTools'] ?? [];
    foreach ($subAgentTools as $toolId) {
      // Extract the agent ID from the tool ID (ai_agents::ai_agent::agent_id).
      $subAgentId = str_replace('ai_agents::ai_agent::', '', $toolId);
      $components[] = new Component(
        $this->modelOwner,
        $subAgentId,
        Api::COMPONENT_TYPE_SUBPROCESS,
        $subAgentId,
        $subAgentId,
        [],
        [],
      );
    }

    return $components;
  }

  /**
   * Ensures any newly added sub-agents exist before saving references.
   */
  protected function ensureSubAgentsExist(): void {
    $nodes = $this->parsedData['nodes'] ?? [];
    $primaryAgentId = $this->parsedData['id'] ?? '';

    if ($primaryAgentId === '' || empty($nodes)) {
      return;
    }

    $storage = $this->getContainer()->get('entity_type.manager')->getStorage('ai_agent');

    foreach ($nodes as $node) {
      $nodeType = $node['data']['nodeType'] ?? $node['type'] ?? '';
      if ($nodeType !== NodeType::AGENT) {
        continue;
      }

      $config = $node['data']['config'] ?? [];
      $metadata = $node['data']['metadata'] ?? [];
      $ownerAgentId = $metadata['ownerAgentId'] ?? $config['agent_id'] ?? '';

      if ($ownerAgentId === '' || $ownerAgentId === $primaryAgentId) {
        continue;
      }

      if ($storage->load($ownerAgentId) instanceof AiAgent) {
        continue;
      }

      $agent = $storage->create([
        'id' => $ownerAgentId,
        'label' => $config['label'] ?? $ownerAgentId,
        'description' => $config['description'] ?? '',
        'system_prompt' => $config['systemPrompt'] ?? '',
        'tools' => [],
        'tool_usage_limits' => [],
        'tool_settings' => [],
        'orchestration_agent' => (bool) ($config['orchestrationAgent'] ?? FALSE),
        'triage_agent' => (bool) ($config['triageAgent'] ?? FALSE),
        'max_loops' => (int) ($config['maxLoops'] ?? 3),
      ]);
      $agent->save();
    }
  }

  /**
   * {@inheritdoc}
   */
  public function updateComponents(ModelOwnerInterface $owner): bool {
    // No update needed for now.
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function configForm(ModelOwnerInterface $owner): AjaxResponse {
    $response = new AjaxResponse();
    // @todo Implement config form for node configuration.
    // This should open a Drupal form in off-canvas for configuring
    // agent/tool properties.
    return $response;
  }

}
