<?php

declare(strict_types=1);

namespace Drupal\Tests\flowdrop_ui_agents\Kernel;

use Drupal\flowdrop_ui_agents\Controller\Api\AssistantSaveController;
use Drupal\ai_agents\Entity\AiAgent;
use Drupal\ai_assistant_api\Entity\AiAssistant;
use Drupal\KernelTests\KernelTestBase;
use PHPUnit\Framework\Attributes\RunTestsInSeparateProcesses;
use Symfony\Component\HttpFoundation\Request;

/**
 * Base class for FlowDrop UI Agents kernel tests.
 *
 * Provides helper methods for creating test entities and building
 * workflow JSON structures for testing the save functionality.
 *
 * @group flowdrop_ui_agents
 */
#[RunTestsInSeparateProcesses]
abstract class FlowdropAgentsTestBase extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'node',
    'field',
    'text',
    'key',
    'modeler_api',
    'ai',
    'ai_agents',
    'ai_assistant_api',
    'flowdrop_ui_agents',
  ];

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->entityTypeManager = $this->container->get('entity_type.manager');

    $this->installEntitySchema('user');
    $this->installEntitySchema('node');
    $this->installConfig(['system', 'ai_agents', 'ai_assistant_api']);
  }

  /**
   * Creates a test AI Agent entity.
   *
   * @param string $id
   *   The agent ID.
   * @param array<string, mixed> $overrides
   *   Optional values to override defaults.
   *
   * @return \Drupal\ai_agents\Entity\AiAgent
   *   The created agent entity.
   */
  protected function createTestAgent(string $id, array $overrides = []): AiAgent {
    $defaults = [
      'id' => $id,
      'label' => "Test Agent $id",
      'description' => "Test description for $id",
      'system_prompt' => 'Default test system prompt',
      'tools' => [],
      'tool_usage_limits' => [],
      'tool_settings' => [],
      'orchestration_agent' => FALSE,
      'triage_agent' => FALSE,
      'max_loops' => 3,
    ];

    $values = array_merge($defaults, $overrides);

    /** @var \Drupal\ai_agents\Entity\AiAgent $agent */
    $agent = $this->entityTypeManager
      ->getStorage('ai_agent')
      ->create($values);
    $agent->save();

    return $agent;
  }

  /**
   * Creates a test AI Assistant entity with a linked agent.
   *
   * @param string $id
   *   The assistant ID.
   * @param string $agentId
   *   The linked agent ID.
   * @param array<string, mixed> $overrides
   *   Optional values to override defaults.
   *
   * @return \Drupal\ai_assistant_api\Entity\AiAssistant
   *   The created assistant entity.
   */
  protected function createTestAssistant(string $id, string $agentId, array $overrides = []): AiAssistant {
    $defaults = [
      'id' => $id,
      'label' => "Test Assistant $id",
      'description' => "Test description for $id",
      'ai_agent' => $agentId,
      'allow_history' => '1',
      'history_context_length' => '5',
      'instructions' => 'Default test instructions',
      'llm_provider' => '__default__',
      'llm_model' => 'gpt-4',
      'llm_configuration' => [],
      'error_message' => 'An error occurred',
      'specific_error_messages' => [],
      'roles' => [],
    ];

    $values = array_merge($defaults, $overrides);

    /** @var \Drupal\ai_assistant_api\Entity\AiAssistant $assistant */
    $assistant = $this->entityTypeManager
      ->getStorage('ai_assistant')
      ->create($values);
    $assistant->save();

    return $assistant;
  }

  /**
   * Builds a workflow JSON structure for testing.
   *
   * @param array<int, array<string, mixed>> $nodes
   *   Array of node definitions.
   * @param array<int, array<string, mixed>> $edges
   *   Array of edge definitions.
   * @param array<string, mixed> $metadata
   *   Optional metadata including agentConfig.
   *
   * @return array<string, mixed>
   *   The complete workflow structure.
   */
  protected function buildWorkflowData(array $nodes, array $edges = [], array $metadata = []): array {
    return [
      'id' => 'test_workflow',
      'nodes' => $nodes,
      'edges' => $edges,
      'metadata' => $metadata,
    ];
  }

  /**
   * Creates an agent node for workflow JSON.
   *
   * @param string $agentId
   *   The agent ID.
   * @param array<string, mixed> $config
   *   Node configuration (can include systemPrompt, maxLoops, etc.).
   * @param array<string, int> $position
   *   Optional position coordinates.
   *
   * @return array<string, mixed>
   *   The node structure.
   */
  protected function createAgentNode(string $agentId, array $config = [], array $position = []): array {
    return [
      'id' => 'agent_' . $agentId,
      'type' => 'agent',
      'position' => $position ?: ['x' => 100, 'y' => 100],
      'data' => [
        'nodeType' => 'agent',
        'config' => $config,
        'metadata' => [
          'ownerAgentId' => $agentId,
        ],
      ],
    ];
  }

  /**
   * Creates an assistant node for workflow JSON.
   *
   * @param string $agentId
   *   The linked agent ID.
   * @param array<string, mixed> $config
   *   Node configuration for the assistant.
   * @param array<string, int> $position
   *   Optional position coordinates.
   *
   * @return array<string, mixed>
   *   The node structure.
   */
  protected function createAssistantNode(string $agentId, array $config = [], array $position = []): array {
    return [
      'id' => 'agent_' . $agentId,
      'type' => 'assistant',
      'position' => $position ?: ['x' => 100, 'y' => 100],
      'data' => [
        'nodeType' => 'assistant',
        'config' => $config,
        'metadata' => [
          'ownerAgentId' => $agentId,
        ],
      ],
    ];
  }

  /**
   * Creates a tool node for workflow JSON.
   *
   * @param string $toolId
   *   The tool plugin ID (e.g., 'tool:entity_bundle_list').
   * @param string $nodeId
   *   Unique node ID.
   * @param array<string, mixed> $config
   *   Optional tool configuration.
   * @param array<string, int> $position
   *   Optional position coordinates.
   *
   * @return array<string, mixed>
   *   The node structure.
   */
  protected function createToolNode(string $toolId, string $nodeId, array $config = [], array $position = []): array {
    return [
      'id' => $nodeId,
      'type' => 'tool',
      'position' => $position ?: ['x' => 300, 'y' => 100],
      'data' => [
        'nodeType' => 'tool',
        'toolId' => $toolId,
        'config' => $config,
        'metadata' => [],
      ],
    ];
  }

  /**
   * Creates a sub-agent node for workflow JSON.
   *
   * @param string $subAgentId
   *   The sub-agent ID.
   * @param string $nodeId
   *   Unique node ID.
   * @param bool $collapsed
   *   Whether to create a collapsed node.
   * @param array<string, int> $position
   *   Optional position coordinates.
   *
   * @return array<string, mixed>
   *   The node structure.
   */
  protected function createSubAgentNode(string $subAgentId, string $nodeId, bool $collapsed = FALSE, array $position = []): array {
    return [
      'id' => $nodeId,
      'type' => $collapsed ? 'agent-collapsed' : 'agent',
      'position' => $position ?: ['x' => 300, 'y' => 200],
      'data' => [
        'nodeType' => $collapsed ? 'agent-collapsed' : 'agent',
        'config' => [
          'agent_id' => $subAgentId,
        ],
        'metadata' => [
          'ownerAgentId' => $subAgentId,
        ],
      ],
    ];
  }

  /**
   * Creates an edge connecting two nodes.
   *
   * @param string $sourceId
   *   The source node ID.
   * @param string $targetId
   *   The target node ID.
   * @param string|null $edgeId
   *   Optional edge ID.
   *
   * @return array<string, string>
   *   The edge structure.
   */
  protected function createEdge(string $sourceId, string $targetId, ?string $edgeId = NULL): array {
    return [
      'id' => $edgeId ?? "edge_{$sourceId}_{$targetId}",
      'source' => $sourceId,
      'target' => $targetId,
    ];
  }

  /**
   * Creates an HTTP request with JSON body.
   *
   * @param array<string, mixed> $workflowData
   *   The workflow data to send.
   *
   * @return \Symfony\Component\HttpFoundation\Request
   *   The request object.
   */
  protected function createJsonRequest(array $workflowData): Request {
    $json = json_encode($workflowData);
    if ($json === FALSE) {
      throw new \RuntimeException('Failed to encode workflow data as JSON');
    }
    return Request::create(
      '/api/flowdrop-agents/assistant/test/save',
      'POST',
      [],
      [],
      [],
      ['CONTENT_TYPE' => 'application/json'],
      $json
    );
  }

  /**
   * Gets the AssistantSaveController instance.
   *
   * @return \Drupal\flowdrop_ui_agents\Controller\Api\AssistantSaveController
   *   The controller instance.
   */
  protected function getSaveController() {
    return AssistantSaveController::create($this->container);
  }

  /**
   * Reloads an entity from storage.
   *
   * @param string $entityType
   *   The entity type ID.
   * @param string $id
   *   The entity ID.
   *
   * @return \Drupal\Core\Entity\EntityInterface|null
   *   The reloaded entity or NULL if not found.
   */
  protected function reloadEntity(string $entityType, string $id) {
    $this->entityTypeManager->getStorage($entityType)->resetCache([$id]);
    return $this->entityTypeManager->getStorage($entityType)->load($id);
  }

  /**
   * Reloads an AI Agent from storage.
   *
   * @param string $id
   *   The agent ID.
   *
   * @return \Drupal\ai_agents\Entity\AiAgent
   *   The reloaded agent entity.
   */
  protected function reloadAgent(string $id): AiAgent {
    $this->entityTypeManager->getStorage('ai_agent')->resetCache([$id]);
    /** @var \Drupal\ai_agents\Entity\AiAgent $agent */
    $agent = $this->entityTypeManager->getStorage('ai_agent')->load($id);
    return $agent;
  }

  /**
   * Reloads an AI Assistant from storage.
   *
   * @param string $id
   *   The assistant ID.
   *
   * @return \Drupal\ai_assistant_api\Entity\AiAssistant
   *   The reloaded assistant entity.
   */
  protected function reloadAssistant(string $id): AiAssistant {
    $this->entityTypeManager->getStorage('ai_assistant')->resetCache([$id]);
    /** @var \Drupal\ai_assistant_api\Entity\AiAssistant $assistant */
    $assistant = $this->entityTypeManager->getStorage('ai_assistant')->load($id);
    return $assistant;
  }

}
