<?php

declare(strict_types=1);

namespace Drupal\Tests\flowdrop_ui_agents\Kernel;

use PHPUnit\Framework\Attributes\Group;

/**
 * Tests that multi-agent workflows save to correct places.
 *
 * This is the critical test for issue #3563756 - ensuring that when multiple
 * agents are in a workflow, each one saves its configuration to the correct
 * entity and tools don't get mixed up between agents.
 */
#[Group('flowdrop_ui_agents')]
final class MultiAgentSaveTest extends FlowdropAgentsTestBase {

  /**
   * Tests that two agents with different configs save to correct entities.
   *
   * This is the primary test for the multi-agent save bug. It ensures that
   * when you have an assistant with two sub-agents, each sub-agent's tools
   * are saved to the correct agent entity.
   */
  public function testTwoAgentsWithDifferentToolsSaveCorrectly(): void {
    // Create main agent (the assistant's underlying agent).
    $mainAgent = $this->createTestAgent('main_orchestrator', [
      'system_prompt' => 'Main orchestrator prompt',
      'tools' => [],
    ]);

    // Create two sub-agents with different initial states.
    $subAgent1 = $this->createTestAgent('sub_agent_alpha', [
      'system_prompt' => 'Alpha agent prompt',
      'tools' => [],
    ]);

    $subAgent2 = $this->createTestAgent('sub_agent_beta', [
      'system_prompt' => 'Beta agent prompt',
      'tools' => [],
    ]);

    // Create an assistant linked to the main agent.
    $assistant = $this->createTestAssistant('multi_assistant', 'main_orchestrator');

    // Build workflow where:
    // - Main agent (assistant) connects to both sub-agents
    // - Sub-agent Alpha connects to tool:entity_bundle_list
    // - Sub-agent Beta connects to tool:system_time
    // Critical: Alpha's tool should NOT end up on Beta or main agent.
    $mainNodeId = 'agent_main_orchestrator';
    $alphaNodeId = 'subagent_sub_agent_alpha';
    $betaNodeId = 'subagent_sub_agent_beta';

    $workflowData = $this->buildWorkflowData(
      nodes: [
        $this->createAssistantNode('main_orchestrator', [
          'label' => 'Multi-Agent Assistant',
          'systemPrompt' => 'Updated orchestrator prompt',
        ]),
        // Sub-agent Alpha (expanded mode - nodeType 'agent').
        $this->createSubAgentNode('sub_agent_alpha', $alphaNodeId, FALSE, ['x' => 300, 'y' => 100]),
        // Sub-agent Beta (expanded mode).
        $this->createSubAgentNode('sub_agent_beta', $betaNodeId, FALSE, ['x' => 300, 'y' => 300]),
        // Tool for Alpha.
        $this->createToolNode('tool:entity_bundle_list', 'tool_alpha', [], ['x' => 500, 'y' => 100]),
        // Tool for Beta.
        $this->createToolNode('tool:system_time', 'tool_beta', [], ['x' => 500, 'y' => 300]),
      ],
      edges: [
        // Main agent connects to both sub-agents.
        $this->createEdge($mainNodeId, $alphaNodeId),
        $this->createEdge($mainNodeId, $betaNodeId),
        // Alpha's tool connection (these should NOT affect the save
        // since we're saving the assistant, not the sub-agents directly).
        $this->createEdge($alphaNodeId, 'tool_alpha'),
        // Beta's tool connection.
        $this->createEdge($betaNodeId, 'tool_beta'),
      ],
      metadata: []
    );

    // Execute save.
    $request = $this->createJsonRequest($workflowData);
    $controller = $this->getSaveController();
    $response = $controller->save('multi_assistant', $request);

    // Verify response.
    $this->assertEquals(200, $response->getStatusCode());

    // Reload and verify main agent has BOTH sub-agents as tools
    // (not the leaf tools).
    $reloadedMain = $this->reloadAgent('main_orchestrator');
    $mainTools = $reloadedMain->get('tools');

    // Main agent should have both sub-agents as tools.
    $this->assertArrayHasKey('ai_agents::ai_agent::sub_agent_alpha', $mainTools, 'Main agent should have Alpha sub-agent');
    $this->assertArrayHasKey('ai_agents::ai_agent::sub_agent_beta', $mainTools, 'Main agent should have Beta sub-agent');

    // Main agent should NOT have the leaf tools (those belong to sub-agents).
    $this->assertArrayNotHasKey('tool:entity_bundle_list', $mainTools, 'Main agent should NOT have Alpha\'s tool');
    $this->assertArrayNotHasKey('tool:system_time', $mainTools, 'Main agent should NOT have Beta\'s tool');

    // Verify system prompt was updated.
    $this->assertEquals('Updated orchestrator prompt', $reloadedMain->get('system_prompt'));
  }

  /**
   * Tests that sidebar agent nodes save as sub-agent tools.
   *
   * This mirrors dragging an available agent into the assistant canvas,
   * where the node ID may be prefixed with ai_agent_ and metadata holds
   * the tool ID without explicit agent_id config.
   */
  public function testSidebarAgentNodeSavesAsSubAgentTool(): void {
    $mainAgent = $this->createTestAgent('sidebar_parent', [
      'tools' => [],
    ]);
    $this->createTestAgent('sidebar_child', [
      'tools' => [],
    ]);

    $assistant = $this->createTestAssistant('sidebar_assistant', 'sidebar_parent');

    $mainNodeId = 'agent_sidebar_parent';
    $childNodeId = 'ai_agent_sidebar_child';

    $workflowData = $this->buildWorkflowData(
      nodes: [
        $this->createAssistantNode('sidebar_parent', [
          'label' => 'Sidebar Assistant',
        ]),
        [
          'id' => $childNodeId,
          'type' => 'agent',
          'position' => ['x' => 300, 'y' => 200],
          'data' => [
            'nodeType' => 'agent',
            'config' => [],
            'metadata' => [
              'tool_id' => 'ai_agents::ai_agent::sidebar_child',
              'executor_plugin' => 'ai_agents::ai_agent::sidebar_child',
              'type' => 'agent',
            ],
          ],
        ],
      ],
      edges: [
        $this->createEdge($mainNodeId, $childNodeId),
      ],
      metadata: []
    );

    $request = $this->createJsonRequest($workflowData);
    $controller = $this->getSaveController();
    $response = $controller->save('sidebar_assistant', $request);
    $this->assertEquals(200, $response->getStatusCode());

    $reloadedMain = $this->reloadAgent('sidebar_parent');
    $tools = $reloadedMain->get('tools');
    $this->assertArrayHasKey('ai_agents::ai_agent::sidebar_child', $tools);
    $this->assertTrue($tools['ai_agents::ai_agent::sidebar_child']);
  }

  /**
   * Tests that sub-agent tools don't leak to the parent agent.
   *
   * This verifies the edge-based child detection works correctly - only
   * direct children of the main node should be processed, not grandchildren.
   */
  public function testSubAgentToolsDontLeakToParent(): void {
    // Create main agent.
    $mainAgent = $this->createTestAgent('parent_agent', [
      'tools' => [],
    ]);

    // Create a sub-agent that has its own tools in its entity config.
    $subAgent = $this->createTestAgent('child_agent', [
      'tools' => [
        'tool:entity_bundle_list' => TRUE,
      ],
    ]);

    // Create assistant.
    $assistant = $this->createTestAssistant('leak_test_assistant', 'parent_agent');

    // Build workflow with parent -> child connection only.
    // The child's existing tools should not be added to parent.
    $mainNodeId = 'agent_parent_agent';
    $childNodeId = 'subagent_child_agent';

    $workflowData = $this->buildWorkflowData(
      nodes: [
        $this->createAssistantNode('parent_agent', [
          'label' => 'Leak Test Assistant',
        ]),
        $this->createSubAgentNode('child_agent', $childNodeId),
      ],
      edges: [
        $this->createEdge($mainNodeId, $childNodeId),
      ],
      metadata: []
    );

    // Execute save.
    $request = $this->createJsonRequest($workflowData);
    $controller = $this->getSaveController();
    $response = $controller->save('leak_test_assistant', $request);

    // Verify response.
    $this->assertEquals(200, $response->getStatusCode());

    // Reload parent and verify it ONLY has the child agent as a tool.
    $reloadedParent = $this->reloadAgent('parent_agent');
    $parentTools = $reloadedParent->get('tools');
    $this->assertIsArray($parentTools);

    $this->assertArrayHasKey('ai_agents::ai_agent::child_agent', $parentTools);
    $this->assertCount(1, $parentTools, 'Parent should only have the child agent, not child\'s tools');

    // Verify child agent still has its original tools.
    $reloadedChild = $this->reloadAgent('child_agent');
    $childTools = $reloadedChild->get('tools');
    $this->assertArrayHasKey('tool:entity_bundle_list', $childTools, 'Child should keep its own tools');
  }

  /**
   * Tests workflow with mixed tools and sub-agents directly under parent.
   *
   * Parent agent should get both tools and sub-agents that are directly
   * connected to it.
   */
  public function testMixedToolsAndSubAgentsSaveCorrectly(): void {
    // Create main agent.
    $mainAgent = $this->createTestAgent('mixed_parent', [
      'tools' => [],
    ]);

    // Create a sub-agent.
    $subAgent = $this->createTestAgent('mixed_child', []);

    // Create assistant.
    $assistant = $this->createTestAssistant('mixed_assistant', 'mixed_parent');

    // Build workflow with parent connected to both a tool AND a sub-agent.
    $mainNodeId = 'agent_mixed_parent';

    $workflowData = $this->buildWorkflowData(
      nodes: [
        $this->createAssistantNode('mixed_parent', [
          'label' => 'Mixed Assistant',
        ]),
        // Direct tool connection.
        $this->createToolNode('tool:entity_bundle_list', 'direct_tool'),
        // Sub-agent connection.
        $this->createSubAgentNode('mixed_child', 'subagent_mixed_child'),
      ],
      edges: [
        $this->createEdge($mainNodeId, 'direct_tool'),
        $this->createEdge($mainNodeId, 'subagent_mixed_child'),
      ],
      metadata: []
    );

    // Execute save.
    $request = $this->createJsonRequest($workflowData);
    $controller = $this->getSaveController();
    $response = $controller->save('mixed_assistant', $request);

    // Verify response.
    $this->assertEquals(200, $response->getStatusCode());

    // Reload and verify parent has BOTH the direct tool AND the sub-agent.
    $reloadedParent = $this->reloadAgent('mixed_parent');
    $parentTools = $reloadedParent->get('tools');
    $this->assertIsArray($parentTools);

    $this->assertArrayHasKey('tool:entity_bundle_list', $parentTools, 'Parent should have direct tool');
    $this->assertArrayHasKey('ai_agents::ai_agent::mixed_child', $parentTools, 'Parent should have sub-agent');
    $this->assertCount(2, $parentTools);
  }

  /**
   * Tests three-level hierarchy only saves direct children.
   *
   * Parent -> Child -> Grandchild
   * Parent should only have Child as a tool, not Grandchild.
   */
  public function testThreeLevelHierarchyOnlySavesDirectChildren(): void {
    // Create three agents in a hierarchy.
    $parentAgent = $this->createTestAgent('hierarchy_parent', [
      'tools' => [],
    ]);
    $childAgent = $this->createTestAgent('hierarchy_child', [
      'tools' => [],
    ]);
    $grandchildAgent = $this->createTestAgent('hierarchy_grandchild', [
      'tools' => [],
    ]);

    // Create assistant.
    $assistant = $this->createTestAssistant('hierarchy_assistant', 'hierarchy_parent');

    // Build workflow: parent -> child -> grandchild.
    $parentNodeId = 'agent_hierarchy_parent';
    $childNodeId = 'subagent_hierarchy_child';
    $grandchildNodeId = 'subagent_hierarchy_grandchild';

    $workflowData = $this->buildWorkflowData(
      nodes: [
        $this->createAssistantNode('hierarchy_parent', [
          'label' => 'Hierarchy Assistant',
        ]),
        $this->createSubAgentNode('hierarchy_child', $childNodeId, FALSE, ['x' => 300, 'y' => 100]),
        $this->createSubAgentNode('hierarchy_grandchild', $grandchildNodeId, FALSE, ['x' => 500, 'y' => 100]),
      ],
      edges: [
        // Parent -> Child.
        $this->createEdge($parentNodeId, $childNodeId),
        // Child -> Grandchild (not direct child of parent).
        $this->createEdge($childNodeId, $grandchildNodeId),
      ],
      metadata: []
    );

    // Execute save.
    $request = $this->createJsonRequest($workflowData);
    $controller = $this->getSaveController();
    $response = $controller->save('hierarchy_assistant', $request);

    // Verify response.
    $this->assertEquals(200, $response->getStatusCode());

    // Reload parent and verify it ONLY has child, NOT grandchild.
    $reloadedParent = $this->reloadAgent('hierarchy_parent');
    $parentTools = $reloadedParent->get('tools');
    $this->assertIsArray($parentTools);

    $this->assertArrayHasKey('ai_agents::ai_agent::hierarchy_child', $parentTools, 'Parent should have child');
    $this->assertArrayNotHasKey('ai_agents::ai_agent::hierarchy_grandchild', $parentTools, 'Parent should NOT have grandchild');
    $this->assertCount(1, $parentTools);
  }

  /**
   * Tests that saving preserves agent identity - each agent keeps its own ID.
   *
   * This prevents the bug where agent A's config might accidentally be
   * saved to agent B.
   */
  public function testAgentIdentityPreserved(): void {
    // Create two completely separate agents with distinct values.
    $agentA = $this->createTestAgent('identity_agent_a', [
      'system_prompt' => 'I am Agent A',
      'description' => 'Agent A description',
      'max_loops' => 5,
    ]);

    $agentB = $this->createTestAgent('identity_agent_b', [
      'system_prompt' => 'I am Agent B',
      'description' => 'Agent B description',
      'max_loops' => 10,
    ]);

    // Create assistant for agent A.
    $assistant = $this->createTestAssistant('identity_assistant', 'identity_agent_a');

    // Build workflow that only modifies agent A (the assistant's agent).
    $workflowData = $this->buildWorkflowData(
      nodes: [
        $this->createAssistantNode('identity_agent_a', [
          'label' => 'Identity Test',
          'systemPrompt' => 'Modified Agent A prompt',
          'maxLoops' => 7,
        ]),
        // Include agent B as a sub-agent but don't modify its config.
        $this->createSubAgentNode('identity_agent_b', 'subagent_identity_agent_b'),
      ],
      edges: [
        $this->createEdge('agent_identity_agent_a', 'subagent_identity_agent_b'),
      ],
      metadata: []
    );

    // Execute save.
    $request = $this->createJsonRequest($workflowData);
    $controller = $this->getSaveController();
    $response = $controller->save('identity_assistant', $request);

    // Verify response.
    $this->assertEquals(200, $response->getStatusCode());

    // Reload both agents and verify identity is preserved.
    $reloadedA = $this->reloadAgent('identity_agent_a');
    $reloadedB = $this->reloadAgent('identity_agent_b');

    // Agent A should be modified.
    $this->assertEquals('Modified Agent A prompt', $reloadedA->get('system_prompt'));
    $this->assertEquals(7, $reloadedA->get('max_loops'));

    // Agent B should be UNCHANGED (no config in node = no changes).
    $this->assertEquals('I am Agent B', $reloadedB->get('system_prompt'));
    $this->assertEquals('Agent B description', $reloadedB->get('description'));
    $this->assertEquals(10, $reloadedB->get('max_loops'));
  }

  /**
   * Tests that sub-agent description saves from assistant editor.
   *
   * This is the specific test for the bug where editing a sub-agent's
   * description from the assistant's FlowDrop editor doesn't save.
   * The fix requires topological sort ordering and sub-agent update logic.
   */
  public function testSubAgentDescriptionSaves(): void {
    // Create main agent.
    $mainAgent = $this->createTestAgent('main_for_desc_test', [
      'description' => 'Main agent description',
    ]);

    // Create sub-agent with original description.
    $subAgent = $this->createTestAgent('sub_for_desc_test', [
      'description' => 'Original sub-agent description',
      'system_prompt' => 'Original sub-agent prompt',
    ]);

    // Create assistant.
    $assistant = $this->createTestAssistant('desc_test_assistant', 'main_for_desc_test');

    // Build workflow with MODIFIED sub-agent description in node config.
    // This simulates what happens when user edits the sub-agent in UI.
    $mainNodeId = 'agent_main_for_desc_test';
    $subNodeId = 'subagent_sub_for_desc_test';

    $workflowData = $this->buildWorkflowData(
      nodes: [
        $this->createAssistantNode('main_for_desc_test', [
          'label' => 'Description Test Assistant',
        ]),
        // Sub-agent with modified description and system prompt.
        [
          'id' => $subNodeId,
          'type' => 'agent',
          'position' => ['x' => 300, 'y' => 100],
          'data' => [
            'nodeType' => 'agent',
            'config' => [
              'agent_id' => 'sub_for_desc_test',
              'description' => 'UPDATED sub-agent description',
              'systemPrompt' => 'UPDATED sub-agent prompt',
              'maxLoops' => 7,
            ],
            'metadata' => [
              'ownerAgentId' => 'sub_for_desc_test',
            ],
          ],
        ],
      ],
      edges: [
        $this->createEdge($mainNodeId, $subNodeId),
      ],
      metadata: []
    );

    // Execute save.
    $request = $this->createJsonRequest($workflowData);
    $controller = $this->getSaveController();
    $response = $controller->save('desc_test_assistant', $request);

    // Verify response.
    $this->assertEquals(200, $response->getStatusCode());

    // Reload sub-agent and verify description was updated.
    $reloadedSub = $this->reloadAgent('sub_for_desc_test');

    $this->assertEquals(
      'UPDATED sub-agent description',
      $reloadedSub->get('description'),
      'Sub-agent description should be updated from workflow save'
    );
    $this->assertEquals(
      'UPDATED sub-agent prompt',
      $reloadedSub->get('system_prompt'),
      'Sub-agent system prompt should be updated from workflow save'
    );
    $this->assertEquals(
      7,
      $reloadedSub->get('max_loops'),
      'Sub-agent max_loops should be updated from workflow save'
    );

    // Verify main agent has sub-agent as tool.
    $reloadedMain = $this->reloadAgent('main_for_desc_test');
    $mainTools = $reloadedMain->get('tools');
    $this->assertArrayHasKey(
      'ai_agents::ai_agent::sub_for_desc_test',
      $mainTools,
      'Main agent should have sub-agent as tool'
    );
  }

  /**
   * Tests sub-agent label saves from assistant editor.
   *
   * Similar to description test but focuses on the label field.
   */
  public function testSubAgentLabelSaves(): void {
    // Create main agent.
    $mainAgent = $this->createTestAgent('main_for_label_test', []);

    // Create sub-agent with original label.
    $subAgent = $this->createTestAgent('sub_for_label_test', [
      'label' => 'Original Label',
    ]);

    // Create assistant.
    $assistant = $this->createTestAssistant('label_test_assistant', 'main_for_label_test');

    // Build workflow with MODIFIED sub-agent label.
    $mainNodeId = 'agent_main_for_label_test';
    $subNodeId = 'subagent_sub_for_label_test';

    $workflowData = $this->buildWorkflowData(
      nodes: [
        $this->createAssistantNode('main_for_label_test', [
          'label' => 'Label Test Assistant',
        ]),
        [
          'id' => $subNodeId,
          'type' => 'agent',
          'position' => ['x' => 300, 'y' => 100],
          'data' => [
            'nodeType' => 'agent',
            'config' => [
              'agent_id' => 'sub_for_label_test',
              'label' => 'UPDATED Label',
            ],
            'metadata' => [
              'ownerAgentId' => 'sub_for_label_test',
            ],
          ],
        ],
      ],
      edges: [
        $this->createEdge($mainNodeId, $subNodeId),
      ],
      metadata: []
    );

    // Execute save.
    $request = $this->createJsonRequest($workflowData);
    $controller = $this->getSaveController();
    $response = $controller->save('label_test_assistant', $request);

    // Verify response.
    $this->assertEquals(200, $response->getStatusCode());

    // Reload sub-agent and verify label was updated.
    $reloadedSub = $this->reloadAgent('sub_for_label_test');

    $this->assertEquals(
      'UPDATED Label',
      $reloadedSub->get('label'),
      'Sub-agent label should be updated from workflow save'
    );
  }

}
