<?php

declare(strict_types=1);

namespace Drupal\Tests\flowdrop_ui_agents\Kernel;

use Drupal\flowdrop_ui_agents\Controller\Api\WorkflowController;
use PHPUnit\Framework\Attributes\Group;
use Symfony\Component\HttpFoundation\Request;

/**
 * Tests workflow API responses used by the FlowDrop UI.
 */
#[Group('flowdrop_ui_agents')]
final class WorkflowControllerTest extends FlowdropAgentsTestBase {

  /**
   * Ensures workflow expansion includes downstream sub-agents.
   */
  public function testWorkflowIncludesDownstreamAgents(): void {
    $child = $this->createTestAgent('child_agent');
    $parent = $this->createTestAgent('parent_agent', [
      'tools' => [
        'ai_agents::ai_agent::child_agent' => TRUE,
      ],
    ]);

    $controller = WorkflowController::create($this->container);
    $request = Request::create('/api/flowdrop-agents/workflow/parent_agent', 'GET', [
      'expansion' => 'expanded',
    ]);

    $response = $controller->getWorkflow('parent_agent', $request);
    $this->assertEquals(200, $response->getStatusCode());

    $payload = json_decode((string) $response->getContent(), TRUE);
    $this->assertTrue($payload['success']);

    $nodes = $payload['data']['nodes'] ?? [];
    $edges = $payload['data']['edges'] ?? [];
    $nodeIds = array_map(static fn(array $node): string => (string) $node['id'], $nodes);

    $this->assertContains('agent_parent_agent', $nodeIds);
    $this->assertContains('agent_child_agent', $nodeIds);

    $hasEdge = FALSE;
    foreach ($edges as $edge) {
      if (($edge['source'] ?? '') === 'agent_parent_agent' && ($edge['target'] ?? '') === 'agent_child_agent') {
        $hasEdge = TRUE;
        break;
      }
    }
    $this->assertTrue($hasEdge, 'Expected edge from parent agent to child agent.');
  }

  /**
   * Ensures expanded workflows include the tools output handle on agent nodes.
   */
  public function testWorkflowAgentOutputsIncludeTools(): void {
    $this->createTestAgent('output_agent');

    $controller = WorkflowController::create($this->container);
    $request = Request::create('/api/flowdrop-agents/workflow/output_agent', 'GET', [
      'expansion' => 'expanded',
    ]);

    $response = $controller->getWorkflow('output_agent', $request);
    $this->assertEquals(200, $response->getStatusCode());

    $payload = json_decode((string) $response->getContent(), TRUE);
    $this->assertTrue($payload['success']);

    $nodes = $payload['data']['nodes'] ?? [];
    $agentNode = NULL;
    foreach ($nodes as $node) {
      if (($node['id'] ?? '') === 'agent_output_agent') {
        $agentNode = $node;
        break;
      }
    }

    $this->assertNotNull($agentNode, 'Expanded workflow should include the agent node.');
    $outputs = $agentNode['data']['metadata']['outputs'] ?? [];
    $hasToolsOutput = FALSE;
    foreach ($outputs as $output) {
      if (($output['dataType'] ?? '') === 'tool') {
        $hasToolsOutput = TRUE;
        break;
      }
    }
    $this->assertTrue($hasToolsOutput, 'Agent node outputs should include a tools handle.');
  }

  /**
   * Ensures expanded workflows include tool nodes and their edges.
   */
  public function testWorkflowExpansionIncludesTools(): void {
    $this->createTestAgent('expansion_agent', [
      'tools' => [
        'ai_agent:list_bundles' => TRUE,
      ],
      'tool_settings' => [
        'ai_agent:list_bundles' => [
          'description_override' => 'List bundles override.',
        ],
      ],
    ]);

    $controller = WorkflowController::create($this->container);
    $request = Request::create('/api/flowdrop-agents/workflow/expansion_agent', 'GET', [
      'expansion' => 'expanded',
    ]);

    $response = $controller->getWorkflow('expansion_agent', $request);
    $this->assertEquals(200, $response->getStatusCode());

    $payload = json_decode((string) $response->getContent(), TRUE);
    $this->assertTrue($payload['success']);

    $nodes = $payload['data']['nodes'] ?? [];
    $edges = $payload['data']['edges'] ?? [];

    $toolNode = NULL;
    foreach ($nodes as $node) {
      if (($node['data']['metadata']['tool_id'] ?? '') === 'ai_agent:list_bundles') {
        $toolNode = $node;
        break;
      }
    }

    $this->assertNotNull($toolNode, 'Expanded workflow should include tool nodes.');

    $hasEdge = FALSE;
    foreach ($edges as $edge) {
      if (($edge['source'] ?? '') === 'agent_expansion_agent' && ($edge['target'] ?? '') === ($toolNode['id'] ?? '')) {
        $hasEdge = TRUE;
        break;
      }
    }
    $this->assertTrue($hasEdge, 'Expected edge from agent to tool node.');
  }

  /**
   * Ensures property restriction actions and overrides survive merge.
   */
  public function testWorkflowToolPropertyRestrictionsIncludeOverrides(): void {
    $this->createTestAgent('restriction_agent', [
      'tools' => [
        'ai_agent:list_bundles' => TRUE,
      ],
      'tool_settings' => [
        'ai_agent:list_bundles' => [
          'property_restrictions' => [
            'entity_type' => [
              'description_override' => 'cheese',
            ],
          ],
        ],
      ],
      'tool_usage_limits' => [
        'ai_agent:list_bundles' => [
          'entity_type' => [
            'action' => 'force_value',
            'hide_property' => 0,
            'values' => ['node'],
          ],
        ],
      ],
    ]);

    $controller = WorkflowController::create($this->container);
    $request = Request::create('/api/flowdrop-agents/workflow/restriction_agent', 'GET', [
      'expansion' => 'expanded',
    ]);

    $response = $controller->getWorkflow('restriction_agent', $request);
    $this->assertEquals(200, $response->getStatusCode());

    $payload = json_decode((string) $response->getContent(), TRUE);
    $this->assertTrue($payload['success']);

    $nodes = $payload['data']['nodes'] ?? [];
    $toolNode = NULL;
    foreach ($nodes as $node) {
      if (($node['data']['metadata']['tool_id'] ?? '') === 'ai_agent:list_bundles') {
        $toolNode = $node;
        break;
      }
    }

    $this->assertNotNull($toolNode, 'Expected tool node for entity_bundle_list.');
    $config = $toolNode['data']['config'] ?? [];

    $this->assertSame('Force value', $config['prop_entity_type_restriction'] ?? '');
    $this->assertSame('node', $config['prop_entity_type_values'] ?? '');
    $this->assertTrue($config['prop_entity_type_override_desc_enabled'] ?? FALSE);
    $this->assertSame('cheese', $config['prop_entity_type_override_desc'] ?? '');
  }

  /**
   * Ensures legacy property_description_override surfaces in tool config.
   */
  public function testWorkflowToolLegacyPropertyOverridesAppear(): void {
    $this->createTestAgent('legacy_override_agent', [
      'tools' => [
        'ai_agent:get_content_type_info' => TRUE,
      ],
      'tool_settings' => [
        'ai_agent:get_content_type_info' => [
          'property_description_override' => [
            'node_type' => 'cheese2',
          ],
        ],
      ],
      'tool_usage_limits' => [
        'ai_agent:get_content_type_info' => [
          'node_type' => [
            'action' => 'only_allow',
            'hide_property' => 0,
            'values' => ['node'],
          ],
        ],
      ],
    ]);

    $controller = WorkflowController::create($this->container);
    $request = Request::create('/api/flowdrop-agents/workflow/legacy_override_agent', 'GET', [
      'expansion' => 'expanded',
    ]);

    $response = $controller->getWorkflow('legacy_override_agent', $request);
    $this->assertEquals(200, $response->getStatusCode());

    $payload = json_decode((string) $response->getContent(), TRUE);
    $this->assertTrue($payload['success']);

    $nodes = $payload['data']['nodes'] ?? [];
    $toolNode = NULL;
    foreach ($nodes as $node) {
      if (($node['data']['metadata']['tool_id'] ?? '') === 'ai_agent:get_content_type_info') {
        $toolNode = $node;
        break;
      }
    }

    $this->assertNotNull($toolNode, 'Expected tool node for get_content_type_info.');
    $config = $toolNode['data']['config'] ?? [];

    $this->assertSame('Only allow certain values', $config['prop_node_type_restriction'] ?? '');
    $this->assertTrue($config['prop_node_type_override_desc_enabled'] ?? FALSE);
    $this->assertSame('cheese2', $config['prop_node_type_override_desc'] ?? '');

    $values = $config['prop_node_type_values'] ?? [];
    $valuesList = is_array($values) ? $values : [$values];
    $this->assertContains('node', $valuesList);
  }

}
