<?php

namespace Drupal\fluidui\Hook;

use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Routing\RouteMatchInterface;

/**
 * Class FluidUiHooks that implements the hooks.
 */
class FluidUiHooks {

  /**
   * Implements hook_help().
   */
  #[Hook('help')]
  public function help($route_name, RouteMatchInterface $route_match): ?string {
    switch ($route_name) {
      case 'help.page.fluidui':
        $output = '';
        $output .= '<h3>' . t('About') . '</h3>';
        $output .= '<p>' . t('Fluidproject UI Options provides accessibility options for users to modify a page font size, line height, font style, contrast, and link style. The changes are retained using cookies. The module uses the <a href=":link">Infusion library</a> to achieve that.', [':link' => 'https://fluidproject.org/infusion']) . '</p>';
        $output .= '<p>' . t('&nbsp;To submit bug reports and feature suggestions, or track changes, visit the <a href=":link">Project Issue Page</a>.', [':link' => 'https://www.drupal.org/project/issues/fluidui']) . '</p>';

        return $output;
    }

    return NULL;
  }

  /**
   * Implements hook_page_top().
   */
  #[Hook('page_top')]
  public function fluiduiPageTop(array &$page_top) {
    $route = \Drupal::routeMatch()->getRouteObject();

    $is_admin = \Drupal::service('router.admin_context')->isAdminRoute($route);

    $current_path = \Drupal::service('path.current')->getPath();

    $url_alias = \Drupal::service('path_alias.manager')->getAliasByPath($current_path);

    $config = \Drupal::config("fluidui.adminsettings");

    $admin_display = $config->get('admin_display') ?? 0;

    $fluidui_as_block = $config->get('fluidui_as_block') ?? 0;

    $url_blacklist = $config->get('url_blacklist') !== NULL ? explode("\r\n", $config->get('url_blacklist')) : [''];

    $url_blacklist_wildcard = [];

    // Handles the wildcard (*) at the end of the url.
    foreach ($url_blacklist as $url) {
      /* No wildcards */
      if (mb_strpos($url, '/*') === FALSE) {
        continue;
      }

      $url2 = str_replace('/*', '', $url);

      if (mb_strpos($current_path, $url2) === 0) {
        $url_blacklist_wildcard[] = $current_path;
        return;
      }

      if (str_contains($url2, $url_alias) === 0) {
        $url_blacklist_wildcard[] = $url_alias;
        return;
      }
    }

    // The url blacklist takes precedence.
    if (in_array($current_path, $url_blacklist) || in_array($url_alias, $url_blacklist)) {
      return;
    }

    if (in_array($current_path, $url_blacklist_wildcard) || in_array($url_alias, $url_blacklist_wildcard)) {
      return;
    }

    if ($admin_display && $is_admin) {
      $page_top['fluidui'] = [
        '#theme' => 'fluid_ui_block',
      ];
    }
    else {
      // Don't display the widget on the admin pages.
      if ($fluidui_as_block === 0) {
        if (!$is_admin) {
          $page_top['fluidui'] = [
            '#theme' => 'fluid_ui_block',
          ];
        }

        if ($admin_display && !$is_admin) {
          unset($page_top['fluidui']);
        }
      }
    }

    $variables['page']['#cache']['contexts'][] = 'url.path';
    $variables['page']['#cache']['contexts'][] = 'user';
  }

  /**
   * Implements hook_page_attachments().
   */
  #[Hook('page_attachments')]
  public function jqueryDeprecatedFunctionsPageAttachments(array &$page): void {
    $page['#attached']['library'][] = 'jquery_deprecated_functions/global-scripts';
  }

}
