<?php

namespace Drupal\flysystem\Plugin;

use Drupal\Core\Url;

/**
 * Helper trait for generating URLs from adapter plugins.
 */
trait FlysystemUrlTrait {

  /**
   * Returns a web accessible URL for the resource.
   *
   * This function should return a URL that can be embedded in a web page
   * and accessed from a browser. For example, the external URL of
   * "youtube://abc123" might be
   * "http://www.youtube.com/watch?v=abc123".
   *
   * @param string $uri
   *   The URI to provide a URL for.
   *
   * @return string
   *   Returns a string containing a web accessible URL for the resource.
   */
  public function getExternalUrl($uri) {
    $path = str_replace('\\', '/', $this->getTarget($uri));
    $scheme = $this->getScheme($uri);
    $arguments = [
      'scheme' => $scheme,
      'filepath' => $path,
    ];
    return Url::fromRoute('flysystem.serve', $arguments, ['absolute' => TRUE])->toString();
  }

  /**
   * Returns the target file path of a URI.
   *
   * @param string $uri
   *   The URI.
   *
   * @return string
   *   The file path of the URI.
   *
   * @throws \LogicException
   */
  protected function getTarget($uri) {
    $path = substr($uri, strpos($uri, '://') + 3);
    $path = str_replace('\\', '/', $path);

    while (preg_match('#\p{C}+|^\./#u', $path)) {
      $path = preg_replace('#\p{C}+|^\./#u', '', $path);
    }

    $parts = [];

    foreach (explode('/', $path) as $part) {
      switch ($part) {
        case '':
        case '.':
          break;

        case '..':
          if (empty($parts)) {
            throw new \LogicException(
                  'Path is outside of the defined root, path: [' . $path . ']'
              );
          }
          array_pop($parts);
          break;

        default:
          $parts[] = $part;
          break;
      }
    }

    return implode('/', $parts);

  }

  /**
   * Returns the scheme from the internal URI.
   *
   * @param string $uri
   *   The URI.
   *
   * @return string
   *   The scheme.
   */
  protected function getScheme($uri) {
    return substr($uri, 0, strpos($uri, '://'));
  }

}
